class Gallery {
  #slider;
  #thumbsSlider;
  #elements;
  #options = {};
  #mediaData = {};

  #SCRIPT = {
    id: 'slider-js',
    src: `${APP.ROOT_PATH}assets/scripts/libs/swiper/bundle.js`
  };
  
  #STYLE = {
    id: 'slider-css',
    src: `${APP.ROOT_PATH}assets/scripts/libs/swiper/bundle.css` // 'https://cdn.jsdelivr.net/npm/swiper@11/swiper-bundle.min.css'
  };

  #defaultOptions = {
    speed: 150,
    initialSlide: 0,
    spaceBetween: 0,

    effect: 'fade',
    allowTouchMove: true,

    containerModifierClass: 'slider-',
    wrapperClass: 'slider__container',
    slideClass: 'slider__item',
    slideNextClass: 'slider__item--next',
    slidePrevClass: 'slider__item--prev',
    slideActiveClass: 'active',

    loop: true,
    autoplay: false,
    pagination: {
      enabled: false,
      clickable: true,
      type: 'bullets',
      el: '.slider__pagination',
      dynamicBullets: false
    },
    navigation: {
      enabled: true,
      nextEl: '.slider__btn--next',
      prevEl: '.slider__btn--prev',
      disabledClass: 'slider__btn--disabled',
    },

    custom: {
      thumbs: {
        enabled: true,
        MAX_THUMBS_VISIBLE: 15,
        THUMB_MAX_WIDTH: 80,
        THUMBS_VISIBILITY_MIN_WIDTH: 768,
        spaceBetween: 15,
        breakpointsResponsive: true,
      },
      prefix: null,
      MAX_FOOTER_CONTENT_WIDTH: 1460,
    }
  };

  constructor(elements, mediaData, options) {
    this.#elements = elements;

    if (!mediaData) return;

    this.#prepareMediaData(mediaData);
    this.#setupOptions(options);
    this.#initialize();
  }

  #prepareMediaData(mediaData) {
    const preparedImagesWithSizes = [];

    const getImagePathWithSize = (imagePath, size) => {
      const imageExtension = imagePath.split('.').reverse()[0];
      const imageExtensionWithSize = `_${size}.${imageExtension}`;

      return imagePath.replace(`.${imageExtension}`, imageExtensionWithSize);
    };

    mediaData.forEach(mediaItem => {
      if (!mediaItem) return;

      if (mediaItem.type === 'VIDEO') {
        preparedImagesWithSizes.push(Object.assign({}, mediaItem, {
          thumb: HELPER.getHotspotMediaUrl(mediaItem.thumb)
        }));

        return;
      }

      const imageSizes = mediaItem.sizes;
      const imagePath = HELPER.getHotspotMediaUrl(mediaItem.src);
      const imageThumb = !!mediaItem.thumb && mediaItem.thumb.length > 0 
        ? HELPER.getHotspotMediaUrl(mediaItem.thumb)
        : null;

      const imageObject = {
        src: imagePath,
        thumb: imageThumb,
        type: mediaItem.type
      };

      if (imageSizes && imageSizes.length > 0) {
        imageObject['sizes'] = {};

        imageSizes.forEach(size => {
          const imageSrcWithSize = getImagePathWithSize(mediaItem.src, size);
  
          imageObject.sizes[size] = imageSrcWithSize;
        })
      }

      preparedImagesWithSizes.push(imageObject);
    });

    this.#mediaData = preparedImagesWithSizes;
  }

  #deepMerge(target, ...sources) {
    sources.forEach(source => {
      Object.keys(source).forEach(key => {
        if (
          typeof source[key] === 'object' &&
          source[key] !== null &&
          !Array.isArray(source[key])
        ) {
          if (!target[key] || typeof target[key] !== 'object') {
            target[key] = {};
          }

          this.#deepMerge(target[key], source[key]);
        } else {
          target[key] = source[key];
        }
      });
    });

    return target;
  }

  #setupOptions(customOptions) {
    const mergedOptions = this.#deepMerge(this.#defaultOptions, customOptions);

    this.#options = mergedOptions;

    if (this.#options.navigation) {
      this.#options.navigation.enabled = this.#options.navigation.enabled ? this.#mediaData.length > 1 : this.#options.navigation.enabled;
    }

    if (this.#options.custom.thumbs) {
      this.#options.custom.thumbs.enabled = this.#options.custom.thumbs.enabled ? this.#mediaData.length > 1 : this.#options.custom.thumbs.enabled;
    }
  }
  
  #getOptions() {
    return this.#options;
  }

  #loadScript() {
    return new Promise((resolve, reject) => {
      const script = document.createElement('script');

      script.id = this.#SCRIPT.id;
      script.src = this.#SCRIPT.src;

      document.body.appendChild(script);

      script.addEventListener('load', resolve, { once: true });
      script.addEventListener('error', reject, { once: true });
    });
  }

  #loadStyle() {
    return new Promise((resolve, reject) => {
      const link = document.createElement("link");

      link.id = this.#STYLE.id;
      link.href = this.#STYLE.src;
      link.type = "text/css";
      link.rel = "stylesheet";

      document.head.appendChild(link);

      link.addEventListener('load', resolve, { once: true });
      link.addEventListener('error', reject, { once: true });
    });
  }

  #isLibLoaded() {
    return !!document.querySelector(`script#${this.#SCRIPT.id}`) && !!document.querySelector(`link#${this.#STYLE.id}`);
  }

  #loadLib() {
    if (this.#isLibLoaded()) return;

    return Promise.all([this.#loadScript(), this.#loadStyle()]);
  }

  #getSmallerAndLargerSizeFromArray(sizes) {
    let larger = null;
    let smaller = null;
  
    for (let i = 0; i < sizes.length; i++) {
      if (sizes[i] > window.innerWidth) {
        if (larger === null) {
          larger = +sizes[i];
        }
      } else {
        smaller = +sizes[i];
      }
    }
  
    return { larger, smaller};
  }

  #getImageSrcWithSize(imageObj) {
    if (imageObj.type === 'VIDEO') {
      return imageObj.thumb;
    }

    if (!imageObj.sizes) return imageObj.src;

    const sortedSizes = Object.keys(imageObj.sizes).sort((a, b) => a - b).map(Number);
    const { larger, smaller } = this.#getSmallerAndLargerSizeFromArray(sortedSizes);

    if (!larger) {
      return imageObj.src;
    }

    if (!imageObj.sizes[larger]) {
      return imageObj.src;
    }

    return imageObj.src;

    // const largerImageSizesArray = Object.entries(imageObj.sizes).filter(([key, value]) => {
    //   if (+key >= larger) {
    //     return {
    //       [+key]: value
    //     };
    //   }
    // }).filter(Boolean).map(item => ({
    //   [item[0]]: item[1]
    // }));

    // if (largerImageSizesArray.length === 0) {
    //   return imageObj.src;
    // }

    // let largerImageSizes = {};

    // largerImageSizesArray.forEach(item => {
    //   largerImageSizes = Object.assign(largerImageSizes, item);
    // });

    // const modifiedImageObj = Object.assign({}, imageObj, {
    //   sizes: largerImageSizes
    // });

    // return this.#loadImageWithFallback(modifiedImageObj);
  }

  #createSlide(index) {
    const slide = document.createElement('div');
    const slideContent = document.createElement('div');

    slide.className = `swiper-slide slider__item slider__item-${index}`;
    slideContent.className = 'slider__item__content';

    const preloader = document.createElement('div');
    preloader.className = 'swiper-lazy-preloader';

    slideContent.appendChild(preloader);
    slide.appendChild(slideContent);

    return { slide, slideContent };
  }

  #createImage(imgSrc, index) {
    const image = document.createElement('img');
    
    image.className = 'slider__item__image';
    image.src = imgSrc;
    image.alt = `Gallery image ${index + 1}`;
    image.draggable = false;

    if (index !== 0) {
      image.loading = 'lazy';
    }

    return image;
  }

  async #loadImageWithFallback(imageObj) {
    const imagePaths = imageObj.sizes;
    const sizes = Object.keys(imagePaths).map(Number).sort((a, b) => a - b);
    
    for (let size of sizes) {
      const imageUrl = imagePaths[size];
      
      try {
        const response = await fetch(imageUrl, { method: 'GET' });
        
        if (response.ok) {
          return imageUrl;
        }
      } catch (error) {
        console.error(`Error checking image ${imageUrl}:`, error);
      }
    }

    return imageObj.src;
  }

  #buildHTML() {
    const slider = document.createElement('div');
    const wrapper = document.createElement('div');

    slider.className = 'swiper slider';
    wrapper.className = 'swiper-wrapper slider__container';
    
    slider.appendChild(wrapper);

    return { slider, wrapper };
  }

  #buildMainSliderHTML() {
    const { slider, wrapper } = this.#buildHTML();

    this.#mediaData.forEach((imageObj, index) => {
      const { slide, slideContent } = this.#createSlide(index);

      slide.classList.add(`slider__item--${imageObj.type.toLowerCase()}`);

      const imagePath = this.#getImageSrcWithSize(imageObj);
      const image = this.#createImage(imagePath, index);

      if (imageObj.type === 'VIDEO') {
        const frame = document.createElement('div');
        const playBtn = document.createElement('button');

        frame.className = `gallery__slide__iframe ${!!this.#options.custom.prefix ? `${this.#options.custom.prefix}-gallery__slide__iframe` : ''}`;
        playBtn.className = `gallery__slide__button--play ${!!this.#options.custom.prefix ? `${this.#options.custom.prefix}-gallery__slide__button--play` : ''}`;
        playBtn.addEventListener('click', () => this.#onPlayerBtnClick(imageObj));

        slideContent.appendChild(frame);
        slideContent.appendChild(playBtn);
      }

      slideContent.appendChild(image);
      wrapper.appendChild(slide);
    });

    return slider;
  }

  #buildThumbsSliderHTML() {
    const { slider, wrapper } = this.#buildHTML();

    this.#mediaData.forEach(async (imageObj, index) => {
      let imagePath = imageObj.type === 'VIDEO' ? imageObj.thumb : imageObj.src; // !!imageObj.thumb ? imageObj.thumb : imageObj.src;

      const { slide, slideContent } = this.#createSlide(index);

      slide.classList.add(`slider__item--${imageObj.type.toLowerCase()}`);
      wrapper.appendChild(slide);

      if (imageObj.sizes && Object.keys(imageObj.sizes).length > 0) {
        const imagePathWithLeastSize = await this.#loadImageWithFallback(imageObj);
        // const leastSize = Math.min.apply(null, Object.keys(imageObj.sizes));
        
        imagePath = imagePathWithLeastSize || imagePath;
      }

      const image = this.#createImage(imagePath, index);

      slideContent.appendChild(image);
    });

    return slider;
  }

  #hideNavigationArrows() {
    if (!this.#elements.controls) return;

    if (this.#elements.controls.arrowLeft) {
      this.#elements.controls.arrowLeft.setAttribute('hidden', true);
    }

    if (this.#elements.controls.arrowRight) {
      this.#elements.controls.arrowRight.setAttribute('hidden', true);
    }
  }

  #showNavigationArrows() {
    if (!this.#elements.controls) return;

    if (this.#elements.controls.arrowLeft) {
      this.#elements.controls.arrowLeft.removeAttribute('hidden');
    }

    if (this.#elements.controls.arrowRight) {
      this.#elements.controls.arrowRight.removeAttribute('hidden');
    }
  }

  #getBreakpointSizes(swiper) {
    if (!swiper.params.breakpoints) return;

    const breakpoints = Object.keys(swiper.params.breakpoints).sort((a, b) => a + b);
    
    return this.#getSmallerAndLargerSizeFromArray(breakpoints).smaller;
  }

  #getThumbMaxWidth() {
    if (!this.#thumbsSlider || !this.#thumbsSlider.swiper || !this.#thumbsSlider.swiper.slides || this.#thumbsSlider.swiper.slides.length === 0) {
      if (window.innerWidth >= window.innerHeight && window.innerHeight <= 768) {
        return 60;
      }

      return this.#options.custom.thumbs.THUMB_MAX_WIDTH;
    }

    const firstThumbSlide = this.#thumbsSlider.swiper.slides[0];

    if (!firstThumbSlide) {
      return this.#options.custom.thumbs.THUMB_MAX_WIDTH;
    }

    const slideContent = firstThumbSlide.querySelector('.slider__item__content');
    const slideContentStyles = window.getComputedStyle(slideContent);
    const slideContentMaxWidth = parseInt(slideContentStyles.getPropertyValue('max-width') || 0);

    if (slideContentMaxWidth) {
      return slideContentMaxWidth;
    }

    return this.#options.custom.thumbs.THUMB_MAX_WIDTH;
  };

  #updateSwiperThumbsConfig(self, swiper, breakpointSize) {
    if (!swiper.params.breakpoints) return;

    let swiperBreakpointSize = breakpointSize;
    const fixedBreakpointSize = this.#getBreakpointSizes(swiper);

    if (+swiperBreakpointSize !== +fixedBreakpointSize) {
      swiperBreakpointSize = fixedBreakpointSize;
    }

    const customBreakpointParams = swiper.params.breakpoints.hasOwnProperty(breakpointSize) ? swiper.params.breakpoints[swiperBreakpointSize] : null;
    
    if (!customBreakpointParams) return;

    const slideContentMaxWidth = this.#getThumbMaxWidth();
    const slidesPerView = Math.min(self.#mediaData.length, customBreakpointParams.custom.MAX_THUMBS_VISIBLE);
    const calculatedFooterContentWidth = slidesPerView * slideContentMaxWidth + (swiper.params.spaceBetween * slidesPerView);
    
    const actualFooterContentMaxWidth = parseInt(self.#elements.thumbsSliderWrapper.style.maxWidth, 10);
    const newFooterContentMaxWidth = Math.min(calculatedFooterContentWidth, self.#options.custom.MAX_FOOTER_CONTENT_WIDTH);

    if (swiper.params.slidesPerView === slidesPerView && actualFooterContentMaxWidth === newFooterContentMaxWidth) return;

    swiper.params.slidesPerView = slidesPerView;
    self.#elements.thumbsSliderWrapper.style.maxWidth = `${newFooterContentMaxWidth}px`;

    swiper.update();
  }

  #initializeThumbs(options) {
    const spaceBetween = options.custom.thumbs.spaceBetween;
    const slideContentMaxWidth = this.#getThumbMaxWidth();
    const slidesPerView = Math.min(this.#mediaData.length, options.custom.thumbs.MAX_THUMBS_VISIBLE);
    const calculatedFooterContentWidth = slidesPerView * slideContentMaxWidth + (spaceBetween * slidesPerView);
    const self = this;

    const modifiedOptions = Object.assign({}, options, {
      effect: 'slide',
      spaceBetween,
      slidesPerView,
      freeMode: true,
      watchSlidesProgress: true,
      allowTouchMove: true,
      navigation: false,
      pagination: false,
    }, Object.assign({}, options.custom.thumbs.breakpointsResponsive ? {
      breakpoints: {
        768: {
          custom: {
            MAX_THUMBS_VISIBLE: 6
          }
        },
        1200: {
          custom: {
            MAX_THUMBS_VISIBLE: 10
          }
        },
        1400: {
          custom: {
            MAX_THUMBS_VISIBLE: 15
          }
        }
      },
      on: {
        breakpoint(swiper) {
          self.#updateSwiperThumbsConfig(self, swiper, swiper.currentBreakpoint);
        },
      }
    }: {}));

    this.#elements.thumbsSliderWrapper.style.maxWidth = `${Math.min(calculatedFooterContentWidth, this.#options.custom.MAX_FOOTER_CONTENT_WIDTH)}px`;

    const thumbsSwiper = new Swiper(this.#thumbsSlider, modifiedOptions);

    return thumbsSwiper;
  }

  #updateMainSliderImagesSize() {
    const swiper = this.#slider.swiper;
    
    this.#mediaData.forEach((imageObj, index) => {
      const slide = swiper.slides[index];

      if (!slide) return;

      const slideImage = slide.querySelector('img');
      
      if (!slideImage) return;
      
      const slideImagePath = slideImage.src;
      const imagePath = this.#getImageSrcWithSize(imageObj);

      if (imagePath && slideImagePath !== imagePath) {
        slideImage.src = imagePath;
      }
    });
  }

  #handleResize = () => {
    this.#updateSwiperThumbsConfig(this, this.#thumbsSlider.swiper, this.#thumbsSlider.swiper.currentBreakpoint);
    this.#updateMainSliderImagesSize();
  }

  #bindEvents() {
    window.addEventListener('resize', this.#handleResize);
  }

  #unbindEvents() {
    window.removeEventListener('resize', this.#handleResize);
  }

  #onPlayerBtnClick(imageObj) {
    const event = new CustomEvent('product.gallery.video.play', {
      detail: {
        mediaDataObject: imageObj
      }
    });

    this.#elements.mainSliderWrapper.dispatchEvent(event);
  }

  #buildSliderPagination() {
    const elClassName = this.#options.pagination.el || '.swiper__pagination';

    if (this.#elements.mainSliderWrapper.querySelector(elClassName)) return;

    const paginationEl = document.createElement('div');
    paginationEl.className = elClassName.replace('.', '');

    this.#elements.mainSliderWrapper.appendChild(paginationEl);
  }

  async #initialize() {
    await this.#loadLib();

    if (!Swiper) {
      throw new Error('Swiper is not defined');
    }

    const self = this;
    const options = this.#getOptions();

    if (!options.navigation || !options.navigation.enabled) {
      this.#hideNavigationArrows();
    } else {
      this.#showNavigationArrows();
    }

    if (options.pagination.enabled) {
      this.#buildSliderPagination();
    }

    this.#slider = this.#buildMainSliderHTML();
    this.#elements.mainSliderWrapper.appendChild(this.#slider);

    if (options.custom.thumbs.enabled) {
      this.#thumbsSlider = this.#buildThumbsSliderHTML();
      this.#elements.thumbsSliderWrapper.appendChild(this.#thumbsSlider);
      this.#initializeThumbs(options);
    }

    new Swiper(this.#slider, Object.assign({}, options, {
      slidesPerView: 1,
      slidesPerGroup: 1,
      longSwipesMs: 1,
      longSwipesRatio: .01,
      watchSlidesProgress: true,
      keyboard: {
        enabled: true,
        onlyInViewport: false,
      },
      lazy: {
        loadPrevNext: true,
        loadPrevNextAmount: 2,
      },
      fadeEffect: {
        crossFade: true
      },
      
      breakpoints: {
        0: {
          keyboard: {
            enabled: false
          },
          allowTouchMove: true,
        }, 
        768: {
          keyboard: {
            enabled: true,
          },
          allowTouchMove: (window.innerHeight < window.innerWidth && window.innerHeight < 768) || !options.navigation.enabled,
        },
        1200: {
          allowTouchMove: true,
        },
      },
      on: {
        slideChange(swiper) {
          const activeIndex = swiper.activeIndex;
          const mediaDataObject = self.#mediaData[activeIndex];
          
          const event = new CustomEvent('product.gallery.slide.change', {
            detail: {
              activeIndex,
              mediaDataObject
            }
          });

          self.#elements.mainSliderWrapper.dispatchEvent(event);
        }
      }
    }, 
      options.custom.thumbs.enabled ? {
        thumbs: {
          swiper: this.#thumbsSlider,
        }
      } : {}
    ));

    this.#bindEvents();
  }

  destroy() {
    const options = this.#getOptions();

    if (this.#slider.swiper && !this.#slider.swiper.destroyed) {
      this.#slider.swiper.destroy();
    }

    this.#slider.remove();

    if (options.custom.thumbs.enabled && this.#thumbsSlider) {
      if (this.#thumbsSlider.swiper && !this.#thumbsSlider.swiper.destroyed) {
        this.#thumbsSlider.swiper.destroy();
      }

      this.#thumbsSlider.remove();
    }

    this.#unbindEvents();
  }
}

APP.Gallery = Gallery;
