class Audio {
  #audioElement;
  #options = {};
  #elements = {};
  #mediaData = {};

  #defaultOptions = {
    id: null,
    page: 0,
    start: 0,
    loop: false,
    muted: false,
    controls: true,
    autoplay: true,
    preload: 'auto',
    showPlayer: true,
    controlsList: 'nodownload nofullscreen noremoteplayback noplaybackrate',
  };

  constructor(elements, mediaData, options) {
    this.#elements = elements;

    if (!mediaData) return;

    this.#prepareMediaData(mediaData);
    this.#setupOptions(options);
    this.#initialize();
  }

  #prepareMediaData(mediaData) {
    if (!mediaData.hasOwnProperty('url') || !mediaData.url) {
      throw new Error('Invalid audio media data');
    }

    const url = HELPER.getHotspotMediaUrl(mediaData.url.trim());
    this.#mediaData = { url };
  }

  #setupOptions(customOptions) {
    this.#options = Object.assign({}, this.#defaultOptions, customOptions);
  }

  getElement() {
    return this.#audioElement;
  }

  getState() {
    return this.#audioElement.paused ? 'PAUSED' : 'PLAYING';
  }

  play() {
    const promise = this.#audioElement.play();

    if (promise !== undefined) {
      promise.catch((error) =>  this.#handleErrorState(error));
    }
  }

  pause() {
    const promise = this.#audioElement.pause();

    if (promise !== undefined) {
      promise.catch((error) =>  this.#handleErrorState(error));
    }
  }

  stop() {
    this.pause();
    this.#audioElement.currentTime = 0;
  }

  updateOnRender(hotspotArea) {
    if (!hotspotArea) return;

    this.#elements.hotspot = hotspotArea;

    const actualState = this.getState();
    const eventType = actualState === 'PLAYING' ? 'play' : 'pause';

    this.#changeHotspotIcon(eventType);
  }

  #changeHotspotIcon(eventType) {
    if (!this.#elements.hotspot) return;

    const hotspotIconElement = this.#elements.hotspot.querySelector('.hotspot-icon');

    if (!hotspotIconElement) return;

    if (eventType === 'destroy' || eventType === 'pause') {
      hotspotIconElement.classList.remove('hotspot-audio-off-icon');
      return;
    }

    hotspotIconElement.classList.add('hotspot-audio-off-icon');
  }

  #handleChangeState(event) {
    const self = this.instance;
    const eventType = event.type;

    self.#changeHotspotIcon(eventType);

    switch (eventType) {
      case 'play':
        const currentPages = (APP.Book.getCurrentPageForLogs() || []).map(page => +page + 1);

        APP.Book.pauseAllEmbededVideos();
        APP.EMBED.pauseAllPlayersOnPage(currentPages);

        if (self.#options.showPlayer) {
          self.#elements.modal.removeAttribute('hidden');
          self.#elements.modal.style.display = 'flex';
        } else {
          self.#elements.modal.setAttribute('hidden', '');
          self.#elements.modal.style.display = 'none';
        }

        break;

      case 'pause':
        break;

      case 'ended':
        if (self.#options.loop) {
          self.#audioElement.currentTime = 0;
          self.play();
        }
        break;
      
      default:
        console.log('Unhandled state', eventType);
        break;
    }
  }

  #handleErrorState(error) {
    const errorName = error.name;

    switch (errorName) {
      case 'NotAllowedError':
        this.#elements.modal.setAttribute('hidden', '');

        if (this.#elements.modal.errorCallbacks && this.#elements.modal.errorCallbacks[errorName]) {
          this.#elements.modal.errorCallbacks[errorName]();
        }

        break;
      
      default:
        console.log('Unhandled error', error);
        break;
    }
  }

  #setNodeAttributes() {
    if (this.#options.autoplay) {
      this.#audioElement.setAttribute('autoplay', '');
    }

    if (this.#options.controls) {
      this.#audioElement.setAttribute('controls', '');
    }

    this.#audioElement.id = this.#options.id;
    this.#audioElement.page = this.#options.page;
    this.#audioElement.showPlayer = this.#options.showPlayer;
    this.#audioElement.controlsList = this.#options.controlsList;

    if (!this.#options.showPlayer) {
      this.#elements.modal.setAttribute('hidden', '');
      this.#elements.modal.style.display = 'none';
    }

    if (Math.max(this.#options.start, 0) > 0) {
      this.#audioElement.currentTime = this.#options.start;
    }
  }

  #bindEvents() {
    this.#audioElement.addEventListener('play', this.#handleChangeState);
    this.#audioElement.addEventListener('pause', this.#handleChangeState);
    this.#audioElement.addEventListener('ended', this.#handleChangeState);
  }

  #unbindEvents() {
    this.#audioElement.removeEventListener('play', this.#handleChangeState);
    this.#audioElement.removeEventListener('pause', this.#handleChangeState);
    this.#audioElement.removeEventListener('ended', this.#handleChangeState);
  }

  #initialize() {
    this.#audioElement = document.createElement('audio');
    this.#audioElement.className = 'modal__body__audio';

    this.#bindEvents();
    this.#setNodeAttributes();

    this.#audioElement.src = this.#mediaData.url;

    this.#elements.audioWrapper.appendChild(this.#audioElement);

    if (this.#options.autoplay) {
      this.play();
    }
  }

  destroy() {
    this.#changeHotspotIcon('destroy');

    if (!this.#audioElement) return;

    this.#unbindEvents();
    
    this.#audioElement.remove();
  }
}

APP.Audio = Audio;
