'use strict';

var Page = function (model) {
  const self = this;
  const SVG_IMAGE_VERSION = true;

  const version = Number(model.version) && Number(model.version) > 0 ? `&version=${model.version}` : '';

  const HOTSPOT_SIZE_FACTOR = 6; 
  const HOTSPOT_CIRCLE_SIZE_FACTOR = 1.75;
  const HOTSPOT_ANIMATION_OFF_CLASSNAME = 'animation-off';
  const HOTSPOT_ANIMATING_CLASSNAME = 'animating';

  const BOOK_IMG_API = `${APP.PATH_V2IMAGE}?iid=${model.bookId}&pid=${model.publisherId}&page=${model.page}${version}`;
  const WEBP = `&webp=${model.Book.useWebP() ? 1 : 0}`;

  const pageModel = {
    vector: `${APP.PATH_V2SVG}?iid=${model.bookId}&pid=${model.publisherId}&page=${model.page}`,
    raster: {
      400: `${BOOK_IMG_API}&size=400&hidetext=0${WEBP}`,
      800: `${BOOK_IMG_API}&size=1200&hidetext=0${WEBP}`,
      1200: `${BOOK_IMG_API}&size=${is_retina() ? '1600' : '1200'}&hidetext=0${WEBP}`,
      1600: `${BOOK_IMG_API}&size=${is_retina() ? '1600' : '1600'}&hidetext=0${WEBP}`,
    },
    model: {
      title: '',
      texts: '',
      page: model.page,
      publisherId: model.publisherId,
      bookId: model.bookId,
      height: model.height,
      width: model.width,
      pageReady: model.pageReady,
      svg: model.svg,
      double: model.width >= model.height,
      publishedAt: model.publishedAt,
      cloudfront: model.cloudfront,
      cloudfrontTxt: model.cloudfrontTxt,
      version: model.version,
      rotation: model.rotation,
      density: model.density
    },
    Book: model.Book,
  };

  const imageCache = {};
  const pageHotspots = pageModel.Book.getInteractivities(pageModel.model.page);

  var pageNode = null;
  let shadowClass = pageModel.model.page % 2 == 0 ? (APP.Book.getBookModelSettings().show_right_to_left ? '--right' : '--left') : (APP.Book.getBookModelSettings().show_right_to_left ? '--left' : '--right');
  
  if (!APP.Book.isDoublePage() && APP.Book.getBookModelSettings().show_right_to_left) shadowClass += ' --single';

  const getInsideShadowClassName = () => {
    return !model.Book.getBookModelSettings().show_shadow_inside ? 'hideshadow' : '';
  };

  const getPageImagesHTML = () => {
    let pageImages;
    const hideShadowClassName = getInsideShadowClassName();

    if (pageModel.Book.showPageLikeIMG()) {
      pageImages = `
				<img class="back noselect _page_raster_${pageModel.model.page} ${shadowClass} ${hideShadowClassName}" />
				<img class="front noselect _page_vector_${pageModel.model.page}" style="top: 0; position: absolute;" />
			`;
    } else {
      pageImages = `
        <div class="back noselect _page_raster_${pageModel.model.page} ${shadowClass} ${hideShadowClassName}">
          <div class="front noselect _page_vector_${pageModel.model.page}"></div>
          <div class="front odblask"></div>
        </div>
      `;
    }

    if (SVG_IMAGE_VERSION) {
      pageImages = `
        <div class="back noselect _page_raster_${pageModel.model.page} ${shadowClass} ${hideShadowClassName}">
          <img class="front noselect _page_vector_${pageModel.model.page}" style="top: 0; position: absolute;" />
          <div class="front odblask"></div>
        </div>
      `;
    }

    return pageImages;
  };

  this.render = function () {
    const pageImages = getPageImagesHTML();
    
    pageNode = document.querySelector(`._page_${pageModel.model.page}`);

    pageNode.insertAdjacentHTML('afterbegin', `<div>${pageImages}</div>`);
    pageNode.raster = document.querySelector(`._page_raster_${pageModel.model.page}`);
    pageNode.vector = document.querySelector(`._page_vector_${pageModel.model.page}`);
    pageNode.vector.style.display = 'none';
    pageNode.page = pageModel.model.page;
    pageNode.bookId = pageModel.model.bookId;
    pageNode.publisherId = pageModel.model.publisherId;
  };

  var _clone = null;
  this.clone = function () {
    if (!_clone) {
      const el = self.render();
      self.visibility();

      _clone = el.cloneNode(true);
      _clone.className.add('__clone');
    }

    return _clone;
  };

  const setBlob = function (el, attr, def, callback) {
    if (!el || el._blobSetted) return;

    const check = pageModel.Book.getFromWebWorkerCache(def);

    if (check) {
      el._blobSetted = true;
      el.setAttribute(attr, check);

      if (is_function(callback)) callback();
    } else {
      setTimeout(function () {
        setBlob(el, attr, def);
      }, 500);
    }
  };

  let controllerPreload;
  try {
    controllerPreload = new AbortController();
  } catch (error) {
    //for old safari versions
    controllerPreload = window.AbortController;
  }

  this.visibility = async function () {
    if (pageNode.dataset.loading === 'end') return;

    const preloadUrl = self.getCover();
    
    pageNode.classList.add('__load');
    pageNode.dataset.preload = 'start';

    const raster = new Promise((resolve, reject) => {
      if (controller) {
        controllerPreload.signal.addEventListener('abort', reject);
      }

      const rasterPreload = new Image();
      rasterPreload.onload = () => resolve(rasterPreload.src);
      rasterPreload.onerror = (error) => reject((pageNode.dataset.preload = ''));

      rasterPreload.src = preloadUrl;
      pageNode.setAttribute('page', model.page);
    });

    raster
      .then((value) => {
        requestAnimationFrame(function () {
          if (pageModel.Book.showPageLikeIMG()) {
            pageNode.raster.setAttribute('src', preloadUrl);
          } else {
            const actualImageSize = pageNode.raster.style.backgroundImage.match(/_(\d+)\./);
            const preloadImageSize = preloadUrl.match(/_(\d+)\./);

            if (actualImageSize && preloadImageSize) {
              if (parseInt(actualImageSize[1]) <= parseInt(preloadImageSize[1])) {
                pageNode.raster.style.backgroundImage = `url("${preloadUrl}")`;
              } else {
                // oLogs.log_send_error(model.bookId, model.publisherId, {
                //   acc: 'SizeError',
                //   data: `Cover after Raster image. actual_size: ${preloadImageSize[1]}. PreloadUrl: ${preloadUrl}. BackgroundImage: ${pageNode.raster.style.backgroundImage}.`,
                // });

                console.log('Size error:', parseInt(actualImageSize[1]), parseInt(preloadImageSize[1]));
              }
            } else {
              pageNode.raster.style.backgroundImage = `url("${preloadUrl}")`;
            }
          }

          pageNode.classList.remove('__load');
          pageNode.dataset.preload = 'end';
        });
      })
      .catch((error) => {
        pageNode.dataset.preload = '';
        const pageNodePage = pageNode.getAttribute('page');

        if (pageNodePage != null && APP.Book.getModel().pages_info && APP.Book.getModel().pages_info[pageNodePage - 1].pageReady == false) {
          pageNode.classList.remove('__load');
          pageNode.classList.add('__converting');
        }
        
        console_log('ABORT CONTROLLER on page: ', pageNode.getAttribute('page'));
      });
  };
  
  this.loadBluredImage = async () => {
    if (!pageModel.Book.getModel().hasOwnProperty('is250BluredExists') || !pageModel.Book.getModel().is250BluredExists) return;
    if (pageNode.dataset.loading === 'end') return;

    const preloadUrl = self.getBlured();

    return new Promise((resolve, reject) => {
      if (controller) {
        controllerPreload.signal.addEventListener('abort', reject);
      }
      
      const bluredPreload = new Image();
      bluredPreload.onload = () => resolve(bluredPreload.src);
      bluredPreload.onerror = (error) => reject(error);

      bluredPreload.src = preloadUrl;
    });
  };

  this.stopPreload = function () {
    pageNode.dataset.preload = 'end';
    controllerPreload.abort();
  };

  this.getModel = function () {
    return pageModel;
  };

  this.getPageNumber = function () {
    return pageModel.model.page;
  };

  this.getPageReady = function () {
    return pageModel.model.pageReady;
  };

  this.getCover = function () {
    return HELPER.pathToTxtImg2(pageModel, 400);
  };

  this.getBlured = function () {
    if (pageModel.Book.getModel().hasOwnProperty('is250BluredExists') && pageModel.Book.getModel().is250BluredExists) {
      return HELPER.pathToTxtImg2(pageModel, 250).replace('_250.', '_250_blured.');
    }

    return HELPER.pathToTxtImg2(pageModel, 400);
  };

  this.getCoverThumb = function () {
    return HELPER.pathToTxtImg2(pageModel, 400);
  };

  this.hideSVG = function () {
    if (!pageNode || !_wasRendered || !pageModel.Book.useVector()) return;

    pageNode.vector.style.visibility = 'hidden';
  };

  this.showSVG = function () {
    if (!pageNode || !_wasRendered || !pageModel.Book.useVector()) return;

    pageNode.vector.style.visibility = '';
  };

  this.preview = function () {
    if (!pageNode) return;
    if (_wasRendered) return;

    pageNode.raster.setAttribute('src', self.getCover('preview'));
    pageNode.raster.style.opacity = 1;
  };

  this.getPage = function () {
    return pageNode;
  };

  let controller;
  try {
    controller = new AbortController();
  } catch (error) {
    controller = window.AbortController;
  }
  
  this.setPageImages = async function (second_try = false) {
    var rasterUrl = HELPER.pathToTxtImg2(pageModel, false, second_try);
    var rasterSize = HELPER.getSizeOfTxtImg2(pageModel);
    let _self = this;

    pageNode.dataset.loading = 'start';

    let raster = new Promise((resolve, reject) => {
      if (controller) controller.signal.addEventListener('abort', reject);
      let rasterImage = new Image();
      rasterImage.onload = () => {
        // przywracanie notatek, jesli nie jest to wersja mobilna
        if (!is_mobile() && APP.Note) {
          APP.Note.restoreNotesOnLoadedPage(pageModel.model.page);
          APP.Note.showNotesByDependsOnWidth();
        }

        hotspotsContainerOn();
        resolve(rasterImage.src);
      };
      rasterImage.onerror = (error) => reject((pageNode.dataset.loading = ''));
      rasterImage.src = rasterUrl;
    });

    raster
      .then((values) => {
        pageNode.dataset.loading = 'end';
        requestAnimationFrame(function () {
          if (pageModel.Book.showPageLikeIMG()) {
            pageNode.raster.setAttribute('src', `${rasterUrl}`);
          } else {
            if (pageNode.dataset.preload === 'end' && pageNode.style.display === 'block') {
              const bgString = setAndOrderBackground(pageNode.raster.style.backgroundImage, rasterUrl);
              pageNode.raster.style.backgroundImage = bgString;
              // if (pageNode.raster.style.backgroundImage.includes(rasterUrl)) {
              //   const backImgArr = pageNode.raster.style.backgroundImage.split(',');
              //   const index = backImgArr.findIndex(item => item.includes(rasterUrl));
              //   backImgArr.splice(index, 1);
              //   backImgArr.unshift(`url('${rasterUrl}')`);
              //   let backImg = backImgArr.join(', ');
              //   pageNode.raster.style.backgroundImage = `${backImg}`;
              // } else {
              //   pageNode.raster.style.backgroundImage = `url("${rasterUrl}"), ${pageNode.raster.style.backgroundImage}`;
              // }
            } else {
              pageNode.raster.style.backgroundImage = `url("${rasterUrl}")`;
            }
          }

          pageNode.raster.setAttribute('size', rasterSize);
          pageNode.classList.remove('__load');
          if (document.getElementsByClassName('loader') && !pageModel.Book.getLoaderStatus()) {
            pageModel.Book.bookLoaded();
          }
        });
      })
      .catch((error) => {
        pageNode.dataset.loading = '';
        if (error.type == 'abort') {
          console_log('Cancel image loading');
        } else {
          if (!second_try) {
//            oLogs.log_send_error(model.bookId, model.publisherId, {
//              acc: 'setPageError',
//              data: 'ERR:' + rasterUrl + ' L:252',
//            });
            _self.setPageImages(true);
          } else {
            
            oLogs.log_send_error(model.bookId, model.publisherId, {
              acc: 'setPageError2',
              data: 'ERR:' + rasterUrl + ' L:252',
            });
          }
          // console_log("setPageError", error);
        }
      });
  };

  const setAndOrderBackground = (actualBackground, newUrl) => {
    if (actualBackground.includes(newUrl)) {
      const backImgArr = actualBackground.split(',');
      const index = backImgArr.findIndex(item => item.includes(newUrl));

      backImgArr.splice(index, 1);
      backImgArr.unshift(`url('${newUrl}')`);

      let backImg = backImgArr.join(',');
      
      if (backImg[0] == ' ') {
        backImg = backImg.slice(1);
      }

      actualBackground = `${backImg}`;
    } else {
      actualBackground = `url('${newUrl}'), ${actualBackground}`;
    }

    return actualBackground;
  };

  this.setZoomedImageAfterSvgError = async function () {
    const raster = new Promise((resolve, reject) => {
      const rasterImage = new Image();
      rasterImage.onload = () => resolve(rasterImage.src);
      rasterImage.onerror = () => reject();
      rasterImage.src = HELPER.pathToTxtImg2(pageModel, 1600);
    });

    raster
      .then((imageUrl) => {
        requestAnimationFrame(function () {
          if (pageModel.Book.showPageLikeIMG()) {
            pageNode.raster.setAttribute('src', imageUrl);
          } else {
            pageNode.raster.style.backgroundImage = `url("${imageUrl}")`;
          }
        });
      })
      .catch((error) => {
        console_log('error', error);
      });
  };

  const loadImage = (src) => {
    if (imageCache[src]) return Promise.resolve(imageCache[src]);

    return new Promise((resolve, reject) => {
      const img = new Image();
      img.onload = () => {
        imageCache[src] = img;
        resolve(img);
      };
      img.onerror = reject;
      img.src = src;
    });
  };

  this.setVectorForZoom = async function () {
    if (this.getModel().model.svg === false) {
      this.setZoomedImageAfterSvgError();
      return; //uciekamy bo moze svg byc walniety
    }
    
    if (pageNode.vector.style.backgroundImage || pageNode.vector.src) return;
    
    const imagePaths = [HELPER.pathToSvg(pageModel), HELPER.pathToNoTxtImg(pageModel)];

    const svgWrapper = document.querySelector('.svg__wrapper');
    const leftOrRight = pageNode.raster.classList.contains('--left') ? 'left' : 'right';
    const isBookView = pageModel.Book.isBookView();

    Promise.all(imagePaths.map(loadImage)).then(([svgImg, noTxtImg]) => {
      requestAnimationFrame(function () {
        if (pageModel.Book.showPageLikeIMG()) {
          pageNode.raster.src = noTxtImg.src;
          pageNode.vector.src = svgImg.src;
        } else if (SVG_IMAGE_VERSION) {
          // const rasterBackground = HELPER.pathToNoTxtImg(pageModel, 1600);
          pageNode.raster.style.backgroundImage = `url("${noTxtImg.src}")`;
          pageNode.vector.src = svgImg.src;

          if (pageModel.model.rotation && pageModel.model.rotation != 0) {
            pageNode.vector.style.transform = `rotate(-${pageModel.model.rotation}deg)`;
          }
        } else {
          //jezeli zoom i svg to pokazujemy 1600 no text
          // const rasterBackground = HELPER.pathToNoTxtImg(pageModel, 1600);
          pageNode.vector.style.backgroundImage = `url("${svgImg.src}")`;
          pageNode.raster.style.backgroundImage = `url("${noTxtImg.src}")`;
        }
          
        if (is_safari() && !is_mobile() && svgWrapper) {
          if (!isBookView) {
            svgWrapper.querySelectorAll('.svg__page').forEach(page => page.style.display = 'none');
          }

          if (svgWrapper.querySelector(`.svg__page--${leftOrRight} img`)) {
            svgWrapper.querySelector(`.svg__page--${leftOrRight} img`).setAttribute('src', svgImg.src);
          } else {
            const image = document.createElement('img');
            image.setAttribute('src', svgImg.src);
            svgWrapper.querySelector(`.svg__page--${leftOrRight}`).appendChild(image);
          }

          if (!isBookView) {
            svgWrapper.querySelector(`.svg__page--${leftOrRight}`).style.display = 'block';
          }
        }

        pageNode.raster.setAttribute('size', is_mobile() ? 1200 : 1600);
        pageNode.vector.style.display = 'block';
      });
    })
    .catch((error) => {
      //cos poszlo nie tak upewnijmy sie ze po zoomie raster nie 1200
      if (is_mobile()) return;

      const rasterBackground = pageNode.raster.style.backgroundImage.replace('size=1200', 'size=1600');
      pageNode.raster.style.backgroundImage = rasterBackground;

      console_log('error', error);
    });
  };

  this.removeShadow = () => {
    pageNode.raster.classList.add('shadowRemoved');
  };

  this.addShadow = () => {
    const shadowClassArray = shadowClass.split(' ');

    shadowClassArray.forEach(className => pageNode.raster.classList.add(className));
    pageNode.raster.classList.remove('shadowRemoved');
  };

  this.clearVectorImage = () => {
    if ((pageNode && pageNode.vector.style.backgroundImage) || (pageNode && pageNode.vector.src)) {
      //tu powinno byc prawdzenie jaki rozmiar
      const rasterUrl = HELPER.pathToTxtImg2(pageModel);

      loadImage(rasterUrl).then((rasterImg) => {
        requestAnimationFrame(function () {
          //tu powinno byc prawdzenie jaki rozmiar
          if (pageModel.Book.showPageLikeIMG()) {
            pageNode.raster.src = rasterImg.src;
            pageNode.vector.removeAttribute('src');
          } if (SVG_IMAGE_VERSION) {
            pageNode.raster.style.backgroundImage = `url("${rasterImg.src}")`;
            pageNode.vector.removeAttribute('src');
          } else {
            pageNode.raster.style.backgroundImage = `url("${rasterImg.src}")`;
            pageNode.vector.style.backgroundImage = '';
          }

          pageNode.vector.style.display = 'none';
        });
      });
    }

    if (is_safari()) {
      const svgWrapper = document.querySelector('.svg__wrapper');
      const isBookView = pageModel.Book.isBookView();

      if (!svgWrapper) return;
      
      const svgPages = Array.from(svgWrapper.querySelectorAll('.svg__page'));
      
      if (svgPages.length === 0) return;
      
      requestAnimationFrame(function () {
        svgPages.forEach(page => {
          page.innerHTML = '';

          if (!isBookView) page.style.display = 'none';
        });
      });
    }
  };

  this.cancelImageLoading = function () {
    controller.abort();
    pageNode.dataset.loading = 'end';
  };

  this.isAbortController = function () {
    return controllerPreload && controller;
  };

  const sendGtmEvent = (data) => {
    if (!window.gtmEvent) return;
    
    window.gtmEvent(data);
  };

  const generateHash = async (data, prefix = null, suffix = null) => {
    const HASH_LENGTH = 10;
    const encoder = new TextEncoder();
    const dataString = JSON.stringify(data);
    const encodedData = encoder.encode(dataString);
  
    const hashBuffer = await crypto.subtle.digest("SHA-256", encodedData);
  
    const base64Hash = btoa(String.fromCharCode(...new Uint8Array(hashBuffer)));
    const shortHash = base64Hash.replace(/[+/=]/g, '').slice(0, HASH_LENGTH);
  
    return [prefix, shortHash, suffix].filter(Boolean).join('_');
  };

  const getHotspotDataForLog = (type, hotspotModel) => {
    let data;

    switch (type) {
      case 'link':
        const linkType = hotspotModel.data.type;

        data = linkType === 'page' ? `#page#${hotspotModel.data.value}` : hotspotModel.data.value;
        break;

      case 'gif':
      case 'gallery':
        data = hotspotModel.data[0];
        break;

      case 'video':
      case 'audio':
        data = hotspotModel.data.value;
        break;

      case 'product-info':
        data = {
          uid: hotspotModel.data.uid,
          name: hotspotModel.data.name
        };
        break;
      
      default:
        break;
    }

    return data;
  };

  const prepareHotspotDataForLog = async (type, hotspotModel) => {
    const hotspotDataForLog = getHotspotDataForLog(type, hotspotModel);

    if (hotspotModel.uid && hotspotModel.uid.indexOf('u_') === 0) {
      return Object.assign({}, hotspotModel, {
        logData: {
          hashUId: hotspotModel.uid,
          data: hotspotDataForLog
        }
      });
    }

    let hashSuffix = '';
    let hashPrefix = 'v';

    if (pageHotspots && pageHotspots.areas && pageHotspots.areas.length > 0) {
      const pageHotspotsAreas = pageHotspots.areas;

      const sameHotsposts = pageHotspotsAreas
        .filter(hotspot => {
          if (hotspotModel.type !== 'link') return hotspot;

          const hotspotConverstionStatus = hotspot.data.conversion || null;
          const actualHotspotConverstionStatus = hotspotModel.data.conversion || null;

          return hotspotConverstionStatus === actualHotspotConverstionStatus;
        })
        .filter(hotspot => {
          if (hotspot.type !== type) return null;

          switch (type) {
            case 'link':
            case 'video':
            case 'audio':
              return hotspot.data.value === hotspotModel.data.value;

            case 'gif':
            case 'gallery':
              return hotspot.data[0] === hotspotModel.data[0];

            case 'product-info':
              return hotspot.data.name === hotspotModel.data.name;
            
            default:
              return null;
          }
      });

      if (sameHotsposts.length > 1) {
        const actualHotspotIndex = sameHotsposts.findIndex(hotspot => hotspot.area_number === hotspotModel.area_number);

        if (actualHotspotIndex !== -1) {
          hashSuffix = actualHotspotIndex + 1;
        }
      }
    } 
    
    const hashData = {
      page: pageModel.model.page - 1,
      issueId: model.bookId,
      data: hotspotDataForLog
    };
    
    const generatedHashUId = await generateHash(hashData, hashPrefix, hashSuffix);

    return Object.assign({}, hotspotModel, {
      logData: {
        hashUId: generatedHashUId,
        data: hotspotDataForLog
      }
    });
  };

  const hotspotEventHandler = function (event, eventArea = null, hotspotModel) {
    const area = eventArea === null ? event.currentTarget : eventArea;
    const hotspot = JSON.parse(area.dataset.areaData);
    const type = area.dataset.type;

    prepareHotspotDataForLog(type, hotspotModel).then(modifiedHotspotModel => {
      oLogs.log_set_area(type, modifiedHotspotModel, pageModel.model.page - 1);
    });
    
    if (type === 'link') {
      let data = hotspot.value;
      const linkType = hotspot.type;

      sendGtmEvent({ 
        type: linkType,
        page: hotspotModel.page, 
        value: data 
      });

      if (linkType == 'page') {
        pageModel.Book.setPage(data, 'hotspot');
      }
      
      if (linkType == 'email' && data.length > 0) {
        const url = `mailto:${data}`;
        window.open(url.replace('mailto:mailto:', 'mailto:'));
      }

      if (linkType == 'phone' && data.length > 0) {
        window.open(`tel:${data}`);
      }
      
      if (linkType == 'url' && data.length > 0) {
        if (data.indexOf('mailto:') !== -1) {
          window.open(data.replace('mailto:mailto:', 'mailto:'));
        } else {
          if (data.indexOf('http') == -1) {
            data = `https://${hotspot.value}`;
          }

          if (!('display' in hotspot) || (('display' in hotspot) && hotspot.display == 'new-tab')) {
            window.open(data, '_blank');
          } else {
            APP.EMBED.pauseAllPlayersOnPage(hotspotModel.page);
            APP.EMBED.pauseBackgroundAudio();
            pageModel.Book.pauseAllEmbededVideos();
            
            if (data.indexOf('.pdf') !== -1 && is_mobile()) {
              window.open(data, '_blank');
            } else {
              APP.Book.openLinkInFrame(data);
            }
          }
        }
      }
    } else if (type === 'video') {
      self.hotspotsOff();

      let url = hotspot.value;

      if (hotspot.value.includes('youtube.com/')) {
        url = url.replace('watch?v=', 'embed/');
      } else if (hotspot.value.includes('youtu.be/')) {
        url = url.replace('.be/', 'be.com/embed/');
      } else if (hotspot.value.includes('vimeo.com')) {
        url = url.replace('https://vimeo.com', 'https://player.vimeo.com/video');
      } else {
        url = HELPER.getHotspotMediaUrl(url);;
      }

      if (hotspot.type === 'embedded') {
        const isUploaded = isUploadedVideo({ data: hotspot });
        const EMBEDDED_VIDEO_MIN_SIZE = 100;
        const shouldShowInPopup = Math.min(area.getBoundingClientRect().width, area.getBoundingClientRect().height) <= EMBEDDED_VIDEO_MIN_SIZE;

        if (isUploaded || url.includes('https://player.vimeo.com/video') || url.includes('youtube.com/')) {
          const id = hotspotModel.uid || hotspotModel.area_number;

          if (isUploaded && APP.EMBED.isPlayerAlreadyInitialized(id)) {
            return;
            const player = APP.EMBED.getPlayerByAreaId(id);

            if (player && player.lastControlsInteractionTime) {
              if (Math.abs(player.lastControlsInteractionTime - new Date()) < 400) return;
            }
          }

          if ((is_mobile() && !is_tablet()) || shouldShowInPopup) {
            url = HELPER.setParametrToUrlIfNotExists(url, 'controls');
            
            if (hotspotModel.data.muted) {
              url = HELPER.setParametrToUrlIfNotExists(url, 'mute');
            }
            
            if (url.includes('/shorts/')) {
              url = url.replace('/shorts/', '/embed/', url)
            }

            pageModel.Book.setDataToMediaViewer(url, 'video', Object.assign({}, hotspotModel, {
              _action: 'INTERACTION'
            }), area);
          } else {
            videoAutoplay(Object.assign({}, hotspot, { autoplay: true }), area, pageModel.model.page, hotspotModel);
          }
        }
      } else {
        url = HELPER.setParametrToUrlIfNotExists(url, 'autoplay');

        url = url.replace('&c', '?c');
        url = url.replace('&t=', '&start=');
        pageModel.Book.setDataToMediaViewer(url, 'video', Object.assign({}, hotspotModel, {
          _action: 'INTERACTION'
        }), area);
      }
    } else if (type === 'gallery') {
      self.hotspotsOff();

      if (hotspot.length > 0) {
        const data = hotspot.map((imgUrl) => imgUrl ? HELPER.getHotspotMediaUrl(imgUrl) : null).filter(Boolean);

        APP.EMBED.pauseAllPlayersOnPage(hotspotModel.page);
        pageModel.Book.pauseAllEmbededVideos();
        
        pageModel.Book.setDataToMediaViewer(data, 'gallery', Object.assign({}, hotspotModel, {
          _action: 'INTERACTION'
        }), area);
      }
    } else if (type === 'audio') {
      const data = HELPER.getHotspotMediaUrl(hotspot.value);

      APP.EMBED.pauseAllPlayersOnPage(hotspotModel.page);
      pageModel.Book.setDataToMediaViewer(data, 'audio', Object.assign({}, hotspotModel, {
        _action: 'INTERACTION'
      }), area);
    } else if (type === 'product-info') {
      APP.Products.renderProductPopup(hotspot);
    }
  };

  const hotspotsContainerOn = () => {
    if (!pageNode) return;

    const hotspotsContainer = pageNode.querySelector('.hotspots-container');
    
    if (!hotspotsContainer) return;

    hotspotsContainer.style.display = 'block';

    const visibilityTimeout = setTimeout(() => {
      hotspotsContainer.style.opacity = '1';
      hotspotsContainer.style.visibility = 'visible';

      clearTimeout(visibilityTimeout);
    }, 50);
  };

  this.hotspotsOn = function () {
    if (pageNode.style.display !== 'block' || !pageNode.hotspotArea) return;
    if (pageNode.dataset.animation === 'on') return;

    pageNode.hotspotArea.forEach((area) => {
      if (area.hidden) return;

      const circle = area.querySelector('.hotspot-circle');
      const shadow = area.querySelector('.hotspot-shadow');

      if (!circle.hidden) {
        restartRotatingAnimation(circle);

        if (shadow) {
          shadow.classList.remove(HOTSPOT_ANIMATION_OFF_CLASSNAME);
        }
      } else {
        setTimeout(() => {
          area.classList.remove(HOTSPOT_ANIMATION_OFF_CLASSNAME);
        }, 200);
      }
    });
    
    pageNode.dataset.animation = 'on';
  };

  this.haveHotspots = function () {
    return !!pageNode.hotspotArea;
  };

  this.hotspotsOff = function () {
    if (!pageNode.hotspotArea) return;
    if (pageNode.dataset.animation === 'off') return;

    pageNode.hotspotArea.forEach((area) => {
      if (area.hidden) return;

      const circle = area.querySelector('.hotspot-circle');
      const shadow = area.querySelector('.hotspot-shadow');

      if (!circle.hidden) {
        stopRotatingAnimation(circle);

        if (shadow) {
          shadow.classList.add(HOTSPOT_ANIMATION_OFF_CLASSNAME);
        }
      } else {
        area.classList.add(HOTSPOT_ANIMATION_OFF_CLASSNAME);
      }
    });

    pageNode.dataset.animation = 'off';
  };

  const getAreaSizeByPageSize = (area) => {
    const page = this.getPage();

    const areaWidth = Number(area.width);
    const areaHeight = Number(area.height);
    const pageWidth = Math.ceil(page.getBoundingClientRect().width);
    const pageHeight = Math.ceil(page.getBoundingClientRect().height);

    return {
      width: Math.ceil((pageWidth * areaWidth) / 100),
      height: Math.ceil((pageHeight * areaHeight) / 100),
    };
  };

  const videoAutoplay = (hotspot, area, page, hotspotModel) => {
    const id = hotspotModel.uid || hotspotModel.area_number;

    if (APP.EMBED.isPlayerAlreadyInitialized(id)) return;

    self.hotspotsOff();

    let url = hotspot.value;
    let mediaContainer = '';

    mediaContainer = document.createElement('div');
    mediaContainer.className = 'embedded-player'
    mediaContainer.name = 'frame1';
    mediaContainer.style.position = 'absolute';
    mediaContainer.style.left = '0px';
    mediaContainer.style.top = '0px';
    mediaContainer.style.width = '100%';
    mediaContainer.style.height = '100%';
    mediaContainer.style.opacity = '1';
    mediaContainer.style.zIndex = '9999';
    mediaContainer.style.pointerEvents = 'all';

    area.appendChild(mediaContainer);
    APP.EMBED.pauseAllPlayersOnPage(page);

    if (isUploadedVideo({ data: hotspot })) {
      const areaSizes = area.dataset && area.dataset.width && area.dataset.height ? { width: +area.dataset.width, height: +area.dataset.height } : getAreaSizeByPageSize(hotspotModel);

      mediaContainer.id = `player_pub_${Date.now()}`;

      APP.EMBED.inializeInternalPlayer(mediaContainer, {
        url: hotspot.value,
        poster: hotspot.thumbnail,
        start: Math.max(0, +hotspot.start),
        muted: !!hotspot.muted,
        controls: !!hotspot.controls,
        autoplay: !!hotspot.autoplay_player,
        loop: !!hotspot.loop,
        preload: 'auto',
        controlslist: 'nodownload noremoteplayback noplaybackrate',
        playsinline: true,
        webkitPlaysinline: true,
        x5Playsinline: true,
        width: areaSizes.width,
        height: areaSizes.height,
      }, page, id, area)

      return;
    }

    if (url.includes('youtube.com/') || url.includes('youtu.be/')) {
      const videoId = HELPER.getVideoIdFromUrl(url, 'youtube');

      mediaContainer.id = `player_yt_${Date.now()}`;
      
      APP.EMBED.inializeYoutubePlayer(mediaContainer, {
        videoId: videoId,
      }, {
        autoplay: hotspot.autoplay_player,
        start: parseInt(hotspot.start) || 0,
        volume: hotspot.muted ? 0 : 100,
        controls: hotspot.controls ? 1 : 0
      }, page, id)
    }

    if (url.includes('vimeo.com')|| url.includes('player.vimeo.com/video')) {
      const videoId = HELPER.getVideoIdFromUrl(url, 'vimeo');
      const urlObject = new URL(url);
      let options = {
        controls: hotspot.controls ? 1 : 0,
      };

      if (urlObject.searchParams.get('h')) {
        const hash = urlObject.searchParams.get('h');

        url = url.split('?')[0];
        url += url.endsWith('/') ? hash : `/${hash}`;

        options = Object.assign(options, {
          url
        });
      } else {
        options = Object.assign(options, {
          id: videoId
        });
      }

      mediaContainer.id = `player_vm_${Date.now()}`;

      APP.EMBED.inializeVimeoPlayer(mediaContainer, options, {
        volume: hotspot.muted ? 0 : 1,
        start: parseInt(hotspot.start) || 0,
        autoplay: hotspot.autoplay_player,
      }, page, id);
    }
  }

  const getLastAutoplayElementOnPage = function (page) {
    const pagesCount = parseInt(pageModel.Book.getModel().pages);
    const isDoublePage = pageModel.Book.isDoublePage();
    let interactivitesOnThisSpread = [];

    function getInteractivityByPage(page) {
      const interactivites = APP.Book.getInteractivities(page);
      if (!interactivites || !interactivites.areas || !interactivites.areas.length) return [];

      return interactivites.areas.filter(item => item && item.data && ((item.data.autoplay_player || item.data.autoplay) && (item.type == 'video' || item.type == 'audio')));
    }

    if (isDoublePage) {
      if (page != 1 && page != pagesCount) {
        if (page % 2 == 1) {
          interactivitesOnThisSpread = [...getInteractivityByPage(page), ...getInteractivityByPage(page - 1)];
        } else if (page % 2 == 0) {
          interactivitesOnThisSpread = [...getInteractivityByPage(page), ...getInteractivityByPage(page + 1)];
        }
      } else if (page == 1 || page == pagesCount) {
        interactivitesOnThisSpread = [...getInteractivityByPage(page)];
      }
    } else {
      interactivitesOnThisSpread = [...getInteractivityByPage(page)]
    }

    interactivitesOnThisSpread.sort((a, b) => {
      if (a.area_number > b.area_number) return 1;
      if (a.area_number < b.area_number) return -1;
    });
    
    return interactivitesOnThisSpread[interactivitesOnThisSpread.length-1];    
    return interactivitesOnThisSpread.at(-1);;
  }

  this.adjustHotspotsSize = async (width, height) => {
    if (!pageNode.hotspotArea) return;

    const hotspotsContainer = pageNode.querySelector('.hotspots-container');

    if (!hotspotsContainer) return;

    const hotspots = Array.from(hotspotsContainer.querySelectorAll('.hotspot-area'));

    if (hotspots.length === 0) return;

    if (is_fireFox()) {
      hotspotsContainer.style.zIndex = '-1';
      
      await HELPER.waitingPromise();
      
      hotspotsContainer.style.display = 'none';
      hotspotsContainer.style.zIndex = '50';

      const bookSize = APP.Book.getBookSize();

      width = bookSize.bookWidth;
      height = bookSize.bookHeight;
    }

    const hotspotSize = getHotspotSize();

    hotspots.forEach((hotspot) => {
      const hotspotData = JSON.parse(hotspot.dataset.areaData);
      const hotspotType = hotspot.dataset.type;
      const hotspotExtraType = hotspot.dataset.extraType;
      const shadow = hotspot.querySelector('.hotspot-shadow');
      const circle = hotspot.querySelector('.hotspot-circle');

      let isVideoEmbed = false;
      let isGalleryEmbed = false;
      let isGif = false;

      if (hotspotData) {
        isVideoEmbed = hotspotType === 'video' && hotspotData.type === 'embedded';
        isGalleryEmbed = hotspotType === 'gallery' && hotspotExtraType === 'slider';
        isGif = hotspotType === 'gif';
      }

      if (shadow) {
        shadow.style.width = `${hotspotSize}px`;
        shadow.style.height = `${hotspotSize}px`;

        if (isGalleryEmbed || isGif) {
          shadow.style.width = `0px`;
          shadow.style.height = `0px`;
        }
      }

      if (circle && !circle.hasAttribute('hiddenstyle')) {
        circle.style.width = `${hotspotSize}px`;
        circle.style.height = `${hotspotSize}px`;

        if (isVideoEmbed) {
          circle.style.top = `calc(50% - ${(hotspotSize * HOTSPOT_CIRCLE_SIZE_FACTOR) / 2}px)`;
          circle.style.left = `calc(50% - ${(hotspotSize * HOTSPOT_CIRCLE_SIZE_FACTOR) / 2}px)`;
          circle.style.width = `${hotspotSize * HOTSPOT_CIRCLE_SIZE_FACTOR}px`;
          circle.style.height = `${hotspotSize * HOTSPOT_CIRCLE_SIZE_FACTOR}px`;
        }

        if (isGalleryEmbed || isGif) {
          circle.style.width = `0px`;
          circle.style.height = `0px`;
        }
      }

      if (isGalleryEmbed || isGif) {
        const sliderIcon = hotspot.querySelector('.hotspot-gallery__slider-icon');
        const fullscreenIcon = hotspot.querySelector('.hotspot-gallery__fullscreen');

        if (sliderIcon) {
          sliderIcon.style.width = `${hotspotSize * HOTSPOT_CIRCLE_SIZE_FACTOR}px`;
          sliderIcon.style.height = `${hotspotSize * HOTSPOT_CIRCLE_SIZE_FACTOR}px`;
        }
        
        if (fullscreenIcon) {
          fullscreenIcon.style.width = `${hotspotSize * HOTSPOT_CIRCLE_SIZE_FACTOR}px`;
          fullscreenIcon.style.height = `${hotspotSize * HOTSPOT_CIRCLE_SIZE_FACTOR}px`;
        }

        updateSliderSizes();
      }

      if (isVideoEmbed) {
        if (!hotspot.parentNode) return;

        const controls = hotspot.parentNode.querySelectorAll('.player__control');

        if (!controls || !!!controls.length) return;

        Array.from(controls).forEach(control => APP.EMBED.setPlayerControlSizes(control));
      }
    });

    APP.EMBED.hidePlayers();
    APP.EMBED.adjustPlayersSizes();
    
    hotspotsContainerOn();

    APP.EMBED.showPlayers();
  };

  this.getHotspotsContainer = () => {
    return pageNode.querySelector('.hotspots-container');
  }

  const getBookSizeFirefox = () => {
    let bookSize = null;

    while (!bookSize) bookSize = APP.Book.getBookSize();

    return bookSize;
  };

  const isUploadedVideo = (hotspot) => {
    if (!hotspot || !hotspot.data) return false;
    if (!hotspot.data.hasOwnProperty('origin') && !hotspot.data.hasOwnProperty('uploaded')) return false;

    return hotspot.data.origin === 'PUBLUU' || !!hotspot.data.uploaded;
  };

  const getHotspotSize = (FORCE_FACTOR = null) => {
    const factor = FORCE_FACTOR || HOTSPOT_SIZE_FACTOR;
    const bookSize = APP.Book.getBookSize();
    let hotspotSize;

    if (bookSize.bookWidth < bookSize.bookHeight) {
      hotspotSize = Math.floor((bookSize.bookWidth * factor) / 100);
    } else {
      hotspotSize = Math.floor((bookSize.bookHeight * factor) / 100);
    }

    return hotspotSize;
  };

  this.hotspotsRender = async function (width, height) {
    if (pageNode.hotspotArea) return;

    if (is_fireFox()) {
      const bookSize = getBookSizeFirefox();

      width = bookSize.bookWidth;
      height = bookSize.bookHeight;
    }

    const hotspots = pageModel.Book.getInteractivities(pageModel.model.page);
    
    if (!hotspots) return;

    const lastAutoplayElement = getLastAutoplayElementOnPage(pageModel.model.page);
    const hotspotSize = getHotspotSize();

    let hotspotsContainer = pageNode.querySelector('.hotspots-container');
    if (!hotspotsContainer) {
      hotspotsContainer = document.createElement('div');
      hotspotsContainer.className = 'hotspots-container';
    }

    hotspots.areas.forEach((hotspot) => {
      if (Array.isArray(hotspot.data)) {
        const exitsingElements = hotspot.data.filter(item => item && item.length > 0);

        if (!exitsingElements.length) return;
      }

      const areaSizes = getAreaSizeByPageSize(hotspot);
      let hotspotArea = document.createElement('div');

      hotspotArea.style.display = 'none';
      hotspotArea.dataset.width = areaSizes.width;
      hotspotArea.dataset.height = areaSizes.height;

      let is_video_embed = hotspot.type === 'video' && hotspot.data.type === 'embedded';
      hotspotArea.dataset.areaData = JSON.stringify(hotspot.data);

      hotspotArea.className = `hotspot-area hotspot-area_${pageModel.model.page} ${
        hotspot.area_is_visible ? '' : HOTSPOT_ANIMATION_OFF_CLASSNAME
      } ${is_video_embed ? 'video_embeded ' + HOTSPOT_ANIMATION_OFF_CLASSNAME : ''}`;

      hotspotArea.id = `${hotspot.area_number}`;
      hotspotArea.style = `width: ${hotspot.width}%; height: ${hotspot.height}%; top: ${
        hotspot.position_y
      }%; left: ${hotspot.position_x}%; pointer-events:${
        (hotspot.extra && hotspot.extra.type === 'slider') ? 'all' : ((!hotspot.area_is_visible && !hotspot.icon_is_visible) || hotspot.area_is_visible) ? 'all' : 'none'
      }`;

      if (hotspot.type == 'gif') {
        hotspotArea.style.pointerEvents = 'none';
      }

      hotspotArea.dataset.type = hotspot.type;
      let hotspotCircle = ``;

      if ((hotspot.extra && hotspot.extra.type === 'slider') || hotspot.type === 'gif') {
        showHotspotArea(hotspotArea, false);

        hotspotArea.dataset.extraType = hotspot.extra.type;

        if (hotspot.type === 'gif') {
          // gif render after loader animation is ended
          const loaderInterval = setInterval(() => {
            if (APP.Book.getLoaderStatus()) {
              renderGallerySlider(hotspot, hotspotArea, hotspots.page);
              clearInterval(loaderInterval);
            }
          }, 100);
        } else {
          renderGallerySlider(hotspot, hotspotArea, hotspots.page);
        }

        hotspotCircle = `<div class="hotspot-circle"></div>`;
      } else {
        if (is_video_embed) {
          if (hotspot.data.show_preview && hotspot.data.thumbnail && hotspot.data.thumbnail.length > 0) {
            const videoThumbURL = HELPER.getHotspotMediaUrl(hotspot.data.thumbnail);
            
            if (videoThumbURL === "https://cms2.publuu.com/img/") {
              hotspotArea.style.background = "black";
              showHotspotArea(hotspotArea, false);
            } else {
              const image = new Image();
              image.src = videoThumbURL;
              hotspotArea.style.backgroundImage = `url(${videoThumbURL})`;

              image.addEventListener('load', () => {
                showHotspotArea(hotspotArea);
              });
            }
            
            hotspotArea.style.backgroundSize = isUploadedVideo(hotspot) ? 'contain' : 'cover';
            hotspotArea.style.backgroundPosition = 'center';
            hotspotArea.style.backgroundRepeat = `no-repeat`;
          } else {
            showHotspotArea(hotspotArea);
          }
          
          hotspotArea.style.pointerEvents = 'all';
          let div_hotspot_icon = `<div class="hotspot-icon hotspot-${hotspot.type}-icon" style="background-color: #${hotspot.icon_color};"></div>`;
          div_hotspot_icon = `<img src="${APP.PATH_CF_FLIP}assets/images/hotspots/videoembed.svg"/>`;
          let div_hotspot_shadow = `<div class="hotspot-shadow hotspot-shadow_${
            pageModel.model.page
          } ${HOTSPOT_ANIMATION_OFF_CLASSNAME}" style="top: calc(50% - ${hotspotSize / 2}px); left: calc(50% - ${
            hotspotSize / 2
          }px); color: #${
            hotspot.icon_background_color
          };width: ${hotspotSize}px; height: ${hotspotSize}px; animation: null;"></div>`;

          hotspotCircle = `<div class="hotspot-circle hotspot-circle_${
            pageModel.model.page
          } ${HOTSPOT_ANIMATION_OFF_CLASSNAME}" ${!hotspot.icon_is_visible ? 'hidden' : ''} style="top: calc(50% - ${
            (hotspotSize * HOTSPOT_CIRCLE_SIZE_FACTOR) / 2
          }px); left: calc(50% - ${
            (hotspotSize * HOTSPOT_CIRCLE_SIZE_FACTOR) / 2
          }px); pointer-events: all; height: ${hotspotSize * HOTSPOT_CIRCLE_SIZE_FACTOR}px;width: ${
            hotspotSize * HOTSPOT_CIRCLE_SIZE_FACTOR
          }px;">${div_hotspot_icon}</div>`; // !is_mobile && isUploadedVideo(hotspot) && ${is_mobile() ? 'display: none;' : ''} 
        } 
        else 
        {
          let type = hotspot.type;

          if (type == 'link' && hotspot.data.type != 'url' && hotspot.data.type != 'page') {
            type = hotspot.data.type;
          } else {
            // SPRAWDZENIE CZY VALUE JEST STRINGIEM, INNACZEJ JEBALO WSZYSTKIE LINKI !!!
            // BO NIE ISTNIEJE INDEXOF FUNCKJI Z NUMBER TYPU
            if (typeof hotspot.data.value == 'string' && hotspot.data.value.indexOf('mailto:') !== -1) {
              type = 'email';
            }
          }

          hotspotCircle = `<div class="hotspot-circle hotspot-circle_${
            pageModel.model.page
          } ${HOTSPOT_ANIMATION_OFF_CLASSNAME}" ${!hotspot.icon_is_visible ? 'hidden' : ''}style="top: ${
            hotspot.icon_y
          }%; left: ${hotspot.icon_x}%; pointer-events: all;background-color: #${
            hotspot.icon_background_color
          }; width: ${hotspotSize}px; height: ${hotspotSize}px;"><div class="hotspot-circle__container"><div class="hotspot-icon hotspot-${
            type
          }-icon" style="background-color: #${
            hotspot.icon_color
          };"></div></div></div><div class="hotspot-shadow hotspot-shadow_${
            pageModel.model.page
          } ${HOTSPOT_ANIMATION_OFF_CLASSNAME}" style="top: ${hotspot.icon_y}%; left: ${hotspot.icon_x}%; color: #${
            hotspot.icon_background_color
          };width: ${hotspotSize}px; height: ${hotspotSize}px; ${
            !hotspot.icon_is_visible ? 'animation: null;' : ''
          }"></div>`;
        }
      }

      hotspotArea.insertAdjacentHTML('afterbegin', hotspotCircle);

      const highlightArea = document.createElement('div');
      highlightArea.className = `hotspot-area-highlight ${hotspot.area_is_visible ? '' : HOTSPOT_ANIMATION_OFF_CLASSNAME}`;
      hotspotArea.firstElementChild.insertAdjacentElement('afterend', highlightArea);

      if (hotspot.hasOwnProperty('area_highlight_color') && hotspot.area_highlight_color.length > 0) {
        highlightArea.style.backgroundColor = `#${hotspot.area_highlight_color}`;
      }

      if (!is_mobile() || isUploadedVideo(hotspot)) {
        const autoplay = hotspot.data.autoplay_player;

        if (hotspot.type == 'video' && hotspot.data.type == 'embedded' && autoplay && lastAutoplayElement && hotspot.area_number == lastAutoplayElement.area_number) {
          videoAutoplay(hotspot.data, hotspotArea, pageModel.model.page, hotspot);
          showHotspotArea(hotspotArea);
        }
      }

      if (!hotspot.extra || hotspot.extra.type !== 'slider') {
        hotspotArea.addEventListener('click', (e) => hotspotEventHandler(e, null, {...hotspot, page: pageModel.model.page}));
      }

      if (hotspot.type === 'audio' && (hotspot.data.autoplay_player || hotspot.data.autoplay) && lastAutoplayElement && hotspot.area_number == lastAutoplayElement.area_number) {
        const data = HELPER.getHotspotMediaUrl(hotspot.data.value);

        pageModel.Book.setDataToMediaViewer(data, 'audio', Object.assign({}, hotspot, {
          page: pageModel.model.page,
          _action: 'RENDER'
        }), hotspotArea);
      }

      hotspotsContainer.append(hotspotArea);
      pageNode.appendChild(hotspotsContainer);
      pageNode.hotspotCircle = document.querySelectorAll(`.hotspot-circle_${pageModel.model.page}`);
      pageNode.hotspotShadow = document.querySelectorAll(`.hotspot-shadow_${pageModel.model.page}`);

      // pageNode.hotspotCircle.forEach(circle => circle.addEventListener('mouseover', () => restartRotatingAnimation(circle)));
      pageNode.hotspotCircle.forEach(circle => circle.addEventListener('animationend', () => animationEndHandler(circle)));

      pageNode.hotspotArea = document.querySelectorAll(`.hotspot-area_${pageModel.model.page}`);
      pageNode.hotspotsContainer = hotspotsContainer;

      if (hotspot.type !== 'video' && (!hotspot.extra || hotspot.extra.type !== 'slider')) {
        showHotspotArea(hotspotArea);
      }
      
      if (hotspot.type == 'video' && hotspot.data && hotspot.data.type === 'url') {
        showHotspotArea(hotspotArea);
      }
    });
    
    hotspotsContainer.style.display = 'none';

    if (pageNode.dataset && pageNode.dataset.loading && pageNode.dataset.loading == 'end') {
      hotspotsContainerOn();
    }
  };

  const showHotspotArea = (hotspotArea, fadeInEffect = true) => {
    if (!hotspotArea) return;

    hotspotArea.style.display = 'block';

    if (fadeInEffect) {
      setTimeout(() => {
        hotspotArea.style.opacity = '1';
        hotspotArea.style.visibility = 'visible';
      }, 50);
    } else {
      hotspotArea.style.opacity = '1';
      hotspotArea.style.visibility = 'visible';
    }
  };

  const animationEndHandler = (circle) => {
    if (!circle) return;

    circle.classList.remove(HOTSPOT_ANIMATING_CLASSNAME);
  };

  const stopRotatingAnimation = (circle) => {
    if (!circle) return;
    if (circle.classList.contains(HOTSPOT_ANIMATION_OFF_CLASSNAME)) return;

    circle.classList.remove(HOTSPOT_ANIMATING_CLASSNAME);
    circle.classList.add(HOTSPOT_ANIMATION_OFF_CLASSNAME);
  };

  const restartRotatingAnimation = (circle) => {
    if (!circle) return;
    if (circle.classList.contains(HOTSPOT_ANIMATING_CLASSNAME)) return;

    circle.classList.add(HOTSPOT_ANIMATION_OFF_CLASSNAME);

    const animationTimeout = setTimeout(() => {
      circle.classList.add(HOTSPOT_ANIMATING_CLASSNAME);
      circle.classList.remove(HOTSPOT_ANIMATION_OFF_CLASSNAME);

      clearTimeout(animationTimeout);
    }, 100);
  };

  const setSliderSizes = (sliderWidth, sliderHeight) => {
    const page = this.getPage();
    const currentPage = document.querySelector(`._page_${APP.Book.getCurrentPage()}`);
    const hardCoverSettings = page.dataset.density === 'hard' && [1, +APP.Book.getModel().pages].indexOf(page.page) !== -1 && (APP.Book.getPageFlipHardCoverSettings());

    let currentPageSizes = {
      width: 0,
      heigth: 0
    }

    if (!currentPage || currentPage.style) {
      const bookSize = APP.Book.getBookSize();

      currentPageSizes = {
        width: Number(bookSize.bookWidth) + (hardCoverSettings ? hardCoverSettings.hardWrapper.WRAPPER_EXTRA_WIDTH / 2 : 0),
        height: Number(bookSize.bookHeight) + (hardCoverSettings ? hardCoverSettings.hardWrapper.WRAPPER_EXTRA_HEIGHT / 2 : 0)
      };
    } else {
      currentPageSizes = {
        width: Number(currentPage.style.width) + (hardCoverSettings ? hardCoverSettings.hardWrapper.WRAPPER_EXTRA_WIDTH / 2 : 0),
        height: Number(currentPage.style.height) + (hardCoverSettings ? hardCoverSettings.hardWrapper.WRAPPER_EXTRA_HEIGHT / 2 : 0)
      };
    }

    return (parseFloat(sliderWidth) * parseFloat(currentPageSizes.width)) / 100;
  };

  const updateSliderSizes = function () {
    const galleries = document.querySelectorAll('[data-type="gallery"]');
    const gifs = document.querySelectorAll('[data-type="gif"]');

    if (galleries && galleries.length) {
      galleries.forEach(gallery => {
        const slides = gallery.querySelectorAll('.hotspot-gallery__slide');
        const gallerySize = parseFloat(gallery.style.width);
  
        slides.forEach(slide => {
          const slideSize = setSliderSizes(gallerySize);
          slide.style.width = `${slideSize}px`;
        });
        setNewSlide(gallery);
      });
    }

    if (gifs && gifs.length) {
      gifs.forEach(gallery => {
        gallery.style.opacity = 0;
        const slides = gallery.querySelectorAll('.hotspot-gallery__slide');
        const gallerySize = parseFloat(gallery.style.width);
  
        slides.forEach(slide => {
          const slideSize = setSliderSizes(gallerySize);
          slide.style.width = `${slideSize}px`;
        });
        gallery.style.opacity = 1;
      });
    }
  }

  const setNewSlide = function (galleryWrapper) {
    const galleryContainer = galleryWrapper.querySelector('.hotspot-gallery__container');
    const allSlides = galleryWrapper.querySelectorAll('.hotspot-gallery__slide');
    const activeSlide = galleryWrapper.querySelector('.slide-active');
    let newSlide = false;

    if (allSlides.length === 1) {
      newSlide = allSlides[0];
    } else {
      newSlide = allSlides[1];
    }
    
    if (activeSlide && activeSlide.classList) {
      activeSlide.classList.remove('slide-active');
    }

    if (!newSlide) {
      return;
    }

    newSlide.classList.add('slide-active');

    galleryContainer.style.transition = 'none';
    galleryContainer.style.marginLeft = `${
      -(newSlide.dataset.id - 1) * parseFloat(activeSlide.style.width)
    }px`;
    galleryContainer.style.transition = 'margin 0.2s ease-in-out';
  };

  const getSlideImageUrl = (slideNode) => {
    const styles = window.getComputedStyle(slideNode);
    const backgroundImage = styles.getPropertyValue('background-image');

    return backgroundImage.slice(5, -2);
  };

  const getAreaSizeByScaleValue = (area, scaleValue) => {
    return {
      width: area.clientWidth * scaleValue,
      height: area.clientHeight * scaleValue
    };
  } ;

  this.loadGalleryImagesByScaleSize = function (scaleValue) {
    if (scaleValue < 1) return;
    if (!pageNode) return;

    const areas = pageNode.querySelectorAll('.hotspot-area[data-type="gallery"][data-extra-type="slider"]');

    if (!areas.length) return;

    const hotspots = pageModel.Book.getInteractivities(pageModel.model.page);

    if (!hotspots || !hotspots.hasOwnProperty('areas') || !Array.isArray(hotspots.areas) || hotspots.areas.length === 0) return;
    
    Array.from(areas).forEach(area => {
      const hotspot = hotspots.areas.find(item => item.area_number == area.id);

      if (!hotspot) return;
      
      const { width: areaWidth, height: areaHeight } = getAreaSizeByScaleValue(area, scaleValue);
      const slides = area.querySelectorAll('.hotspot-gallery__slide:not(.hotspot-gallery__clone)');

      Array.from(slides).forEach((item, index) => {
        let imageUrl = getSlideImageUrl(item);
        const actualBgString = item.style.backgroundImage.trim().replaceAll(`"`, `'`);

        if (hotspot.hasOwnProperty('data_sizes') && hotspot.data_sizes && hotspot.data_sizes.hasOwnProperty(index)) {
          const sizes = hotspot.data_sizes[index].filter(size => parseFloat(size) >= Math.max(areaWidth, areaHeight));
          const splitedUrl = hotspot.data[index].split('.');
  
          splitedUrl[splitedUrl.length - 2] += sizes.length ? `_${sizes[0]}` : `_thumb`;
  
          imageUrl = HELPER.getHotspotMediaUrl(splitedUrl.join('.'));
        } else if (hotspot.data_thumbs && hotspot.data_thumbs.length > index) {
          imageUrl = HELPER.getHotspotMediaUrl(hotspot.data_thumbs[index]);
        } else {
          imageUrl = HELPER.getHotspotMediaUrl(hotspot.data[index]);
        }

        const bgString = setAndOrderBackground(actualBgString, imageUrl);

        if (actualBgString != bgString) {
          item.style.backgroundImage = bgString;
        }
      });
    });
  };

  this.loadGalleryImagesByBookSize = function () {
    if (!pageNode) return;

    const areas = pageNode.querySelectorAll('.hotspot-area[data-type="gallery"][data-extra-type="slider"]');

    if (!areas.length) return;

    const hotspots = pageModel.Book.getInteractivities(pageModel.model.page);

    if (!hotspots || !hotspots.hasOwnProperty('areas') || !Array.isArray(hotspots.areas) || hotspots.areas.length === 0) return;
    
    Array.from(areas).forEach(area => {
      const hotspot = hotspots.areas.find(item => item.area_number == area.id);

      if (!hotspot) return;
      
      const slideWidthInPx = setSliderSizes(area.style.width, area.style.height);
      const slides = area.querySelectorAll('.hotspot-gallery__slide:not(.hotspot-gallery__clone)');

      Array.from(slides).forEach((item, index) => {
        let imageUrl = getSlideImageUrl(item);
        const actualBgString = item.style.backgroundImage.trim().replaceAll(`"`, `'`);

        if (hotspot.hasOwnProperty('data_sizes') && hotspot.data_sizes && hotspot.data_sizes.hasOwnProperty(index)) {
          const sizes = hotspot.data_sizes[index].filter(size => parseFloat(size) >= slideWidthInPx * 2);
          const splitedUrl = hotspot.data[index].split('.');
  
          splitedUrl[splitedUrl.length - 2] += sizes.length ? `_${sizes[0]}` : `_thumb`;
  
          imageUrl = HELPER.getHotspotMediaUrl(splitedUrl.join('.'));
        } else if (hotspot.data_thumbs && hotspot.data_thumbs.length > index) {
          imageUrl = HELPER.getHotspotMediaUrl(hotspot.data_thumbs[index]);
        } else {
          imageUrl = HELPER.getHotspotMediaUrl(hotspot.data[index]);
        }

        const bgString = setAndOrderBackground(actualBgString, imageUrl);

        if (actualBgString != bgString) {
          item.style.backgroundImage = bgString;
        }
      });
    });
  };

  const renderGallerySlider = function (hotspot, area, page) {
    let galleryNavClickedTime = null;
    let isMouseEntered = false;
    let firstImage = null;
    let autoplayTime = 500;
    let isGif = hotspot.type == 'gif';
    let imagesSizes = [];
    const galleryWrapper = document.createElement('div');
    const galleryContainer = document.createElement('ul');
    const slideWidthInPx = setSliderSizes(area.style.width, area.style.height);

    const handleMouseEnter = function () {
      isMouseEntered = true;
      this.setAttribute('hovered', true);
    };

    const handleMouseLeave = function () {
      isMouseEntered = false;
      this.removeAttribute('hovered');
    };

    galleryWrapper.style.opacity = 0;
    galleryWrapper.style.visibility = 'hidden';

    const loader = document.createElement('div');
    loader.className = 'gif-loader';
    loader.innerHTML = HELPER.buttonsIcons.loaderIcon;
    area.appendChild(loader);

    galleryWrapper.className = `hotspot-gallery__wrapper ${hotspot.data.length > 1 && hotspot.extra.autoplay ? 'autoplay' : ''}`;
    galleryContainer.className = 'hotspot-gallery__container';

    galleryWrapper.addEventListener('mouseenter', handleMouseEnter);
    galleryWrapper.addEventListener('mouseleave', handleMouseLeave);

    // Usuwanie pustych danych w tablice
    hotspot.data = hotspot.data.filter((item) => item !== '' && item !== null);

    if (!hotspot.data.length) return;

    galleryContainer.style = `width: ${
      (hotspot.data.length > 1 ? hotspot.data.length + 2 : hotspot.data.length) * 100
    }%`;

    hotspot.data.forEach((item, index) => {
      const hotspotId = hotspot.data.indexOf(item) + 1;
      const gallerySlide = document.createElement('li');

      gallerySlide.className = `hotspot-gallery__slide hotspot-gallery__slide-${hotspotId}`;
      gallerySlide.style = `width: ${slideWidthInPx}px`;

      let imageUrl = HELPER.getHotspotMediaUrl(hotspot.data[index]);
      // let imageUrl = (hotspot.data_thumbs && hotspot.data_thumbs.length) ? hotspot.data_thumbs[index] : item;

      if (hotspot.hasOwnProperty('data_sizes') && hotspot.data_sizes && hotspot.data_sizes.hasOwnProperty(index)) {
        const sizes = hotspot.data_sizes[index].filter(size => parseFloat(size) >= slideWidthInPx * 2);
        const splitedUrl = hotspot.data[index].split('.');

        splitedUrl[splitedUrl.length - 2] += sizes.length ? `_${sizes[0]}` : `_thumb`;

        imageUrl = HELPER.getHotspotMediaUrl(splitedUrl.join('.'));
      } else if (hotspot.data_thumbs && hotspot.data_thumbs.length > index) {
        imageUrl = HELPER.getHotspotMediaUrl(hotspot.data_thumbs[index]);
      } else {
        imageUrl = HELPER.getHotspotMediaUrl(hotspot.data[index]);
      }

      const img = new Image();
      img.addEventListener('load', () => {
        const ratio = img.width / img.height;

        if (!isGif) {
          imagesSizes.push({
            id: hotspotId,
            ratio: ratio
          });
  
          setBackgoundColorToSlides();
        }
      })
      img.src = HELPER.getHotspotMediaUrl(isGif ? `${imageUrl}?rand=${Math.random()}` : imageUrl);

      if (isGif) {
        gallerySlide.style.backgroundColor = 'transparent';
        gallerySlide.style.backgroundImage = `url('${img.src}')`;
      } else {
        gallerySlide.style.backgroundImage = `url('${img.src}')`;
      }

      if (index == 0) {
        firstImage = img;
      }
 
      galleryContainer.appendChild(gallerySlide);
    });

    if (hotspot.data.length == 1) {
      area.classList.add('hotspot-area__single-slide');
    }

    if (hotspot.data.length > 1) {
      const firstSlideClone = galleryContainer
        .querySelector(`.hotspot-gallery__slide-${hotspot.data.length}`)
        .cloneNode(true);

      const lastSlideClone = galleryContainer
        .querySelector('.hotspot-gallery__slide-1')
        .cloneNode(true);

      if (firstSlideClone && lastSlideClone) {
        firstSlideClone.className =
          'hotspot-gallery__slide hotspot-gallery__clone hotspot-gallery__clone-f';
        lastSlideClone.className =
          'hotspot-gallery__slide hotspot-gallery__clone hotspot-gallery__clone-l';
      }

      galleryContainer.prepend(firstSlideClone);
      galleryContainer.appendChild(lastSlideClone);

      const slides = galleryContainer.querySelectorAll('.hotspot-gallery__slide');
      if (slides && slides.length) {
        slides.forEach((item, key) => item.dataset.id = key + 1);
      }

      const hotspotSize = getHotspotSize(10);
      let showFullscreenButton = HELPER.getNestedValueByPath(hotspot, 'extra.fullscreen_button');
      showFullscreenButton = showFullscreenButton === undefined ? true : showFullscreenButton;

      if (is_mobile() && !is_tablet() && showFullscreenButton) {
        const fullScreenToggle = document.createElement('div');

        fullScreenToggle.className = 'hotspot-gallery__fullscreen';
        fullScreenToggle.innerHTML = HELPER.buttonsIcons.galleryFullscreenIcon;
        fullScreenToggle.style = `width: ${hotspotSize}px; height: ${hotspotSize}px`;

        galleryWrapper.addEventListener('click', (event) => hotspotEventHandler(event, area, Object.assign({}, hotspot, { page })));
        galleryWrapper.appendChild(fullScreenToggle);
      }

      if ((!is_mobile() || is_tablet()) && !hotspot.extra.autoplay) {
        const sliderInfoIconWrapper = document.createElement('div');
        const sliderInfoIcon = document.createElement('img');
        sliderInfoIconWrapper.className = 'hotspot-gallery__slider-icon';
        sliderInfoIcon.src = `${APP.PATH_CF_FLIP}assets/images/hotspots/gallerysliderinfo.svg`;
        sliderInfoIconWrapper.appendChild(sliderInfoIcon);

        sliderInfoIconWrapper.style = `width: ${hotspotSize}px; height: ${hotspotSize}px`;
        galleryWrapper.appendChild(sliderInfoIconWrapper);
      }
    }

    const setBackgoundColorToSlides = function () {
      if (!imagesSizes || !imagesSizes.length || !imagesSizes.find(item => item.id == 1)) return;

      const firstImage = imagesSizes.find(item => item.id == 1);
      imagesSizes.forEach(item => {
        const slide = galleryWrapper.querySelector(`.hotspot-gallery__slide-${item.id}`);
        if (!slide) return;

        if ((item.ratio <= firstImage.ratio - 0.1) || (item.ratio >= firstImage.ratio + 0.1)) {
          slide.style.backgroundColor = '#000';
        } else {
          slide.style.backgroundColor = 'transparent';
        }
      })
    }

    const onFirstImageLoad = function () {
      area.style.display = 'block';
      setTimeout(() => {
        area.style.opacity = '1';
        area.style.visibility = 'visible';

        galleryWrapper.style.opacity = '1';
        galleryWrapper.style.visibility = 'visible';

        const loader = area.querySelector('.gif-loader');

        if (loader) {
          area.removeChild(loader);
        }
      }, 50);

      if (hotspot.data.length > 1 && hotspot.extra.autoplay) {
        setAutoplay();
      }

      const gallerySliderIcon = document.querySelector('.hotspot-gallery__slider-icon');

      if (gallerySliderIcon) {
        gallerySliderIcon.classList.add('hotspot-gallery__slider-animation');
      }

      this.removeEventListener('load', onFirstImageLoad);
    };

    if (firstImage) {
      firstImage.addEventListener('load', onFirstImageLoad);
    }

    if (hotspot.data.length > 1 && (!is_mobile() || is_tablet())) {
      renderGallerySliderArrows(galleryWrapper);
    }

    const firstSlide = galleryContainer.querySelector(
      '.hotspot-gallery__slide:not(.hotspot-gallery__clone)',
    );
    firstSlide.classList.add('slide-active');
    galleryContainer.style.marginLeft = `${
      -(firstSlide.dataset.id - 1) * parseFloat(firstSlide.style.width)
    }px`;

    galleryWrapper.appendChild(galleryContainer);
    area.appendChild(galleryWrapper);

    function singleOrDblClickEvents(event, onClick, onDblClick) {
      let timer;

      if (event.target.dataset.dblclick == null) {
        event.target.dataset.dblclick = 1;
        timer = setTimeout(() => {
          if (event.target.dataset.dblclick == 1) {
            onClick();
          }
          delete event.target.dataset.dblclick;
        }, 0);
      } else {
        clearTimeout(timer);
        delete event.target.dataset.dblclick;
        onDblClick();
      }
    }

    function renderGallerySliderArrows(galleryWrapper) {
      const galleryNavWrapper = document.createElement('div');
      galleryNavWrapper.className = 'hotspot-gallery__arrow-wrapper';

      const galleryNavArrowLeftArea = document.createElement('button');
      const galleryNavArrowRightArea = document.createElement('button');

      galleryNavArrowLeftArea.className = 'hotspot-gallery__arrow hotspot-gallery__arrow--left';
      galleryNavArrowRightArea.className = 'hotspot-gallery__arrow hotspot-gallery__arrow--right';

      galleryNavArrowLeftArea.appendChild(document.createElement('div'));
      galleryNavArrowRightArea.appendChild(document.createElement('div'));

      galleryNavArrowLeftArea.addEventListener('click', (event) =>
        singleOrDblClickEvents(
          event,
          () => onArrowClick(galleryWrapper, 'prev'),
          () => hotspotEventHandler(event, area),
        ),
      );

      galleryNavArrowRightArea.addEventListener('click', (event) =>
        singleOrDblClickEvents(
          event,
          () => onArrowClick(galleryWrapper, 'next'),
          () => hotspotEventHandler(event, area),
        ),
      );

      galleryNavWrapper.appendChild(galleryNavArrowLeftArea);
      galleryNavWrapper.appendChild(galleryNavArrowRightArea);

      galleryWrapper.appendChild(galleryNavWrapper);
    }

    function onArrowClick(galleryWrapper, way) {
      const galleryContainer = galleryWrapper.querySelector('.hotspot-gallery__container');
      const allSlides = galleryWrapper.querySelectorAll('.hotspot-gallery__slide');
      const activeSlide = galleryWrapper.querySelector('.slide-active');
      let newSlide;
      galleryNavClickedTime = Date.now();
      galleryContainer.style.transition = 'margin 0.2s ease-in-out';

      if (way === 'prev') {
        newSlide = activeSlide.previousElementSibling
          ? activeSlide.previousElementSibling
          : allSlides[allSlides.length - 1];
      } else {
        newSlide = activeSlide.nextElementSibling ? activeSlide.nextElementSibling : allSlides[0];
      }
      activeSlide.classList.remove('slide-active');
      newSlide.classList.add('slide-active');

      galleryContainer.style.marginLeft = `${
        -(newSlide.dataset.id - 1) * parseFloat(activeSlide.style.width)
      }px`;

      if (newSlide.classList.contains('hotspot-gallery__clone')) {
        const arrows = galleryWrapper.querySelectorAll('.hotspot-gallery__arrow');
        if (arrows && arrows.length) {
          arrows.forEach((btn) => btn.setAttribute('disabled', true));
        }
      }

      const area = galleryWrapper.parentNode;
      
      const slideWidthInPx = setSliderSizes(area.style.width, area.style.height);
      Array.from(allSlides).forEach(slide => slide.style.width = `${slideWidthInPx}px`);

      galleryContainer.addEventListener('transitionend', (e) => {
        const arrows = galleryWrapper.querySelectorAll('.hotspot-gallery__arrow');
        if (arrows && arrows.length) {
          arrows.forEach((btn) => btn.removeAttribute('disabled'));
        }

        if (newSlide.classList.contains('hotspot-gallery__clone')) {
          galleryContainer.style.transition = 'none';
          if (newSlide.classList.contains('hotspot-gallery__clone-f')) {
            newSlide.classList.remove('slide-active');
            newSlide = galleryContainer.querySelector(
              `.hotspot-gallery__slide-${allSlides.length - 2}`,
            );
          }
          if (newSlide.classList.contains('hotspot-gallery__clone-l')) {
            newSlide.classList.remove('slide-active');
            newSlide = galleryContainer.querySelector('.hotspot-gallery__slide-1');
          }
          newSlide.classList.add('slide-active');
          galleryContainer.style.marginLeft = `${
            -(newSlide.dataset.id - 1) * parseFloat(activeSlide.style.width)
          }px`;
        }
      });
    }

    function setAutoplay() {
      if (model.Book.autoplayInInitialized) return;

      model.Book.autoplayInInitialized = true;

      model.Book.autoplayIn = setInterval(() => {
        if (APP.Book.getLoaderStatus()) {
          if (!galleryNavClickedTime) {
            galleryNavClickedTime = Date.now();
          }

          if (Date.now() - galleryNavClickedTime >= autoplayTime) {
            galleryNavClickedTime = Date.now();
            const galleries = Array.from(document.querySelectorAll('.hotspot-gallery__wrapper.autoplay'));

            galleries.forEach(gallery => {
              if (!gallery.hasAttribute('hovered') || is_mobile()) {
                onArrowClick(gallery, 'next');
              }
            });
  
            if (autoplayTime != 2000) {
              autoplayTime = 2000;
            }
          }
        }
      }, 200);
    }
  };

  function resetAutoplay() {
    model.Book.autoplayInInitialized = false;

    if (model.Book.autoplayIn) {
      clearInterval(model.Book.autoplayIn);
    }
  }

  this.removeHotspots = function () {
    if (!pageNode.hotspotArea) return;

    pageNode.dataset.animation = 'off';
    pageNode.hotspotCircle = '';
    pageNode.hotspotShadow = '';
    pageNode.hotspotArea.forEach((area) => area.remove());
    pageNode.hotspotArea = '';
    resetAutoplay();
  };
};
