'use strict';
var Book = function (model) {
  var self = this;

  var settings = {
    padding: is_mobile() ? 20 : 10, // book padding
    paddingX: 0, // book padding for hard cover
    menuHProc: 8.5,
    menuMaxH: 45,
    menuMinH: 35,
    forceOnePage: forceOnePage,
    menuHeight: 90, // Footer height on MD size
    PreloadImages: PreloadImages, // Preload images by WebWorker if possible
    detectOrientationChange: true, // Rerender current page and layout on orientationchange
    nextPrevButtons: true, // Render next, prev buttons
    pagesRanger: true, // Render pages ranger bar
    Scaller: true, // Render Scale control
    sidebar: true, // Render sidebar
    MIN_SCALE: 0, // Minimal scale value
    MAX_SCALE: 20, // Maximal scale value
    MOBILE_MIN_SCALE: 1, // Minimal scale value on mobile
    MOBILE_MAX_SCALE: 10, // Maximal scale value on mobile
    DEFAULT_SCALE: 0, // Default scale value
    PVTtime: 100, // Time before rerender in vertical mode
    useVector: true, // Use vector layer
    fakeRequest: false, // This option block Request. DO NOT USE
    showSinglePageMode: false,
    thumbsNumPreload: 50,
    webpSupport: false,
    shadowsInZoom: false,
    hideArrowsInZoom: false,
    loadSvgOnZoom: true,
    showPageLikeIMG: false,
    minLoaderTime: 1700,
  };
  const LANDSCAPE = 90;
  const PORTRAIT = 0;

  const FORBIDDEN = 403;
  const NOTEXIST = 404;
  const PROTECTED = 2016;
  const INVALID_PASSWORD = 2017;
  const LOCATION_HREF = window.location.href;
  const BUTTON_ICON_TYPES = {
    INLINE: 'inline',
    IMG: 'img'
  };
  const BUTTON_ICON_TYPE = BUTTON_ICON_TYPES.INLINE;
  const DEFAULT_INNER_SIDE_OFFSET = 40;
  const MODAL_FADE_TIME = 200;

  const COLOR_THEME = {
    DARK: 'dark',
    LIGHT: 'light'
  };

  var startPage = 1;
  let pageWidth, pageHeight;
  let prevBookSize = null;
  var cornerTimeout;
  let isBookLoaded = false;
  var isVideoFullscreen = false;
  let mainInitiator = null;
  let fsToggleShowedTimes = 0;
  let fsTimeoutTime = 500;
  let fsAnimationStarted = false;
  let fsAnimationTimeout = undefined;
  let isMouseOverBook = false;
  let isFirstUserInteractionHandled = false;

  var timerStart = 0;
  this.password = null;

  let isThumbsLoaded = false;
  let isFirstRender = true;
  let URLChanged = false;
  let isOrientationChangeResize = false;

  let isBookInnerTransitionEnd = true;

  this.autoplayInInitialized = false;
  this.autoplayIn = null;

  const _global = {
    val: 0,
    listener: function(_page) {},
    set page(_page) {
      this.val = _page;
      this.listener(_page);
    },
    get page() {
      return this.val;
    },
    initializeListener: function(_listener) {
      this.listener = _listener;

      window.onerror = function(error, url, line) {
        oLogs.log_send_error(model.ref_id || model.id, model.publisher_id, {
          acc: 'error', 
          data: `ERR: ${error} URL: ${url} L: ${line}`
        });
      };
    }
  };

  _global.initializeListener(function(_page) {
    oLogs.log_set_page();
  });

  let bookSlideshowInterval = null;
  let bookAutoFlipInitialized = false;
  let bookAutoFlipEnded = false;
  let bookAutoFlipOptions = {};

  const bookAutoFlip = (data) => {
    const model = self.getModel();

    if (!data.state || !data.duration || data.duration <= 0) {
      return;
    }

    return new Promise(resolve => {
      bookSlideshowInterval = setInterval(() => {
        const currentPage = self.getCurrentPage();
        const bookPagesCount = parseInt(model.pages, 10);
  
        if (this.isDoublePage() && parseInt(currentPage + 1) >= bookPagesCount) {
          clearInterval(bookSlideshowInterval);
          return resolve();
        }
  
        if (currentPage >= bookPagesCount) {
          clearInterval(bookSlideshowInterval);
          return resolve();
        }
  
        self.nextPage();
      }, parseInt(data.duration, 10) * 1000);
    })
  };

  const stopBookAutoFlip = () => {
    if (!bookAutoFlipInitialized) {
      return;
    }

    changeAutoFlipButtonState();

    clearInterval(bookSlideshowInterval);
  };
  
  const endBookAutoFlip = () => {
    stopBookAutoFlip();
    unbindAutoFlipEvents();

    bookAutoFlipOptions = {};
    bookAutoFlipEnded = true;
  };

  const startBookAutoFlip = async ({ loops, state, duration, loop, times }) => {
    changeAutoFlipButtonState(true);

    for await (const index of loops) {
      await createAutoPlayLoop(state, duration, loop, times);

      if (index === loops.length - 1) {
        endBookAutoFlip();

        return;
      }

      self.setPage(0, 'customize');
      updateBookAutoFlipOptions({ times: Number(bookAutoFlipOptions.times) - 1 });
    }
  };

  const createAutoPlayLoop = async (state, duration, loop, times) => await bookAutoFlip({ state, duration, loop, times });
  const getElementsToBindAutoFlipEvents = () => {
    return [{
      event: 'click',
      elements: [
        elements.innerChild, 
        elements.navNextPage, 
        elements.navPrevPage, 
        elements.navNextPageAbsolute, 
        elements.navPrevPageAbsolute,
        elements.navAreaNextPage,
        elements.navAreaPrevPage,
        elements.pagesRangerContainer, 
        elements.pagesRangerLeftNum,
        elements.pagesRangerRightNum
      ],
      handler: stopBookAutoFlip
    }, {
      event: 'change',
      elements: [elements.pageNumInput],
      handler: stopBookAutoFlip
    }];
  };

  const unbindAutoFlipEvents = () => {
    if (!bookAutoFlipInitialized) {
      return;
    }

    bookAutoFlipInitialized = false;

    getElementsToBindAutoFlipEvents().forEach(item => item.elements.forEach(elem => elem.removeEventListener(item.event, item.handler)));
  };
  
  const bindAutoFlipEvents = () => {
    if (bookAutoFlipInitialized) {
      return;
    }

    bookAutoFlipInitialized = true;

    getElementsToBindAutoFlipEvents().forEach(item => item.elements.forEach(elem => elem.addEventListener(item.event, item.handler)));
  };

  const getAutoFlipOptions = (data = null, forceData = false) => {
    const model = self.getModel();

    if (!!!Object.keys(bookAutoFlipOptions).length || forceData) {
      bookAutoFlipOptions = {
        state: !data || data.state === null ? model.settings.auto_flip_enabled : data.state,
        duration: !data || data.duration === null ? model.settings.auto_flip_duration : data.duration,
        loop: !data || data.loop === null ? model.settings.auto_flip_loop_continously : data.loop,
        times: !data || data.times === null ? model.settings.auto_flip_loop_times : data.times,
      };

      model.settings.auto_flip_enabled = bookAutoFlipOptions.state;
      model.settings.auto_flip_duration = bookAutoFlipOptions.duration;
      model.settings.auto_flip_loop_continously = bookAutoFlipOptions.loop;
      model.settings.auto_flip_loop_times = bookAutoFlipOptions.times || 1;
    }

    const loops = Array.from({length: bookAutoFlipOptions.loop ? 999 : bookAutoFlipOptions.times}, (_, i) => i);

    return {
      state: bookAutoFlipOptions.state,
      duration: bookAutoFlipOptions.duration,
      loop: bookAutoFlipOptions.loop,
      times: bookAutoFlipOptions.times, 
      loops
    };
  };

  const updateBookAutoFlipOptions = (options) => Object.assign(bookAutoFlipOptions, options);

  const isLastPage = () => {
    const model = self.getModel();
    const currentPages = self.getCurrentPageForLogs();
    const bookPages = Number(model.pages);

    return currentPages.indexOf(bookPages - 1) !== -1;
  };

  this.initializeBookAutoFlip = async (data) => {
    const options = getAutoFlipOptions(data, true);

    if (!bookAutoFlipInitialized) {
      bindAutoFlipEvents();
    }

    if (bookAutoFlipEnded) {
      if (isLastPage()) {
        self.setPage(0, 'customize');
      }

      bookAutoFlipEnded = false;
    }

    clearInterval(bookSlideshowInterval);

    if (!!options.state) {
      showAutoFlipButton();
      startBookAutoFlip({
        loops: options.loops, 
        state: options.state, 
        duration: options.duration, 
        loop: options.loop, 
        times: options.times
      });
    } else {
      hideAutoFlipButton();
      stopBookAutoFlip();
    }
  };

  const hideAutoFlipButton = () => {
    elements.autoFlip.hidden = true;
  };

  const showAutoFlipButton = () => {
    elements.autoFlip.removeAttribute('hidden');
  };

  const changeAutoFlipButtonState = (isPlaying = false) => {
    if (isPlaying) {
      elements.autoFlip.classList.add('playing');

      if (BUTTON_ICON_TYPE === BUTTON_ICON_TYPES.INLINE) {
        elements.autoFlip.innerHTML = HELPER.buttonsIcons.autoFlipPauseIcon;
      }
    } else {
      elements.autoFlip.classList.remove('playing');

      if (BUTTON_ICON_TYPE === BUTTON_ICON_TYPES.INLINE) {
        elements.autoFlip.innerHTML = HELPER.buttonsIcons.autoFlipPlayIcon;
      }
    }
  };

  const autoFlipHandler = () => {
    if (!bookAutoFlipInitialized) {
      const model = self.getModel();

      self.initializeBookAutoFlip({ 
        state: model.settings.auto_flip_enabled,
        duration: model.settings.auto_flip_duration,
        loop: model.settings.auto_flip_loop_continously,
        times: model.settings.auto_flip_loop_times
      });

      return;
    }

    if (elements.autoFlip.classList.contains('playing')) {
      stopBookAutoFlip();
    } else {
      const options = getAutoFlipOptions();

      startBookAutoFlip(options);
    }
  };

  this.initializeBookHardCover = (data) => {
    if (!pageFlip) return;

    bookModel.settings.hard_cover_enabled = !!data.state;
    bookModel.settings.hard_cover_color = data.color;

    if (data.state) {
      settings.paddingX = (is_mobile() ? 20 : 36) + (is_mobile() ? 0 : 20);
      pageFlip.initializeBookHardCover(data);
    } else {
      settings.paddingX = 0;
      pageFlip.destroyBookHardCover();
    }

    setBookInnerSize();
  };

  this.initializeBookHardCoverThickness = (data) => {
    if (!pageFlip) return;

    if (data) {
      pageFlip.initializeBookHardCoverThickness();
    } else {
      pageFlip.destroyBookHardCoverThickness();
    }
  }

  this.getPageFlipHardCoverSettings = () => {
    if (!bookModel.settings.hard_cover_enabled) return;

    const defaultSettings = {
      hardWrapper: {
        WRAPPER_EXTRA_WIDTH: 36
      }
    };
    
    if (!pageFlip) return defaultSettings;

    return pageFlip.getHardCoverSettings();
  };

  this.updateBookFlipType = (flipType) => {
    bookModel.settings.flip_type = flipType;
    pageFlip.updateBookFlipType(flipType);
  };

  const updateConvertingStatusForHotspots = () => {
    const statusInterval = setInterval(() => {
      fetch(
        `${APP.VIEWER_API}/${model.publisherId}/${model.bookId}?v=${
          APP.v
        }&cmsauth=${cmsAuth}&uid=${HELPER.uid()}`, {
          headers: {
             'Accept': 'application/json',
             'Content-Type': 'application/json'
          }
      })
        .then((response) => response.json())
        .then((data) => {
          if (data.is_converted) {
            if (customizemode) {
              window.parent.postMessage('enableHotspots', '*');

              bookModel = data;
              APP.bookModel = bookModel;

              if (isCanva) {
                window.location.reload();
              } else{
                clearInterval(statusInterval);
                return;
              }
            }
          }
        })
        .catch((error) => {
          console_log(error);
        });
    }, 3000);
  }

  const loadBookClassic = async () => {
    const response = await fetch(
      `${APP.VIEWER_API}/${model.publisherId}/${model.bookId}?v=${
        APP.v
      }&cmsauth=${cmsAuth}&uid=${HELPER.uid()}`, {
        // headers: {
        //   'Accept': 'application/json',
        //   'Content-Type': 'application/json'
        // }
      });

    const data = await response.json();
    bookModel = data;
    
    if (data.code === FORBIDDEN && data.error_code === PROTECTED) {
      self.renderProtected( data );
    } else if (data.code === NOTEXIST) {
      window.location = APP.PATH_FORLANDING + '/#nle2';
    } else {
      bookModel = data;

      if (!bookModel.is_converted) {
        updateConvertingStatusForHotspots();
      }

      self.customizeBook();
      self.render();
    }
  };

  const assignCustomizeSettings = () => {
    let customizeModelSettings = {};
    const settingsOptionsName = [
      'show_', 
      'background_', 
      'lead_', 
      'toplogo_', 
      'password_', 
      'bottom_menu_color_mode', 
      'enable_protected_domain', 
      'protected_domains', 
      'hard_cover_', 
      'auto_flip_'
    ];
    
    Object.entries(APP.customizeSettings).forEach(item => {
      if (settingsOptionsName.some(optName => item[0].startsWith(optName))) {
        customizeModelSettings[item[0]] = item[1];
        delete APP.customizeSettings[item[0]];
      }
    });

    bookModel = {
      ...bookModel,
      ...APP.customizeSettings
    };

    bookModel.settings = {
      ...bookModel.settings,
      ...customizeModelSettings
    };
    console_log(bookModel);
  }

  this.tryLoadBookEmbeded = (iteration = 0) => {
    var _iteration = iteration + 1;
    if (embeduid === '') {
      if (iteration >= 4) {
        return this.loadBook();
      }
      sleep(250).then(() => {
        return this.tryLoadBookEmbeded(_iteration);
      });
    } else {
      return this.loadBook();
    }
  };
//LOCATION_HREF.includes(APP.ROOT_PATH) &&
  if ( LOCATION_HREF.length > APP.ROOT_PATH.length) {
    let hrefNoGet = LOCATION_HREF.split('?')[0];
    let hrefArr = hrefNoGet.split('/');

    if (hrefArr.indexOf('page') !== -1 && Number(hrefArr[hrefArr.length - 1])) {
      let pageIndex = hrefArr.indexOf('page');
      if (Number(hrefArr[pageIndex + 1])) {
        startPage = Number(hrefArr[pageIndex + 1]);
      }
    }
  }

  var bookModel;
  settings.webpSupport = document.createElement('canvas').toDataURL('image/webp').indexOf('data:image/webp') == 0;

  const loadProtectedBook = async (password) => {
    const passwordString = new URLSearchParams({
      'password': password
    });

    password = passwordString.toString().replace('password=', '');
    
    const response = await fetch(
      `${APP.VIEWER_API}/${model.publisherId}/${model.bookId}?v=${
        APP.v
      }&password=${password}&uid=${HELPER.uid()}`, {
        // headers: {
        // 'Accept': 'application/json',
        //  'Content-Type': 'application/json'
        // }
      }
    );

    const data = await response.json();

    if (data.code === FORBIDDEN && data.error_code === INVALID_PASSWORD) {
      const protectedModal = document.querySelector('.protected-modal');

      protectedModal.style.display = 'flex';
      protectedModal.querySelector('.protected-modal__header').innerText = APP._t('Password Form Title');
      protectedModal.querySelector('.protected-modal__error-message').innerText = APP._t('Password Form Error Invalid');
      protectedModal.querySelector('.protected-modal__input').focus();
    } else {
      this.password = password;
      bookModel = data;
      APP.bookModel = bookModel;
      pagesNum = bookModel.pages;

      self.customizeBook();
      self.render();
      
      if (is_mobile() && HELPER.isLocalStorageAvailable()) {
        setBookPassToLocalStorage(model.bookId, password);
      }
    }
  };

  function setBookPassToLocalStorage (bookId, password) {
    let localPass = localStorage.getItem('_book-pass');

    if (localPass && JSON.parse(localPass) && JSON.parse(localPass).length) {
      localPass = JSON.parse(localPass);
      let currentBook  = localPass.map(item => item.bookId == bookId);
      const date = new Date();

      if (currentBook) {
        currentBook = {
          ...currentBook,
          expiredAt: date.getTime() + 100 * 60 * 60
        }
      } else {
        currentBook = {
          bookId: bookId,
          pass: password,
          expiredAt: date.getTime() + 100 * 60 * 60
        }
      }

      localPass.forEach(item => {
        if (item.bookId == item.bookId)  {
          item = currentBook
        }
      })
    } else {
      let data = []
      const date = new Date();

      data.push({
        bookId: bookId,
        pass: password,
        expiredAt: date.getTime() + 100 * 60 * 60
      })

      localStorage.setItem('_book-pass', JSON.stringify(data));
    }
  }

  function componentToHex(c) {
    var hex = c.toString(16);
    return hex.length == 1 ? '0' + hex : hex;
  }

  function rgbToHex(r, g, b) {
    return '' + componentToHex(r) + componentToHex(g) + componentToHex(b);
  }

  this.customizeBookBackgroundColor__getCorrectAvgHex = () => {
    if (bookModel.settings.background_mode == 0) {
      return bookModel.settings.background_color;
    } else if (bookModel.settings.background_mode == 1) {
      return bookModel.settings.background_texture_color;
    } else if (bookModel.settings.background_mode == 2) {
      if (bookModel.settings.background_image && bookModel.settings.background_image !== '') {
        return bookModel.settings.background_image_color;
      }

      if (bookModel.settings.prev_background_mode === 0) {
        return bookModel.settings.background_color;
      } else {
        return bookModel.settings.background_texture_color;
      }
    }
  };

  this.customizeBookBackgroundTheme = () => {
    var rgb = APP.hexToRgb(`#${this.customizeBookBackgroundColor__getCorrectAvgHex()}`);
    var theme = COLOR_THEME.LIGHT;
    if (rgb !== null) {
      theme = rgb['r'] * 0.299 + rgb['g'] * 0.587 + rgb['b'] * 0.114 > 186 ? COLOR_THEME.LIGHT : COLOR_THEME.DARK;
    }
    bookModel.settings.bottom_menu_color_mode = theme;

    document.body.classList.remove(COLOR_THEME.DARK);
    document.body.classList.remove(COLOR_THEME.LIGHT);
    document.body.classList.add(theme);

    if (elements.pagesRangerContainer) {
      elements.pagesRangerContainer.classList.remove(COLOR_THEME.DARK);
      elements.pagesRangerContainer.classList.remove(COLOR_THEME.LIGHT);
      elements.pagesRangerContainer.classList.add(theme);
    }
  };
  this.updateFlipBook = () => {
    pageFlip.update();
  };

  this.updateBookModelSettings = function (key, value) {
    bookModel.settings[key] = value;
  };
  
  this.updateTOC = function (toc) {
    bookModel.toc = toc;

    const sidebarTOC = document.getElementById('sidebarTOC');
    
    if (!sidebarTOC) {
      return;
    }

    // if (sidebarTOC.querySelector('.sidebar__wrapper')) {
    //   sidebarTOC.querySelector('.sidebar__wrapper').remove();
    // }
    
    APP.Layout.updateTOC(bookModel.toc);
    APP.Layout.getSidebarTOC().show();
  };

  const customizeSidebarsAndLoader = (color) => {
    const bookLoader = document.querySelector('.loader');

    if (bookLoader) {
      const rgb = APP.hexToRgb(`#${color}`);
      let theme = COLOR_THEME.LIGHT;
      
      if (rgb !== null) {
        theme = rgb['r'] * 0.299 + rgb['g'] * 0.587 + rgb['b'] * 0.114 > 186 ? COLOR_THEME.LIGHT : COLOR_THEME.DARK;
      }

      bookLoader.style.backgroundColor = color ? `#${color}` : '';
      bookLoader.style.color = theme == COLOR_THEME.LIGHT ? 'var(--light-color)' : 'var(--dark-color)';
    }

    const sidebarElem = document.querySelector('#sidebar');
    const sidebarTocElem = document.querySelector('#sidebarTOC');
    const sidebarSearchElem = document.querySelector('#sidebarSearch');
    const sidebarWishlistElem = document.querySelector('#sidebarWishlist');
    const sidebarScrolls = document.querySelectorAll('.sidebar__scrollbar');

    if (sidebarElem) sidebarElem.style.background = `#${color}FF`;
    if (sidebarTocElem) sidebarTocElem.style.background = `#${color}FF`;
    if (sidebarSearchElem) sidebarSearchElem.style.background = `#${color}FF`;
    if (sidebarWishlistElem) sidebarWishlistElem.style.background = `#${color}FF`;

    if (sidebarScrolls && sidebarScrolls.length > 0) {
      Array.from(sidebarScrolls).forEach(scroll => scroll.style.backgroundColor = `#${color}FF`);
    }
  }

  this.customizeBookBackground = (action = null) => {
    const bg_color = this.customizeBookBackgroundColor__getCorrectAvgHex();

    if (transparent && !__is_full_screen) {
      if (root) {
        root.style.background = 'transparent';
        root.style.backgroundImage = '';
      }
      
      document.body.classList.remove(COLOR_THEME.DARK);
      document.body.classList.remove(COLOR_THEME.LIGHT);
      document.body.style.background = 'transparent';
      return;
    } else {
      document.body.style.background = bg_color ? `#${bg_color}` : '';
    }

    customizeSidebarsAndLoader(bg_color);
    //RGB z bgcolor bierzemy dla kazdej z 3 trybow bo najwyzej mamy tu AVG potrzbny dla theme
    this.customizeBookBackgroundTheme();

    if (bookModel.settings.background_mode == 0) {
      bookModel.settings.background = bg_color ? `#${bg_color}` : '';
      // root.style.removeProperty('background-image');
      if (root) root.style.backgroundImage = '';
      removeAnimatedBackdround();
    } else if (bookModel.settings.background_mode == 1 || bookModel.settings.background_mode == 2) {
      if (action === 'image_removed') {
        bookModel.settings.background_mode = 1;

        const splitedImagesUrl = root.style.backgroundImage.split(',');
        const newSplitedImagesUrl = splitedImagesUrl.filter(url => url.includes('/textures/'));

        root.style.backgroundImage = APP.bookModel.settings.background_texture.length > 1 ? `url(${APP.bookModel.settings.background_texture})` : newSplitedImagesUrl.join(',');

        return self.customizeBookBackground();
      } else {
        let bgImg = new Image();
        bookModel.settings.background = bg_color ? `#${bg_color}` : '';

        bgImg.onload = () => {
          if (root.style.backgroundImage) {
            if (!root.style.backgroundImage.includes(bgImg.src)) {
              root.style.backgroundImage = `url("${bgImg.src}"), ${root.style.backgroundImage}`;
              
              if (root.style.backgroundImage.split(',').length > 3) {
                const splitedImagesUrl = root.style.backgroundImage.split(',');
                splitedImagesUrl.pop();
                root.style.backgroundImage = splitedImagesUrl.join(',');
              }
            } else {
              const splitedImagesUrl = root.style.backgroundImage.split(',');
              const imageUrlIndex = splitedImagesUrl.findIndex(url => url.includes(bgImg.src));
  
              splitedImagesUrl.splice(imageUrlIndex, 1);
              splitedImagesUrl.unshift(`url("${bgImg.src}")`);
              root.style.backgroundImage = splitedImagesUrl.join(',');
            }
          } else {
            root.style.backgroundImage = `url("${bgImg.src}")`;
          }

          root.style.backgroundSize = 'cover';
          root.style.backgroundPosition = 'center center';
          root.style.backgroundRepeat = 'no-repeat';
        };

        bgImg.onerror = (error) => {
          //oLogs.log_send_error(<?=$iid?>, <?=$pid?>, {acc:'error', data:'ERR:'+error+' URL:'+url+' L:'+line});
          oLogs.log_send_error(model.bookId, model.publisherId, {
            acc: 'setBGError',
            data: 'ERR:' + bgImg.src + ' L:252',
          });
          // console_log('error', error);
        };

        if (bookModel.settings.background_mode == 1) {
          const newResolution = HELPER.getBgSizeByResolution();
          const background_texture = bookModel.settings.background_texture;

          let newUrl = HELPER.setBgUrlWithResolution(background_texture, newResolution);
          
          if (!root || 
            (root && !root.style.backgroundImage) || 
            (root && root.style.backgroundImage && !root.style.backgroundImage.includes(newUrl))
          ) {
            bgImg.src = newUrl;
          }
          if (root && root.style.backgroundImage.includes(newUrl)) {
            let backImgArr = root.style.backgroundImage.split(',');

            backImgArr = backImgArr.map(item => {
              item = item.replace(/^\s+/g, '');
              item = item.replace(/\s+$/g, '');
              return item
            });

            if (backImgArr && backImgArr.length > 1 && !backImgArr[0].includes(newUrl)) {
              const index = backImgArr.findIndex(item => item.includes(newUrl));
              backImgArr.splice(index, 1);
              backImgArr.unshift(`url('${newUrl}')`);
              let backImg = backImgArr.join(', ');
              root.style.backgroundImage = `${backImg}`;
            }
          }

          if (background_texture && background_texture.indexOf('an_bg_') !== -1) {
            const newVideoResolution = HELPER.getBgSizeByResolution(true);            
            let url = background_texture;

            if (!!newVideoResolution) {
              url = HELPER.setBgUrlWithResolution(url, newVideoResolution);
            }

            url = url.replace('.jpg', '_v.mp4');

            const video = initializeAnimatedBackdround();

            if (video.src !== url) {
              video.src = url;
            }

            return;
          } else {
            removeAnimatedBackdround();
          }
        } else {
          if (bookModel.settings.background_image && bookModel.settings.background_image.startsWith('data')) {
            bgImg.src = bookModel.settings.background_image;

            removeAnimatedBackdround();
          } else {
            bgImg.src = bookModel.settings.background_image;
          }
        }
      }

    }
  };

  const initializeAnimatedBackdround = () => {
    if (document.querySelector('#root video.player--animated-background')) {
      return document.querySelector('#root video.player--animated-background');
    }

    const video = document.createElement('video');

    video.muted = true;
    video.autoplay = true;
    video.loop = true;
    video.preload = "auto";

    video.setAttribute('playsinline', 'playsinline')
    video.className = 'player--animated-background';

    if (document.querySelector('#main')) {
      document.querySelector('#main').insertAdjacentElement('beforebegin', video);
    }

    return video;
  };

  const removeAnimatedBackdround = () => {
    if (document.querySelector('#root video.player--animated-background')) {
      document.querySelector('#root video.player--animated-background').remove();
    }
  };

  this.customizeBook = () => {
    if (bookModel) {
      this.customizeBookBackground();

      settings.showSinglePageMode = forceOnePage; //bookModel.settings.show_single_page_mode;
      forceOnePage = bookModel.settings.show_single_page_mode;
      //tu dla poziomych wymuszalismy 1 page
      if (bookModel.pages_info.length == 1) {
        // Sometimes, when all pages are same, array has only one item, so we need create this array by our self
        var s = {
          // page has base model from API, width, height
          width: parseInt(bookModel.pages_info[0].width),
          height: parseInt(bookModel.pages_info[0].height),
        };
        bookModel.pages_info = [];
        var n = 0;
        while (n < bookModel.pages) {
          bookModel.pages_info.push(s);
          n++;
        }
      } else {
        // if array has more than one item, we should parseInt data about pages
        for (var i = bookModel.pages_info.length - 1; i >= 0; i--) {
          bookModel.pages_info[i].width = parseInt(bookModel.pages_info[i].width);
          bookModel.pages_info[i].height = parseInt(bookModel.pages_info[i].height);
        }
      }
    }
  };

  this.renderProtected = ( data ) => {
    if (is_mobile()) {
      const windowWidth = window.innerWidth;
      const windowHeight = window.innerHeight;

      document.body.style.width = `${windowWidth}px`;
      document.body.style.height = `${windowHeight}px`;

      let localPass = localStorage.getItem('_book-pass');

      if (localPass && JSON.parse(localPass) && JSON.parse(localPass).length) {
        localPass = JSON.parse(localPass);
        const currentBook = localPass.filter(item => item.bookId == model.bookId)

        if (currentBook && currentBook.length && currentBook[0].pass) {
          const date = new Date();

          if (currentBook[0].expiredAt - date.getTime() > 0) {
            loadProtectedBook(decodeURIComponent(currentBook[0].pass || ''));
            return;
          } else {
            localPass = localPass.filter(item => item.bookId != model.bookId);
            localStorage.setItem('_book-pass', JSON.stringify(localPass));
          }
        }
      } 
    }

    APP.loadCSS('ProtectedModal', ['xs'], function () {
      const modal = document.querySelector('.protected-modal');
      const coverImage = modal.querySelector('.protected-modal__image');

      if (coverImage) {
        const coverUrl = data['thumbnailUrl'] && data['thumbnailUrl'].length > 10 
          ? data['thumbnailUrl']
          : `${APP.PATH_V2IMAGE}?iid=${model.bookId}&pid=${model.publisherId}&page=1&size=400&m=1&webp=${settings.webpSupport ? 1 : 0}&x=184`;

        coverImage.src = coverUrl;
      }

      modal.querySelector('.protected-modal__header').innerText = APP._t('Password Form Title');

      setTimeout(() => {
        if (modal) modal.style.display = 'flex';
      }, 100);

      const form = modal.querySelector('.protected-modal__form');
      const pswInput = modal.querySelector('.protected-modal__input');
      const passwordVisibility = modal.querySelector('.password-view');
      const btnSendPassword = modal.querySelector('#send-protected');

      pswInput.setAttribute('placeholder', APP._t('Password Form Placeholder'));
      btnSendPassword.innerText = APP._t('Password Form Send');

      if (passwordVisibility) {
        passwordVisibility.addEventListener('click', function () {
          if (!pswInput) return;

          passwordVisibility.classList.toggle('password-view--off');
          pswInput.type = pswInput.type === 'password' ? 'text' : 'password'
        });
      }

      form.addEventListener('submit', (event) => {
        event.preventDefault();

        if (pswInput && pswInput.value) {
          modal.style.display = 'none';
          loadProtectedBook(pswInput.value);
        }
      });
    });
  };

  var bookNode = null, // The book Node element
    bookLoader = null,
    root = null,
    elements = {}, // The book sub elements storage
    setBookInnerSize,
    updateArrows,
    setMouseEvents;

  this.getElements = function () {
    return elements;
  };

  const changeMenuButtonsOnZoom = (isZoomed) => {
    if (!isZoomed) {
      elements.navTOC.hidden = true;
      elements.navThumbs.hidden = true;
      elements.download.hidden = true;
      elements.share.hidden = true;
      elements.note.hidden = true;
      elements.audioIcon.hidden = true;
      elements.print.hidden = true;
      elements.wishlist.hidden = true;
      elements.autoFlip.hidden = true;
      elements.navScaler.hidden = false;
      elements.navScaler.classList.add('show');
      elements.zoomIn.classList.add('Book__zoom--out');

      if (elements.zoomIn.dataset && elements.zoomIn.dataset.type === BUTTON_ICON_TYPES.INLINE) {
        elements.zoomIn.innerHTML = HELPER.buttonsIcons.closeIcon;
      }
    } else {
      elements.navTOC.hidden = 
        !(bookModel.settings.show_toc_button && Array.isArray(bookModel.toc) && !!APP.isStringified(bookModel.toc).length);
      elements.navThumbs.hidden = !bookModel.settings.show_thumbnails_button;
      elements.download.hidden = !bookModel.settings.show_download_button;
      elements.share.hidden = !bookModel.settings.show_share_button;
      elements.note.hidden = !bookModel.settings.show_note_button;
      elements.audioIcon.hidden = !bookModel.settings.show_audio_button;
      elements.print.hidden = !bookModel.settings.show_print_button;
      elements.wishlist.hidden = self.getInteractivitiesByType('product-info', { 'show_wishlist': [1, true] }, false).length <= 0;
      elements.autoFlip.hidden = !bookModel.settings.auto_flip_enabled;
      elements.navScaler.classList.remove('show');
      elements.zoomIn.classList.remove('Book__zoom--out');
      elements.navScaler.hidden = true;

      if (elements.zoomIn.dataset && elements.zoomIn.dataset.type === BUTTON_ICON_TYPES.INLINE) {
        elements.zoomIn.innerHTML = HELPER.buttonsIcons.zoomInIcon;
      }
    }
  };

  this.render_setMenuIconsListeners__zoomin = function () {
    if (is_mobile()) return;

    elements.zoomIn.addEventListener('click', function () {
      if (elements.zoomIn.classList.contains('Book__zoom--out')) {
        changeMenuButtonsOnZoom(true);
        self.scale(0);
      } else {
        changeMenuButtonsOnZoom(false);
        self.scale(settings.MAX_SCALE / 5);
      }
    });
  };

  const generateQRCode = (shareLink) => {
    const qrCodeWrapper = elements.shareModalElements.qrCodeWrapper;

    if (!qrCodeWrapper || qrCodeWrapper.querySelector('img')) return;

    const shareUrl = new URL(shareLink);
    shareUrl.searchParams.set('ref_type', 'qrcode');

    const qr = new QRious({
      element: qrCodeWrapper,
      value: shareUrl.toString(),
      background: 'white',
      foreground: 'black',
      backgroundAlpha: 1,
      foregroundAlpha: 1,
      level: 'L',
      mime: 'image/png',
      size: 100,
      padding: null
    });

    const qrCodeImage = document.createElement('img');
    const qrCodeDownloadBtn = document.createElement('a');

    qrCodeImage.src = qr.toDataURL("image/png");
    qrCodeWrapper.appendChild(qrCodeImage);

    qr.size = 256;
    qrCodeDownloadBtn.href = qr.toDataURL("image/png");
    qrCodeDownloadBtn.className = 'modal__body__content_qrcode__btn'
    qrCodeDownloadBtn.innerHTML = '<svg width="20px" height="20px" viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg"><path fill="#fff" d="M10 17.5L3.5 11H7V3h6v8h3.5L10 17.5z"/></svg>';
    qrCodeDownloadBtn.download = `${APP._t('QR Code Name')}.png`;
    
    qrCodeWrapper.appendChild(qrCodeDownloadBtn);
  };

  const defineShareModal = () => {
    if (elements.shareModalElements) {
      return elements.shareModalElements;
    }

    elements.shareModalElements = {};

    const closeShareModal = () => {
      const callbackFn = shareModal.handleClose && typeof(shareModal.handleClose) === 'function' ? shareModal.handleClose : null;

      HELPER.fadeOutEffect(shareModal, shareModalContainer, MODAL_FADE_TIME, () => {
        HELPER.sendPostMessageToParent('publuu.viewer.popup.share.close');

        if (callbackFn) {
          callbackFn();
        }
      });
    };

    const closeShareModalOnClickOutside = (event) => {
      const shouldCloseModal = !(event.path || event.composedPath()).find(el => el === shareModalContainer);

      if (!shouldCloseModal) return;

      closeShareModal();
    };

    const shareModal = document.createElement('div');
    const shareModalContainer = document.createElement('div');
    const shareModalCloseBtn = document.createElement('button');

    const shareModalHeader = document.createElement('div');
    const shareModalHeaderIconWrapper = document.createElement('div');
    const shareModalHeaderIcon = document.createElement('img');
    const shareModalHeaderTitle = document.createElement('h2');

    const shareModalBody = document.createElement('div');
    const shareModalBodyContent = document.createElement('div');
    const shareModalBodyContentLinks = document.createElement('div');
    const shareModalBodyContentLinkFacebook = document.createElement('a');
    const shareModalBodyContentLinkLinkedIn = document.createElement('a');
    const shareModalBodyContentLinkTwitter = document.createElement('a');
    const shareModalBodyContentLinkEmail = document.createElement('a');
    const shareModalBodyContentLinkDownload = document.createElement('a');
    const shareModalBodyContentLinkFacebookBtn = document.createElement('button');
    const shareModalBodyContentLinkLinkedInBtn = document.createElement('button');
    const shareModalBodyContentLinkTwitterBtn = document.createElement('button');
    const shareModalBodyContentLinkEmailBtn = document.createElement('button');
    const shareModalBodyContentLinkDownloadBtn = document.createElement('button');
    const shareModalBodyContentQrCodeWrapper = document.createElement('div');
    
    const shareModalFooter = document.createElement('div');
    const shareModalFooterText = document.createElement('p');
    const shareModalFooterLinkWrapper = document.createElement('div');
    const shareModalFooterLink = document.createElement('div');
    const shareModalFooterBtn = document.createElement('button');

    shareModal.className = 'modal modal--share';
    shareModalContainer.className = 'modal__container';
    shareModalCloseBtn.className = 'modal__close';

    shareModalHeaderIconWrapper.className = 'modal__header__icon';
    shareModalHeader.className = 'modal__header';
    shareModalHeaderTitle.className = 'modal__title';
    
    shareModalBody.className = 'modal__body';
    shareModalBodyContent.className = 'modal__body__content';
    shareModalBodyContentLinks.className = 'modal__body__content__links';

    shareModalBodyContentLinkFacebook.id = 'share-modal__fb';
    shareModalBodyContentLinkLinkedIn.id = 'share-modal__linkedin';
    shareModalBodyContentLinkTwitter.id = 'share-modal__twitter';
    shareModalBodyContentLinkEmail.id = 'share-modal__mail';
    shareModalBodyContentLinkDownload.id = 'share-modal__download';
    shareModalBodyContentLinkFacebookBtn.className = 'share-modal__fb';
    shareModalBodyContentLinkLinkedInBtn.className = 'share-modal__linkedin';
    shareModalBodyContentLinkTwitterBtn.className = 'share-modal__twitter';
    shareModalBodyContentLinkEmailBtn.className = 'share-modal__mail';
    shareModalBodyContentLinkDownloadBtn.className = 'share-modal__download';
    shareModalBodyContentLinkFacebook.target = '_blank';
    shareModalBodyContentLinkLinkedIn.target = '_blank';
    shareModalBodyContentLinkTwitter.target = '_blank';
    shareModalBodyContentLinkEmail.target = '_blank';

    shareModalBodyContentQrCodeWrapper.className = 'modal__body__content_qrcode';

    shareModalFooter.className = 'modal__footer';
    shareModalFooterText.className = 'modal__footer__text';
    shareModalFooterLinkWrapper.className = 'modal__footer__link__wrapper';
    shareModalFooterLink.className = 'modal__footer__link';
    shareModalFooterBtn.className = 'modal__footer__button';

    shareModalHeaderTitle.textContent = APP._t('Share Form Title');
    shareModalHeaderIcon.alt = 'Share icon';
    shareModalHeaderIcon.src = `${APP.PATH_CF_FLIP}assets/images/ico_share2.svg`;
    shareModalFooterText.textContent = APP._t('Share Form Copy Link');
    shareModalFooterBtn.textContent = APP._t('Share Form Copy');

    shareModalBodyContentLinkDownload.hidden = !bookModel.settings.show_download_button;
    shareModalBodyContentLinkDownload.addEventListener('click', (event) => {
      event.preventDefault();
      downloadPdf();
    });

    shareModal.hidden = true;
    shareModal.hide = closeShareModal;

    shareModalCloseBtn.innerHTML = HELPER.buttonsIcons.popupCloseIcon;
    
    shareModalCloseBtn.addEventListener('click', closeShareModal);
    shareModal.addEventListener('click', closeShareModalOnClickOutside); 
    shareModalFooterBtn.addEventListener('click', (event) => HELPER.copyToClipboard(event.target, 'modal__footer__link', APP._t('Share Form Copied')));
    shareModalFooterLink.addEventListener('click', (event) => HELPER.copyToClipboard(event.target, 'modal__footer__link', APP._t('Share Form Copied')));

    window.addEventListener('keydown', (event) => {
      if (!shareModal || shareModal.hasAttribute('hidden')) return;
      if (event.code !== 'Escape') return;

      closeShareModal();
    });

    shareModalHeaderIconWrapper.appendChild(shareModalHeaderIcon);
    shareModalHeader.appendChild(shareModalHeaderIconWrapper);
    shareModalHeader.appendChild(shareModalHeaderTitle);

    shareModalFooter.appendChild(shareModalFooterText);
    shareModalFooterLinkWrapper.appendChild(shareModalFooterLink);
    shareModalFooterLinkWrapper.appendChild(shareModalFooterBtn);
    shareModalFooter.appendChild(shareModalFooterLinkWrapper);
  
    shareModalBody.appendChild(shareModalCloseBtn);
    shareModalBodyContentLinkFacebook.appendChild(shareModalBodyContentLinkFacebookBtn);
    shareModalBodyContentLinkLinkedIn.appendChild(shareModalBodyContentLinkLinkedInBtn);
    shareModalBodyContentLinkTwitter.appendChild(shareModalBodyContentLinkTwitterBtn);
    shareModalBodyContentLinkEmail.appendChild(shareModalBodyContentLinkEmailBtn);
    shareModalBodyContentLinkDownload.appendChild(shareModalBodyContentLinkDownloadBtn);
    shareModalBodyContentLinks.appendChild(shareModalBodyContentLinkFacebook);
    shareModalBodyContentLinks.appendChild(shareModalBodyContentLinkLinkedIn);
    shareModalBodyContentLinks.appendChild(shareModalBodyContentLinkTwitter);
    shareModalBodyContentLinks.appendChild(shareModalBodyContentLinkEmail);
    shareModalBodyContentLinks.appendChild(shareModalBodyContentLinkDownload);
    shareModalBodyContent.appendChild(shareModalBodyContentLinks);
    shareModalBodyContent.appendChild(shareModalBodyContentQrCodeWrapper);
    shareModalBody.appendChild(shareModalBodyContent);
    shareModalBody.appendChild(shareModalFooter);

    shareModalContainer.appendChild(shareModalHeader);
    shareModalContainer.appendChild(shareModalBody);
    shareModal.appendChild(shareModalContainer);

    document.body.appendChild(shareModal);

    elements.shareModalElements = {
      modal: shareModal,
      container: shareModalContainer,
      bodyContent: shareModalBodyContent,
      closeBtn: shareModalCloseBtn,
      footerLink: shareModalFooterLink,
      footerBtn: shareModalFooterBtn,
      qrCodeWrapper: shareModalBodyContentQrCodeWrapper,
      socialsLinks: {
        facebook: shareModalBodyContentLinkFacebook,
        linkedIn: shareModalBodyContentLinkLinkedIn,
        twitter: shareModalBodyContentLinkTwitter,
        email: shareModalBodyContentLinkEmail,
      }
    };
  };

  const replaceShareLinks = (shareLink) => {
    const name = encodeURIComponent((bookModel.name || '').toString().split('"').join("'"));
    
    elements.shareModalElements.footerLink.innerHTML = shareLink.replace('-canva', '');
  
    elements.shareModalElements.socialsLinks.facebook.href = `https://www.facebook.com/sharer.php?u=${shareLink}&x=${Date.now()}`;
    elements.shareModalElements.socialsLinks.twitter.href = `http://twitter.com/share?text=${name}&url=${shareLink}`;
    elements.shareModalElements.socialsLinks.linkedIn.href = `http://www.linkedin.com/shareArticle?mini=true&url=${shareLink}`;
    elements.shareModalElements.socialsLinks.email.href = `mailto:?subject=${name}&body=${shareLink}`;
  };

  const onShareClick = function (_, sharedPageIndex) {
    const shareLink = sharedPageIndex
      ? window.location.href.split('/page/')[0] + `/page/${sharedPageIndex}`
      : window.location.href;

    defineShareModal();
    replaceShareLinks(shareLink);

    const script = document.createElement('script');
    script.type = 'text/javascript';
    script.src = `${APP.PATH_CF_FLIP}assets/scripts/workers/qrious.min.js`;
    script.addEventListener('load', () => generateQRCode(shareLink));
    document.body.appendChild(script);

    const shareModal = elements.shareModalElements.modal;
    const shareModalContainer = elements.shareModalElements.container;
    HELPER.fadeInEffect(shareModal, shareModalContainer, MODAL_FADE_TIME, () => HELPER.sendPostMessageToParent('publuu.viewer.popup.share.open'));
  };

  const downloadPdf = () => {
    if (!bookModel.settings.show_download_button) return;

    const pass = this.password ? `password=${this.password}` : '';
    const bookId = bookModel.hasOwnProperty('ref_id') && !!bookModel.ref_id ? bookModel.ref_id : model.bookId;
    const version = Number(bookModel.version) && Number(bookModel.version) > 0 ? `&v=${bookModel.version}` : '';

    if (bookModel.hasOwnProperty('download_link') && bookModel.download_link) {
      window.open(bookModel.download_link);

      fetch(`${APP.API_PUBLUU}/publishers/${model.publisherId}/issues/${bookId}/downloaded-pdf?cm=${customizemode}`);
      return;
    }

    window.open(
      `${APP.API_PUBLUU}/publishers/${model.publisherId}/issues/${bookId}/pdf?${pass}&cm=${customizemode}${version}`,
    );
  };

  this.render_setMenuIconsListeners = function () {
    //Menu icons listeners
    this.render_setMenuIconsListeners__zoomin();

    elements.audioIcon.addEventListener('click', function (e) {
      elements.audioIcon.classList.toggle('sound-off');
      const divForAudioHotspot = document.querySelector('.divForAudioHotspot');

      if (elements.audioIcon.classList && elements.audioIcon.classList.contains) {
        const descriptionText = 
          elements.audioIcon.classList.contains('sound-off') 
            ? APP._t('Elements Description Audio On') 
            : APP._t('Elements Description Audio Off');
        
        elements.audioIcon.ariaLabel = descriptionText;
        elements.audioIcon.title = descriptionText;

        if (BUTTON_ICON_TYPE === BUTTON_ICON_TYPES.INLINE) {
          const audioIcon = elements.audioIcon.classList.contains('sound-off') ? HELPER.buttonsIcons.soundOffIcon : HELPER.buttonsIcons.soundOnIcon;
          elements.audioIcon.innerHTML = audioIcon;
        }
      }

      if (divForAudioHotspot && divForAudioHotspot.querySelector('audio')) {
        divForAudioHotspot.querySelector('audio').volume = elements.audioIcon.classList.contains('sound-off') ? 0 : 1;
      }
    });

    elements.download.addEventListener('click', downloadPdf);
    elements.share.addEventListener('click', onShareClick);
    elements.print.addEventListener('click', printModalHandler);
  };

  this.render__createBookLoader = function () {
    // (transparent || coverMode) && 
    if (document.querySelector('.app__loader')) {
      bookLoader = document.querySelector('.app__loader');
      return;
    }
    // Create Nodes for all other elements
    bookLoader = document.createElement('div');
    bookLoader.className = 'loader';

    const bookLoaderWrapper = document.createElement('div');
    bookLoaderWrapper.className = 'loader__wrapper';

    if (coverMode) {
      bookLoader.classList.add('no-animation');
    }

    if (!transparent) {
      bookLoader.style.backgroundColor = `#${this.customizeBookBackgroundColor__getCorrectAvgHex()}`;
      const logo = HELPER.bookLoaderDiv_LogoImageDiv(bookModel);

      if (bookModel.settings.toplogo_image) {
        let logoUrl = bookModel.settings.toplogo_image;

        // if (logoUrl && model.is_canva == 1) {
        //   logoUrl = logoUrl.replace('/publishers//', '/publishers/').replace('/publishers/', '/publishers//');
        // }

        const img = new Image();
        img.addEventListener('load', (e) => {
          document.querySelector('.loader__logo-container').classList.add('show');
        })
        img.src = logoUrl;
      }

      bookLoaderWrapper.insertAdjacentHTML(
        'afterbegin',
        HELPER.bookLoaderDiv(
          logo,
          isConverted ? bookModel.name : APP._t('State Still Converting'),
        ),
      );
      bookLoader.appendChild(bookLoaderWrapper);
    } else {
      //transparent
      
      let loaderContent = `<div class='loader__transparent'></div>`;
      bookLoader.insertAdjacentHTML('afterbegin', loaderContent);
    }

    document.body.appendChild(bookLoader);
    this.timerStart = performance.now();
  };

  this.render__createPages = function () {
    let fragmentPages = document.createDocumentFragment();
    for (let i = 1; i <= bookModel.pages; i++) {
      let page = document.createElement('div');
      page.className = `Page _page_${i}`;
      fragmentPages.append(page);
    }
    elements.innerChild.append(fragmentPages);
  };

  this.render__createNextPrevAbsoluteButtons = function () {
    // navigation buttons next/prev page
    elements.navPrevPageAbsolute = document.createElement('button');
    elements.navNextPageAbsolute = document.createElement('button');
    elements.navPrevPageAbsolute.className = 'Book__navPrevPage arrows-absolute';
    elements.navNextPageAbsolute.className = 'Book__navNextPage arrows-absolute';
    elements.navNextPageAbsolute.hidden = true;
    elements.navPrevPageAbsolute.hidden = true;
    elements.navPrevPageAbsolute.addEventListener('click', self.prevPage);
    elements.navNextPageAbsolute.addEventListener('click', self.nextPage);
    bookNode.appendChild(elements.navNextPageAbsolute);
    bookNode.appendChild(elements.navPrevPageAbsolute);
  };

  this.render__setArrowsArea = function () {
    elements.navAreaPrevPage = document.createElement('div');
    elements.navAreaNextPage = document.createElement('div');

    elements.navAreaPrevPage.className = 'Book__navAreaPrevPage';
    elements.navAreaNextPage.className = 'Book__navAreaNextPage';
    elements.navAreaPrevPage.hidden = true; // prev page by default is hidden
    elements.navAreaNextPage.hidden = true;
    elements.navAreaPrevPage.addEventListener('click', self.prevPage);
    elements.navAreaNextPage.addEventListener('click', self.nextPage);
  };

  this.render__setArrows = function () {
    // navigation buttons next/prev page
    elements.navPrevPage = document.createElement('button');
    elements.navNextPage = document.createElement('button');
    elements.navPrevPage.className = 'Book__navPrevPage arrows-relative';
    elements.navNextPage.className = 'Book__navNextPage arrows-relative';
    elements.navPrevPage.name = APP._t('Elements Description Previous Page');
    elements.navNextPage.name = APP._t('Elements Description Next Page');
    elements.navPrevPage.ariaLabel = APP._t('Elements Description Previous Page');
    elements.navNextPage.ariaLabel = APP._t('Elements Description Next Page');
    elements.navPrevPage.title = APP._t('Elements Description Previous Page');
    elements.navNextPage.title = APP._t('Elements Description Next Page');
    elements.navNextPage.innerHTML = HELPER.arrow;
    elements.navPrevPage.innerHTML = HELPER.arrow;
    elements.navPrevPage.hidden = true; // prev page by default is hidden
    elements.navNextPage.hidden = true;
    elements.navPrevPage.addEventListener('click', self.prevPage);
    elements.navNextPage.addEventListener('click', self.nextPage);

    if (transparent && !__is_full_screen) {
      elements.navPrevPage.className += ' transparent';
      elements.navNextPage.className += ' transparent';
    }
  };

  this.getCurrentPageForLogs = function (callback) {
    let availablePage = currentPage || startPage;

    if (availablePage < 1) {
      availablePage = 1;
    } 

    if (!bookModel || !bookModel.pages) {
      return null;
    }

    if (availablePage > bookModel.pages) {
      availablePage = bookModel.pages;
    }

    if (
      is_vertical() ||
      settings.showSinglePageMode ||
      (is_mobile() && !mobile_test_horizontal())
    ) {
      return [availablePage - 1];
    } else {
      if (availablePage === 1) {
        return [availablePage - 1];
      } else {
        if (availablePage === parseInt(bookModel.pages)) {
          return [availablePage - 1];
        }
        return [availablePage - 1, availablePage];
      }
    }
  };

  const createOrientationChangeLoader = () => {
    const bgColor = transparent ? 'fff' : this.customizeBookBackgroundColor__getCorrectAvgHex();

    const loader = document.createElement('div');
    loader.className = 'loader--simple';
    loader.innerHTML = HELPER.buttonsIcons.loaderIcon;

    loader.style.background = bgColor ? `#${bgColor}` : '';

    document.body.appendChild(loader);

    return loader;
  };

  const hideOrientationChangeLoader = () => {
    const loader = document.querySelector('.loader--simple');

    if (!loader) return;

    loader.classList.remove('show');

    setTimeout(() => {
      loader.hidden = true;
    }, 300);
  };

  const showOrientationChangeLoader = (cb = null) => {
    let loader = document.querySelector('.loader--simple');

    document.body.style.width = '100%';
    document.body.style.height = '100%'

    if (!loader) {
      loader = createOrientationChangeLoader();
    }

    loader.removeAttribute('hidden');
    loader.classList.add('show');

    if (cb && typeof(cb) === 'function') {
      loader.addEventListener('transitionend', handlerOrientationChangeLoader);
    }
  };

  const handlerOrientationChangeLoader = (e) => {
    e.target.removeEventListener('transitionend', handlerOrientationChangeLoader);

    if (is_horisontal()) {
      hideMenuIfNeeded(true);
    } else {
      if (transparent && !__is_full_screen) return;

      showMenuIfNeeded();
    }
  };

  const handleInnerTransitionEnd = (event) => {
    if (event.srcElement === elements.inner && event.propertyName === "transform") {
      isBookInnerTransitionEnd = event.type === 'transitionend';
    }
  };

  this.render = function () {
    /*
     The main method to render Book.
     return false if has no bookModel or if bookNode not rendered and true if bookNode is rendered
     This method should render once.
     */

    //APP.log('Book->render');
    if (!bookModel) return false; // if we have no bookModel, we can't render
    if (bookNode) return false; // if we allready render book, we shouldn't do it again

    // Prepare preview before all other stuff rendered
    bookNode = document.createElement('div'); // Main Book element
    bookNode.className = ['Book _publisherId_', model.publisherId, ' _bookId_', model.bookId].join(
      '',
    );
    APP.Layout.getMain().appendChild(bookNode); // Insert Main Book element into Layout.main
    root = document.getElementById('root');
    root.hidden = true;
    root.classList.add('roothidden');

    root.addEventListener('click', function (e) {
      if (
        !e.target.classList.contains('page-num-input') &&
        document.activeElement.classList.contains('page-num-input')
      ) {
        self.setFooterPage(currentPage);
        elements.pageNumInput.blur();
      }
    });

    if (bookModel.settings.hard_cover_enabled) {
      settings.paddingX = (is_mobile() ? 20 : 36) + (is_mobile() ? 0 : 20);
    } else {
      settings.paddingX = 0;
    }

    const createSidebars = () => {
      if (!settings.sidebar) {
        return;
      }

      const bookSettings = self.getBookModelSettings();

      elements.sidebar = APP.Layout.render__createSidebar();
      self.renderSidebar();

      if (bookSettings.show_toc_button || customizemode) {
        elements.sidebarToc = APP.Layout.render__createSidebarTOC(bookModel.toc);
      }

      if (bookSettings.show_search_button) {
        APP.loadComponent('Search', () => {
          elements.sidebarSearch = APP.Layout.render__createSidebarSearch();
        });
      }

      const localStorageProducts = HELPER.getLocalStorageProducts(bookModel.id);

      if (!!self.getInteractivitiesByType('product-info').length || localStorageProducts) {
        APP.loadCSS('Products', ['xs']);

        APP.loadComponent('Products', () => {
          elements.sidebarWishlist = APP.Layout.render_createSidebarWishlist();
        });
      }
    };

    // Load css for the Book, after onload we can start rendering
    APP.loadCSS('Book', ['xs'], function () {
      // Create Nodes for all other elements
      self.render__createBookLoader();
      //For Canva 
      if(!bookModel.is_converted && isCanva) { console_log("Still converting..."); return false;}

      elements.footer = document.getElementById('footer');
      if (!is_mobile()) {
        hideMenuIfNeeded();
      }

      if (transparent) {
        hideMenuIfNeeded();
        hideInnerBookButtons();
      }

      defineMenuButtons();
      defineHeaderLogo();
      setMenuHeader();
      setMenuFooter();

      elements.preloaders = document.createElement('div'); // preload images pagent. required to preload images
      elements.preloaders.className = 'Book__preloaders';

      bookNode.appendChild(elements.preloaders);
      if (settings.thumbsNumPreload < 0) {
        self.preloadThumbnails();
      }

      elements.bookWrapper = document.createElement('div');
      elements.inner = document.createElement('div'); // book inner
      elements.innerChild = document.createElement('div'); // book innerChild

      elements.inner.addEventListener('transitionstart', handleInnerTransitionEnd, false);
      elements.inner.addEventListener('transitionend', handleInnerTransitionEnd, false);

      window.addEventListener('beforeunload', () => {
        handleBeforeUnloadInFullscreen();

        elements.inner.removeEventListener('transitionstart', handleInnerTransitionEnd, false);
        elements.inner.removeEventListener('transitionend', handleInnerTransitionEnd, false);
      });

      elements.bookWrapper.className = 'Book__wrapper';
      elements.inner.className = 'Book__inner';
      elements.innerChild.className = 'Book__innerChild';
      elements.innerChild.id = 'book';

      defineInnerChildButtons();
      defineFullscreenModal();
      defineAbsoluteFullscreenButton();

      self.render__createPages(); //coment to lock loader //LOADER

      self.render_setMenuIconsListeners();

      // Let's add some Events
      if (settings.detectOrientationChange) {
        window.addEventListener('orientationchange', function () {
          isOrientationChangeResize = true;
          showOrientationChangeLoader(handlerOrientationChangeLoader);
          
          // this event fires before real resize, so...
          // APP.Layout.getSidebar().hide();
          self.hideSidebar();
          setTimeout(function () {
            if (is_mobile()) {
              if (mobile_test_horizontal()) {
                elements.sidebar.classList.add('Book__sidebar--xs-landscape');
              } else {
                elements.sidebar.classList.remove('Book__sidebar--xs-landscape');
              }
            }
            elements.pagesRanger.updateMax(getCorrectNumberOfIndexes());
            var prevPage = currentPage; // backup currentPage
            currentPage = -1; // set minused currentPage
            self.setPage(prevPage, 'orientationchange'); // restore currentPage
            pageFlip.update();
            self.renderSidebar();
            self.changeArrows(false);
          }, 300);
        });
      }
      
      createSidebars();
      
      elements.previewBack.addEventListener('click', previewBackToggle);
      elements.navFullScreen.addEventListener('click', fullScreenToggle);
      elements.navThumbs.addEventListener('click', self.showSidebar);
      elements.navTOC.addEventListener('click', self.toggleSidebarTOC);
      elements.search.addEventListener('click', self.showSidebarSearch);
      elements.wishlist.addEventListener('click', self.showSidebarWishlist);
      elements.autoFlip.addEventListener('click', autoFlipHandler);

      if (settings.nextPrevButtons) {
        self.render__createNextPrevAbsoluteButtons();
      }

      if (!is_mobile() && !is_touchscreen()) {
        // For desktop also simple logic, but plus wheel and keyboard events and minus searching doubletouch events
        APP.Layout.getBody().addEventListener('keydown', function (e) {
          const mediaViewer = document.querySelector('.media-viewer');
          const galleryModal = elements.galleryModalElements && elements.galleryModalElements.modal ? elements.galleryModalElements.modal : document.querySelector('.modal--gallery');
          const videoModal = elements.videoModalElements && elements.videoModalElements.modal ? elements.videoModalElements.modal : document.querySelector('.modal--video');

          if (mediaViewer && !mediaViewer.hidden) return;
          if (galleryModal && !galleryModal.hidden) return;
          if (videoModal && !videoModal.hidden) return;

          let should_stop = false;

          switch (e.key) {
            case 'ArrowRight':
              self.nextPage();
              should_stop = true;
              break;
            case 'ArrowLeft':
              self.prevPage();
              should_stop = true;
              break;
          }

          if (should_stop) {
            stopBookAutoFlip();
            stopEvent(e);
            return;
          };
        });

        elements.inner.addEventListener('mousedown', mouseDownHandler);
        elements.inner.addEventListener('mousemove', mouseMoveHandler);
        elements.inner.addEventListener('mouseup', mouseUpHandler);
        elements.innerChild.addEventListener('dblclick', doubleClickHandler);

        if (!is_mobile()) {
          window.addEventListener('contextmenu', renderContextMenu);
          elements.inner.addEventListener('click', self.contextMenuToggle);
        }

        elements.innerChild.addEventListener(
          'wheel',
          function (e) {
            if (embedmode && !__is_full_screen) {
              //console_log("block");
            } else {
              e.preventDefault();

              if (pageFlip.isBookZoomed()) {
                if (isNaN(xMove) || isNaN(yMove)) {
                  xMove = 0;
                  yMove = 0;
                }

                xMove += -1 * e.deltaX;
                yMove += -1 * e.deltaY;

                let coordinates = self.scaledTransform(xMove, yMove);
                xMove = coordinates.x;
                yMove = coordinates.y;
                moveCoordinates = { x: xMove, y: yMove };
              }

              return stopEvent(e);
            }
          },
          { passive: false },
        );

        window.addEventListener('mouseup', mouseUpHandler);

        [
          'fullscreenchange',
          'webkitfullscreenchange',
          'mozfullscreenchange',
          'MSFullscreenChange',
        ].forEach((eventType) => document.addEventListener(eventType, handleFullscreenChange, false));
      } else {
        //Touch events
        // if( is_mobile() )
        if (!transparent) {
          Object.values(elements).forEach(item => {
            let element = item;

            if (!(item instanceof HTMLElement)) {
              element = item.hasOwnProperty('modal') && item.modal instanceof HTMLElement ? item.modal : null;
            }

            if (!element) return;

            element.addEventListener('touchstart', e => {
              if (element != elements.inner && !canBeZoomedOnThisElement(e)) {
                return stopEvent(e);
              }
            });
          });
          
          document.querySelector('.loader').addEventListener('touchstart', stopEvent);
          elements.inner.addEventListener('touchstart', touchStartHandler);
          elements.inner.addEventListener('touchmove', touchMoveHandler);
          elements.inner.addEventListener('touchend', touchEndHandler);
          elements.inner.addEventListener('touchcancel', touchEndHandler);
        }
      }

      updateArrows = function () {
        elements.navPrevPage.className = 'Book__navPrevPage arrows-relative';
        elements.navNextPage.className = 'Book__navNextPage arrows-relative';
        if (transparent && !__is_full_screen) {
          elements.navPrevPage.className += ' transparent';
          elements.navNextPage.className += ' transparent';
        }
      };

      setMouseEvents = function () {
        APP.Layout.getMain().addEventListener(
          'mouseleave',
          function (event) {
            if (pageFlip.getState() === 'user_fold') {
              var evt = document.createEvent('Event');
              evt.initEvent('publuuwindowleave', true, true);

              window.dispatchEvent(evt);
            }
          },
          false,
        );
      };

      setBookInnerSize = async function (e, initiator) {
        if (isOrientationChangeResize) {
          await HELPER.waitingPromise(300);
        }
        
        // This event created to resize innerChild by page size
        elements.inner.scrollLeft = 0;
        elements.inner.scrollTop = 0;
        const root = document.querySelector('#root');
        let windowWidth = APP.Layout.getRootWidthDependOnSidebarState();
        let windowHeight = window.innerHeight;

        root.style.width = `${windowWidth}px`;
        root.style.height = `${window.innerHeight}px`;

        document.body.style.width = `${window.innerWidth}px`;
        document.body.style.height = `${window.innerHeight}px`;

        // bookNode.style.width = `${root.clientWidth}px`;
        // bookNode.style.height = `${root.clientHeight}px`;

        bookNode.style.width = `${windowWidth}px`;
        bookNode.style.height = `${windowHeight}px`;

        if (is_iphone_safari()) {
          if (is_vertical()) {
            document.documentElement.style.position = 'absolute';
          } else {
            document.documentElement.style.position = 'fixed';
          }
          document.documentElement.style.position = 'fixed';
          document.body.scrollTop = 0;
          document.documentElement.scrollTop = 0;
        }

        let maxWidth = windowWidth - settings.padding - settings.paddingX;

        if (!is_mobile()) {
          const thicknessWidth = self.getThicknessWidth().reduce((acc, item) => acc + Math.abs(item), 0);
          maxWidth = windowWidth - ((settings.padding + settings.paddingX) * 1 + 90) - thicknessWidth;
        }

        let maxHeight = windowHeight - settings.padding * 2 - 0;

        if (!is_mobile()) {
          let menuHeight = windowHeight * (settings.menuHProc / 100);
          menuHeight = menuHeight < settings.menuMinH ? settings.menuMinH : menuHeight;
          menuHeight = menuHeight > settings.menuMaxH ? settings.menuMaxH : menuHeight;

          maxHeight = windowHeight - settings.padding * 2 - 2 * menuHeight - (bookModel.settings.hard_cover_enabled ? 28 : 0);
        }

        const indexWidth = maxWidth / bookModel.pages_info[0].width;
        const indexHeight = maxHeight / bookModel.pages_info[0].height;

        let iOn = indexWidth < indexHeight ? indexWidth : indexHeight;
        let width = bookModel.pages_info[0].width * iOn * 2;
        let height = bookModel.pages_info[0].height * iOn;

        if (is_mobile()) {
          let mnoznik_szerokosci = forceOnePage ? 1 : 2;
          mnoznik_szerokosci = !mobile_test_horizontal() ? 1 : mnoznik_szerokosci;

          width = bookModel.pages_info[0].width * iOn * mnoznik_szerokosci;
          height = bookModel.pages_info[0].height * iOn;
        } else {
          if (is_vertical() || settings.showSinglePageMode || forceOnePage) {
            height = height * 2;
          }
        }

        if (height > maxHeight) {
          iOn = maxHeight / height;
          width = width * iOn;
          height = height * iOn;
        }

        if (width > maxWidth) {
          iOn = maxWidth / width;
          width = width * iOn;
          height = height * iOn;
        }

        if (width % 2 === 1) {
          width -= 1;
        }

        requestAnimationFrame(function () {
          elements.inner.style.paddingTop = `${getBookTopOffset()}px`;
          elements.innerChild.style.width = `${Math.round(width)}px`;
          elements.innerChild.style.height = `${Math.round(height)}px`;
          elements.navAreaPrevPage.style.height = `${Math.round(height)}px`;
          elements.navAreaNextPage.style.height = `${Math.round(height)}px`;

          const { bookWidth, bookHeight } = self.getBookSize();

          if (pageFlip) pageFlip.setSettings(bookWidth, bookHeight);
          pageWidth = bookWidth;
          pageHeight = bookHeight;

          self.setPageImageByBookSize(bookHeight, false);

          if (APP.Note && APP.Note.setNoteSizeOnResize) {
            APP.Note.setNoteSizeOnResize(pageWidth, pageHeight);
          }

          updateArrowsPositions();

          setTimeout(() => {
            for (let i of pagesToLoad) {
              if (pages[i]) {
                if (pages[i].getPage().style.display === 'block') {
                  pages[i].hotspotsRender(pageWidth, pageHeight);
                  pages[i].adjustHotspotsSize(pageWidth, pageHeight);
                  pages[i].hotspotsOn();
                } else {
                  pages[i].removeHotspots();
                }
              }
            }

            normalizeInnerPageTransform('setBookInnerSize');
            hideOrientationChangeLoader();
            
            isOrientationChangeResize = false;
          }, 300);
        });
      };

      window.addEventListener('resize', function (e) {
        if (is_safari() && !is_mobile()) {
          self.svgWrapperSetSizes();
        }
        
        if (document.querySelector('.stf__marcinBookShadow')) {
          document.querySelector('.stf__marcinBookShadow').style.opacity = '0.0';
        }

        if (is_mobile()) {
          HELPER.setThumbSizeAndCountMobile();
        }

        if (pageFlip && pageFlip.isBookZoomed()) {
          elements.navScaler.hidden = false;
          self.scale(0);
        }

        isVideoFullscreen = APP.EMBED.isPlayerFullscreen() || APP.EMBED.isMediaViewerFullscreen();

        if (!document.fullscreen) {
          if (!isVideoFullscreen) {
            for (let i of pagesToLoad) {
              if (pages[i]) {
                // pages[i].removeHotspots();
                const hotspotsContainer = pages[i].getHotspotsContainer();
                if (hotspotsContainer) {
                  hotspotsContainer.style.display = 'none';
                  hotspotsContainer.style.opacity = '0';
                  hotspotsContainer.style.visibility = 'hidden';
                }

                setTimeout(() => {
                  if (pages[i] && pages[i].getPage().style.display === 'block') {
                    pages[i].adjustHotspotsSize(pageWidth, pageHeight);
                  } 
                }, 300);
              }
            }
          } else {
            isVideoFullscreen = false;
          }
        }

        if (elements.pagesRanger) {
          elements.pagesRanger.updateMax(getCorrectNumberOfIndexes());
        }

        setTimeout(() => {
          setBookInnerSize(e, 'resize');
          self.customizeBookBackground();
          updateArrows();

          if (is_horisontal()) {
            if (is_mobile()) {
              self.changeArrows(false);
            } else {
              elements.navNextPageAbsolute.hidden = true;
              elements.navPrevPageAbsolute.hidden = true;
            }

            if (settings.showSinglePageMode || (is_mobile() && !mobile_test_horizontal())) {
              if (pageFlip && pageFlip.getOrientation() !== 'portrait') {
                pageFlip.updateOrientation('portrait');
              }

              self.connectObserver();
            } else {
              if (pageFlip && pageFlip.getOrientation() !== 'landscape') {
                pageFlip.updateOrientation('landscape');
              }
              
              if (observer) {
                observer.disconnect();
              }
            }
          }

          if (is_verticalOld()) {
            // elements.navNextPage.hidden = true;
            // elements.navPrevPage.hidden = true;
            if (pageFlip && pageFlip.getOrientation() !== 'portrait') {
              pageFlip.updateOrientation('portrait');
            }
            self.connectObserver();
          }
          var prevPage = currentPage; // backup currentPage
          currentPage = -1; // set minused currentPage
          self.setPage(prevPage, 'resize'); // restore currentPage
          setTimeout(() => {
            document.querySelector('.stf__marcinBookShadow') && (document.querySelector('.stf__marcinBookShadow').style.opacity = '1.0');

            loadGalleryImagesByBookSize();
          }, 400);
        }, 200);
      }); // yepp, so simple

      setBookInnerSize();

      // Lets fill all other elements
      if (settings.nextPrevButtons) {
        self.render__setArrows();
        self.render__setArrowsArea();
      }

      elements.inner.appendChild(elements.navAreaPrevPage);
      elements.inner.appendChild(elements.navAreaNextPage);

      elements.inner.appendChild(elements.navPrevPage);
      elements.inner.appendChild(elements.innerChild);
      elements.inner.appendChild(elements.navNextPage);
      elements.bookWrapper.appendChild(elements.inner);
      bookNode.appendChild(elements.bookWrapper);

      //@todo
      setMouseEvents();
    });

    return true;
  };

  const updateArrowsPositions = () => {
    self.setArrowsOffsetByThickness();
  };

  const loadGalleryImagesByBookSize = () => {
    if (Array.isArray(pages) && pages.length > 0) {
      pages.forEach(page => {
        if (Pages[page + 1]) Pages[page + 1].loadGalleryImagesByBookSize();
      });
    } else {
      const currentPages = self.getCurrentPageForLogs();

      if (currentPages && currentPages.length > 0) {
        currentPages.forEach(page => {
          if (Pages[page + 1]) Pages[page + 1].loadGalleryImagesByBookSize();
        });
      }
    }
  };

  const waitForHelperInit = () => {
    return new Promise((resolve, reject) => {
      const waitForHelperInterval = setInterval(() => {
        console_log('HELPER CHECK');
        if (window.HELPER && HELPER && HELPER.isLocalStorageAvailable) {
          clearInterval(waitForHelperInterval);
          resolve();
        }
      }, 100);
    })
  }

  this.loadBook = async () => {
    if (!window.HELPER || !HELPER || !HELPER.isLocalStorageAvailable) {
      await waitForHelperInit();
    }

    if ('error' in model) {
      console_log('CLASSIC BOOK LOAD');
      return loadBookClassic();
    }

    bookModel = model;

    if (customizemode && rtlChange) {
      assignCustomizeSettings();
    }

    if (bookModel.is_converted) {
      if (customizemode) {
        window.parent.postMessage('enableHotspots', '*');
      }

      self.customizeBook();
      self.render();
    } else {
      if (isCanva) {
        self.render__createBookLoader();

        setTimeout(() => {
          window.location.reload();
        }, 3000);

        return;
      }

      if (!isCanva && customizemode) {
        updateConvertingStatusForHotspots();
      }

      self.customizeBook();
      self.render();
    }
  }

  if (embedmode) {
    this.tryLoadBookEmbeded(0);
  } else {
    this.loadBook();
  }

  this.onUserFoldState = function () {
    const iframes = document.querySelectorAll('iframe');
    if (iframes && iframes.length) {
      iframes.forEach(item => item.style.pointerEvents = 'none');
    }
  }

  this.onReadState = function () {
    const iframes = document.querySelectorAll('iframe');
    if (iframes && iframes.length) {
      iframes.forEach(item => item.style.pointerEvents = null);
    }
  }

  // Methods
  // pages
  var currentPage = 0, // Protected var, contain current page
    prevPage = 0, // Protected var, contain previous page
    pageFlip = null,
    Pages = {};

  /*
   * Metoda okresla czy dla danego flipa pokazywac czy nie Menu / guzik Fullscreen
   *
   * @returns {undefined}
   */
  const isSmallDesktopMode = function () {
    const windowHeight = window.innerHeight;
    const windowWidth = window.innerWidth;

    if (customizemode || (embedmode && !transparent)) return false;

    if (windowHeight < 325 || windowWidth < 350 || windowWidth <= windowHeight) return true;
    if (transparent && !__is_full_screen) return true;

    return false;
  };

  const hideFSButtonIfNeeded = function () {
    if (fsAnimationStarted) return;

    clearTimeout(APP.fsShowTimeout);
    clearTimeout(fsAnimationTimeout);
    fsAnimationStarted = false;

    if (elements.fullscreenModalElements && elements.fullscreenModalElements.modal) {
      const isHovered = elements.fullscreenModalElements.modal.matches(':hover');
  
      if (!is_mobile() && isHovered) {
        elements.fullscreenModalElements.modal.addEventListener('mouseleave', () => {
          elements.fullscreenModalElements.modal.classList.remove('show');
        }, { once: true });
  
        return;
      }
  
      elements.fullscreenModalElements.modal.classList.remove('show');
    }
  };

  const showFSButtonIfNeeded = function () {
    if (!bookModel.settings.show_fullscreen_button) return;
    
    const isEmbed = embedmode || is_embed().embed;

    if (elements.fullscreenModalElements && elements.fullscreenModalElements.modal) {
      if (isEmbed && transparent && !customizemode) {
        elements.fullscreenModalElements.modal.classList.add('show');
      } else if (is_mobile()) {
        if (isEmbed && !customizemode) {
          elements.fullscreenModalElements.modal.classList.add('show');
        }
      } else {
        if (!__is_full_screen && isSmallDesktopMode()) {
          elements.fullscreenModalElements.modal.classList.add('show');
        }
      }
    }

    if (isEmbed) {
      hideMenuIfNeeded();
    }
  };

  const isLayoutBlocked = () => {
    return is_mobile() && window.blockLayout === '1';
  };

  const hideMenuIfNeeded = function (forced = false) {
    const header = document.getElementById('header');
    const footer = document.getElementById('footer');

    if (!forced) {
      forced = isLayoutBlocked();
    }

    const shouldHideLayout = (transparent || forced) && !__is_full_screen; // isSmallDesktopMode() || 

    if (footer && shouldHideLayout) { 
      footer.classList.add('hide');
    }

    if (header && shouldHideLayout) { 
      header.classList.add('hide');
    }
  };

  this.bookTransparentHandler = () => {
    hideInnerBookButtons();
    showInnerBookButtons();
  };

  const hideInnerBookButtons = () => {
    if (elements.search) {
      elements.search.hidden = !bookModel.settings.show_search_button || !!transparent;
    }

    if (elements.noteBtnBottom) {
      elements.noteBtnBottom.hidden = !bookModel.settings.show_note_button || !!transparent;
    }
  };

  const showInnerBookButtons = () => {
    if (elements.search) {
      elements.search.hidden = !(bookModel.settings.show_search_button && (!transparent || __is_full_screen));
    }

    if (elements.noteBtnBottom) {
      // elements.noteBtnBottom.hidden = !(bookModel.settings.show_note_button && (!transparent || __is_full_screen));
      APP.Note.showNotesByDependsOnWidth();
    }
  };

  this.getInteractivitiesByType = function (type, optCondition = null, strict = false) {
    const interactivites = this.getModel().interactivites;
    let filteredInteractivities = [];

    if (!Array.isArray(interactivites) || interactivites.length <= 0) {
      return [];
    }

    interactivites.forEach(page => {
      if (!Array.isArray(page.areas) || page.areas.length <= 0) {
        return;
      }

      const filteredByType = page.areas.filter(item => {
        if (item.type != type) {
          return false;
        }

        if (!optCondition) {
          return item;
        }

        if (!item.data.hasOwnProperty(Object.keys(optCondition)[0]) && !strict) {
          return item;
        }

        if (Array.isArray(Object.values(optCondition)[0])) {
          if (Object.values(optCondition)[0].indexOf(item.data[Object.keys(optCondition)[0]]) !== -1) {
            return item;
          }
        } else {
          if (item.data[Object.keys(optCondition)[0]] == Object.values(optCondition)[0]) {
            return item;
          }
        }

        return null;
      });

      filteredInteractivities = filteredInteractivities.concat(filteredByType);
    })

    return filteredInteractivities;
  };

  const createButton = (config) => {
    elements[config[0]] = document.createElement('button');
    elements[config[0]].className = config[1].className;
    elements[config[0]].name = config[1].description;
    elements[config[0]].ariaLabel = config[1].description;
    elements[config[0]].title = config[1].description;
    elements[config[0]].hidden = config[1].hidden;

    if (!config[1].hasOwnProperty('icon') || !config[1].icon.hasOwnProperty('type') || config[1].icon.type === 'img') {
      elements[config[0]].appendChild(document.createElement('div'));
    } else {
      elements[config[0]].dataset.type = config[1].icon.type;
      elements[config[0]].innerHTML = config[1].icon.elem;
      elements[config[0]].classList.add(config[1].icon.type);

      if (config[1].icon.shapes) {
        elements[config[0]].className += ` ${config[1].icon.shapes}`
      }
    }

    if (!config[1].hasOwnProperty('style')) {
      return;
    }

    Object.entries(config[1].style).forEach(styleConfig => elements[config[0]].style[styleConfig[0]] = styleConfig[1]);
  };

  const defineMenuButtons = function () {
    const buttonsConfig = {
      print: {
        className: 'Book__print',
        description: APP._t('Elements Description Print'),
        hidden: !bookModel.settings.show_print_button,
        icon: {
          type: BUTTON_ICON_TYPE,
          shapes: 'path circle',
          elem: HELPER.buttonsIcons.printIcon
        },
      },
      audioIcon: {
        className: 'Book__audio',
        description: APP._t('Elements Description Audio Off'),
        hidden: !bookModel.settings.show_audio_button,
        icon: {
          type: BUTTON_ICON_TYPE,
          shapes: 'path',
          elem: HELPER.buttonsIcons.soundOnIcon
        },
      },
      download: {
        className: 'Book__download',
        description: APP._t('Elements Description Download'),
        hidden: !bookModel.settings.show_download_button,
        icon: {
          type: BUTTON_ICON_TYPE,
          shapes: 'line polyline',
          elem: HELPER.buttonsIcons.downloadIcon
        },
      },
      share: {
        className: 'Book__share',
        description: APP._t('Elements Description Share'),
        hidden: !bookModel.settings.show_share_button,
        icon: {
          type: BUTTON_ICON_TYPE,
          shapes: 'line circle ellipse',
          elem: HELPER.buttonsIcons.shareIcon
        },
      },
      search: {
        className: 'search__button',
        description: APP._t('Elements Description Search'),
        hidden: !bookModel.settings.show_search_button || !!transparent,
        icon: {
          type: BUTTON_ICON_TYPE,
          shapes: 'path',
          elem: HELPER.buttonsIcons.searchIcon
        },
      },
      navTOC: {
        className: 'Book__navTOC',
        description: APP._t('Elements Description Table Of Content'),
        hidden: !(
          bookModel.settings.show_toc_button &&
          Array.isArray(APP.isStringified(bookModel.toc)) &&
          !!APP.isStringified(bookModel.toc).length
        ),
        icon: {
          type: BUTTON_ICON_TYPE,
          shapes: 'path',
          elem: HELPER.buttonsIcons.tocIcon
        },
      },
      navThumbs: {
        className: 'Book__navThumbs',
        description: APP._t('Elements Description Thumbnails'),
        hidden: !bookModel.settings.show_thumbnails_button,
        style: {
          margin: '0px'
        },
        icon: {
          type: BUTTON_ICON_TYPE,
          shapes: 'path',
          elem: HELPER.buttonsIcons.thumbnailIcon
        },
      },
      previewBack: {
        className: 'Book__navBack',
        description: APP._t('Elements Description Preview Back'),
        hidden: !!!mobilepreview,
        icon: {
          type: 'img',
          elem: HELPER.buttonsIcons.previewBackIcon
        },
      },
      navFullScreen: {
        className: 'Book__navFullScreen',
        description: APP._t('Elements Description Fullscreen'),
        hidden: !bookModel.settings.show_fullscreen_button || !!transparent,
        icon: {
          type: BUTTON_ICON_TYPE,
          shapes: 'path',
          elem: HELPER.buttonsIcons.fullscreenOnIcon
        },
      },
      zoomIn: {
        className: 'Book__zoom',
        description: APP._t('Elements Description Zoom In'),
        hidden: !bookModel.settings.show_zoom_button,
        icon: {
          type: BUTTON_ICON_TYPE,
          shapes: 'path',
          elem: HELPER.buttonsIcons.zoomInIcon
        },
      },
      wishlist: {
        className: 'Book__wishlist empty-wishlist',
        description: APP._t('Elements Description Wishlist'),
        hidden: self.getInteractivitiesByType('product-info', { 'show_wishlist': [1, true] }, false).length <= 0,
        icon: {
          type: BUTTON_ICON_TYPE,
          shapes: 'path',
          elem: HELPER.buttonsIcons.wishlistIcon
        },
      },
      autoFlip: {
        className: 'Book__autoFlip',
        description: APP._t('Elements Description Auto Flip'),
        hidden:  !bookModel.settings.auto_flip_enabled,
        icon: {
          type: BUTTON_ICON_TYPE,
          shapes: 'path',
          elem: HELPER.buttonsIcons.autoFlipPlayIcon
        },
      },
    };

    Object.entries(buttonsConfig).forEach(createButton);
    
    ///tu slider zoom
    self.setZoomSlider();

    if (is_mobile()) return;

    createButton(['note', {
      className: 'Book__note',
      description: APP._t('Elements Description Notes'),
      hidden: !bookModel.settings.show_note_button,
      icon: {
        type: BUTTON_ICON_TYPE,
        shapes: 'path',
        elem: HELPER.buttonsIcons.noteOutlineIcon
      }
    }]);

    elements.note.addEventListener('click', () => {
      if (!APP.Note) return;

      APP.Note.createNote();
    });
  };

  const defineInnerChildButtons = () => {
    if (!elements.innerChild) return;

    if (!is_mobile()) {
      elements.search.style.right = `-38px`;
      elements.search.className = `search__button ${BUTTON_ICON_TYPE}`;
      elements.innerChild.appendChild(elements.search);

      if (bookModel.settings.show_note_button) {
        APP.loadComponent('Notes', () => APP.Note.init(bookId));
  
        createButton(['noteBtnBottom', {
          className: 'note__button',
          description: APP._t('Elements Description Add Note'),
          hidden: !bookModel.settings.show_note_button || !!transparent,
          style: {
            right: '-38px'
          },
          icon: {
            type: BUTTON_ICON_TYPE,
            shapes: 'path',
            elem: HELPER.buttonsIcons.noteFillIcon
          },
        }]);
  
        elements.noteBtnBottom.addEventListener('click', () => {
          if (!APP.Note) return;
  
          APP.Note.createNote();
        });
  
        elements.innerChild.appendChild(elements.noteBtnBottom);
      }
    } else {
      elements.search.className = `mobile-search__button ${BUTTON_ICON_TYPE}`;
    }
  };

  const defineHeaderLogo = function () {
    elements.headerLogo = document.createElement('a');
    elements.headerLogo.className = 'header__logo';

    if (bookModel.settings.toplogo_image) {
      const topLogoImageUrl = getTopLogoImageUrl();

      if (topLogoImageUrl) {
        elements.headerLogo.href = topLogoImageUrl;
      } else {
        elements.headerLogo.removeAttribute('href');
      }

      if (bookModel.settings.toplogo_image.includes('.svg')) {
        elements.headerLogo.target = '_blank';
        elements.headerLogo.style.cursor = 'pointer';
        elements.headerLogo.style.zIndex = 99;
        elements.headerLogo.style.color = 'inherit';

        elements.headerLogo.innerHTML = (is_mobile() || is_small_embed_not_mobile())
          ? HELPER.headerLogoDefaultMobile
          : HELPER.headerLogoDefault;

        return;
      }

      elements.headerLogo.target = '_blank';
      elements.headerLogoImg = document.createElement('img');
      elements.headerLogoImg.src = bookModel.settings.toplogo_image;

      let tryCounter = 0;

      elements.headerLogoImg.addEventListener('error', () => {
        if (
          tryCounter >= 1 || 
          !bookModel.settings.toplogo_image || 
          bookModel.settings.toplogo_image.indexOf('/publishers//') === -1
        ) {
          return;
        }

        tryCounter++;
        elements.headerLogoImg.src = bookModel.settings.toplogo_image.replace('/publishers//', '/publishers/');
      });

      elements.headerLogoImg.alt = bookModel.name;
      elements.headerLogo.appendChild(elements.headerLogoImg);
    }
  };

  const getTopLogoImageUrl = () => {
    const isUrlValid = bookModel.settings.toplogo_image_url && bookModel.settings.toplogo_image_url.length > 10;

    if (!bookModel.settings.toplogo_image) {
      return null;
    }

    if (bookModel.settings.toplogo_image.includes('.svg')) {
      if (isUrlValid) {
        return bookModel.settings.toplogo_image_url;
      } else {
        return APP.PATH_FORLOGODEF + `#${bookModel.publisher_id}_${bookModel.bookId || bookModel.id}`;
      }
    }

    if (isUrlValid) {
      return bookModel.settings.toplogo_image_url;
    }

    return null;
  };

  const getCorrectNumberOfIndexes = function () {
    let numberOfIndexes = Number(bookModel.pages);

    if (
      is_horisontal() &&
      !settings.showSinglePageMode &&
      !(is_mobile() && !mobile_test_horizontal())
    ) {
      numberOfIndexes = Math.floor(Number(bookModel.pages) / 2) + 1;
    }

    return numberOfIndexes;
  };

  const setCorrectRangerColorWithPageNum = function (currentPage, id) {
    const numberOfIndexes = getCorrectNumberOfIndexes();
    let curIndex = currentPage;

    if (
      is_horisontal() &&
      !settings.showSinglePageMode &&
      !(is_mobile() && !mobile_test_horizontal())
    ) {
      curIndex = Math.floor(currentPage / 2) + 1;
    }

    self.setCorrectRangerColor(curIndex, id, numberOfIndexes);
  };

  this.setCorrectRangerColor = function (v, id, maxVal) {
    var colorLeft = '#0f0';
    var colorLeftAlpha = '#0f0';
    let x = document.getElementById(id);

    if (x) {
      if (bookModel.settings.bottom_menu_color_mode) {
        colorLeft =
          bookModel.settings.bottom_menu_color_mode == COLOR_THEME.DARK
            ? 'rgba(255,255,255,1.0)'
            : 'rgba(68,68,68,1.0)';
        colorLeftAlpha =
          bookModel.settings.bottom_menu_color_mode == COLOR_THEME.DARK
            ? 'rgba(255,255,255,0.5)'
            : 'rgba(68,68,68,.35)';
        x.classList.remove(COLOR_THEME.DARK);
        x.classList.remove(COLOR_THEME.LIGHT);
        x.classList.add(bookModel.settings.bottom_menu_color_mode);
      }

      var value = ((v - 1) / (Number(maxVal) - 1)) * 100;

      x.style.background = `linear-gradient(to right, ${colorLeft} 0%, ${colorLeft} ${value}%, ${colorLeftAlpha} ${value}%, ${colorLeftAlpha} 100%)`;
    }
  };

  this.setZoomSlider = function () {
    if (settings.Scaller) {
      // scaler (aka zoomer)
      elements.navScaler = document.createElement('div'); // Scaller parent
      elements.navScaler.className = 'Book__navScaler';
      elements.navScaler.hidden = true;
      elements.navScaler.navScalerP = document.createElement('button'); // Scaller btn -
      elements.navScaler.navScalerP.className = 'Book__navScaler-left';
      elements.navScaler.navScalerM = document.createElement('button'); // Scaller btn +
      elements.navScaler.navScalerM.className = 'Book__navScaler-right';

      if (BUTTON_ICON_TYPE === BUTTON_ICON_TYPES.INLINE) {
        elements.navScaler.navScalerP.dataset.type = BUTTON_ICON_TYPE;
        elements.navScaler.navScalerM.dataset.type = BUTTON_ICON_TYPE;
        
        elements.navScaler.navScalerP.classList.add(BUTTON_ICON_TYPE);
        elements.navScaler.navScalerM.classList.add(BUTTON_ICON_TYPE);

        elements.navScaler.navScalerP.innerHTML = HELPER.buttonsIcons.zoomOutIcon;
        elements.navScaler.navScalerM.innerHTML = HELPER.buttonsIcons.zoomInIcon;
      }

      APP.loadComponent('Ranger', function () {
        // page scale ranger
        elements.navScaler.navScalerR = new Ranger({
          min: settings.MIN_SCALE,
          max: 5 * settings.MAX_SCALE,
          id: 'zoomranger',
          value: scale,
          showInt: false,
          theme: bookModel.settings.bottom_menu_color_mode,
          onchange: function (v) {
            self.setCorrectRangerColor(v, 'zoomranger', 5 * settings.MAX_SCALE);

            const pages = self.getCurrentPageForLogs();

            if (Array.isArray(pages) && pages.length > 0) {
              pages.forEach(page => {
                if (Pages[page + 1]) Pages[page + 1].loadGalleryImagesByScaleSize(1 + v / 20);
              });
            } else {
              const page = self.getCurrentPage();

              if (page) {
                if (Pages[page]) Pages[page].loadGalleryImagesByScaleSize(1 + v / 20);
              }
            }
          },
          oninput: function (v) {
            if (v != scale) {
              // Book scale ranger just set scale for pages
              self.scale(v / 5);
            }
            self.setCorrectRangerColor(v, 'zoomranger', 5 * settings.MAX_SCALE);
          },
        });
        requestAnimationFrame(function () {
          elements.navScaler.appendChild(elements.navScaler.navScalerP);
          elements.navScaler.appendChild(elements.navScaler.navScalerR.render().parent);
          elements.navScaler.appendChild(elements.navScaler.navScalerM);
          self.scale(settings.DEFAULT_SCALE);
        });
      });
      elements.navScaler.navScalerP.addEventListener('click', self.scaleOutStep);
      elements.navScaler.navScalerM.addEventListener('click', self.scaleInStep);
    }
  };

  const setMenuFooterRanger = () => {
    if (!settings.pagesRanger) return;

    const numberOfIndexes = getCorrectNumberOfIndexes();
    
    elements.pagesRangerContainer = document.createElement('div');
    elements.pagesRangerContainer.className = 'Ranger__container';

    if (startPage) {
      startPage =
        is_horisontal() &&
        !settings.showSinglePageMode &&
        !(is_mobile() && !mobile_test_horizontal())
          ? Math.floor(startPage / 2) + 1
          : startPage;
    }

    APP.loadComponent('Ranger', function () {
      elements.pagesRanger = new Ranger({
        min: 1,
        id: 'pageranger',
        max: Number(numberOfIndexes),
        showInt: false,
        className: 'Book__pagesRanger',
        timeout: 100,
        value: startPage ? startPage : 1,
        onchange: function (v) {
          const newIndex = v;
          let newPage =
            is_horisontal() &&
            !settings.showSinglePageMode &&
            !(is_mobile() && !mobile_test_horizontal())
              ? (newIndex - 1) * 2 + 0
              : newIndex;
              
          if (forceOnePage) {
            newPage = newIndex;
          }
          
          if (is_mobile() && !mobile_test_horizontal()) {
            newPage = newIndex;
          }

          newPage = newPage > Number(bookModel.pages) ? Number(bookModel.pages) : newPage;
          newPage = newPage < 1 ? 1 : newPage;

          self.setPage(newPage, 'ranger');
          self.setFooterPage(newPage);
        },
        oninput: function (v) {
          const newIndex = v;
          let newPage =
            is_horisontal() &&
            !settings.showSinglePageMode &&
            !(is_mobile() && !mobile_test_horizontal())
              ? (newIndex - 1) * 2 + 0
              : newIndex;

          newPage = newPage > Number(bookModel.pages) ? Number(bookModel.pages) : newPage;
          newPage = newPage < 1 ? 1 : newPage;

          setCorrectRangerColorWithPageNum(newPage, 'pageranger');
        },
      });

      elements.pagesRangerLeftNum = document.createElement('div');
      elements.pagesRangerRightNum = document.createElement('div');
      elements.pagesRangerLeftNum.className = 'Ranger_page-num current-page-num';
      elements.pagesRangerRightNum.className = 'Ranger_page-num total-pages-num';
      elements.pagesRangerRightNum.innerText = bookModel.pages;

      elements.footerPageNumber = document.createElement('div');
      elements.footerPageNumber.className = 'footer__page-num';
      elements.footerPageNumber.insertAdjacentHTML(
        'afterbegin',
        `<span>/&nbsp${bookModel.pages}</span>`,
      );
      elements.footerPageNumber.ariaLabel = APP._t('Elements Description Page Number');
      elements.footerPageNumber.title = APP._t('Elements Description Page Number');
      elements.footerPageNumber.hidden = !bookModel.settings.show_page_number;

      elements.pageNumInput = document.createElement('input');
      elements.pageNumInput.setAttribute('type', 'text');
      if (is_mobile()) {
        elements.pageNumInput.setAttribute('name', 'page_num');
        elements.pageNumInput.setAttribute('id', 'page_num');
        elements.pageNumInput.setAttribute('pattern', '[0-9]*');
        elements.pageNumInput.setAttribute('autocomplete', 'off');
        elements.pageNumInput.setAttribute('autocorrect', 'off');
        elements.pageNumInput.setAttribute('autocapitalize', 'off');
        elements.pageNumInput.setAttribute('inputmode', 'decimal');
      }

      elements.pageNumInput.className = 'page-num-input';
      elements.pageNumInput.ariaLabel = APP._t('Elements Description Page Number');
      elements.pageNumInput.title = APP._t('Elements Description Page Number');
      elements.footerPageNumber.prepend(elements.pageNumInput);
      requestAnimationFrame(function () {
        elements.pagesRangerContainer.appendChild(elements.pagesRangerLeftNum);
        elements.pagesRangerContainer.appendChild(elements.pagesRanger.render().parent);
        elements.pagesRangerContainer.appendChild(elements.pagesRangerRightNum);
        elements.footerLeft.appendChild(elements.footerPageNumber);
        if (!is_mobile()) {
          elements.footerCenter.appendChild(elements.pagesRangerContainer);
          elements.pagesRanger.getNode('parent').hidden =
            !bookModel.settings.show_page_slider || parseInt(bookModel.pages) === 1;
        }
      });
      let enterPressed = false;
      elements.pageNumInput.addEventListener('keydown', function (e) {
        if (e.key === 'Enter') {
          enterPressed = true;
          elements.pageNumInput.blur();
          let page = Number(this.value.trim());

          if (page && page > 0 && page <= bookModel.pages) {
            if (
              ((page === 1 || page === bookModel.pages) && page !== currentPage) ||
              (is_horisontal() &&
                page !== 1 &&
                page !== bookModel.pages &&
                page % 2 == 0 &&
                page + 1 !== currentPage) ||
              ((is_vertical() ||
                (is_horisontal() &&
                  page !== 1 &&
                  page !== bookModel.pages &&
                  page % 2 !== 0)) &&
                page !== currentPage)
            ) {
              self.setPage(page, 'footerInput');
            }
          } else {
            self.setFooterPage(currentPage);
          }
        }
      });

      elements.pageNumInput.addEventListener('input', function (e) {
        const MAX_VALUE_LENGTH = `${bookModel.pages}`.length;
        const valueLength = Math.max(1, Math.min(e.target.value.length, MAX_VALUE_LENGTH));

        elements.pageNumInput.style.width = `${valueLength * (is_mobile() ? 15 : 14)}px`;
      });

      elements.pageNumInput.addEventListener('focusout', function (e) {
        if (is_mobile()) {
          enterPressed = true;
          elements.pageNumInput.blur();
          let page = Number(this.value.trim());

          if (page && page > 0 && page <= bookModel.pages) {
            if (
              ((page === 1 || page === bookModel.pages) && page !== currentPage) ||
              (is_horisontal() &&
                page !== 1 &&
                page !== bookModel.pages &&
                page % 2 == 0 &&
                page + 1 !== currentPage) ||
              ((is_vertical() ||
                (is_horisontal() &&
                  page !== 1 &&
                  page !== bookModel.pages &&
                  page % 2 !== 0)) &&
                page !== currentPage)
            ) {
              self.setPage(page, 'footerInput');
            }
          } else {
            self.setFooterPage(currentPage);
          }
          let width = page.toString().length * 15;
          elements.pageNumInput.style.width = `${width}px`;

          return;
        }
        if (!enterPressed) {
          self.setFooterPage(currentPage);
        }
        enterPressed = false;
      });

      elements.pageNumInput.addEventListener('focus', function (e) {
        e.target.value = '';
        if (pageFlip && pageFlip.isBookZoomed()) {
          self.scale(0);
        }
      });
    });
  };

  const setMenuFooter = () => {
    elements.footer = document.getElementById('footer');

    elements.footerLeft = document.createElement('div');
    elements.footerLeft.className = 'footer__left';

    elements.footerCenter = document.createElement('div');
    elements.footerCenter.className = 'footer__center';

    elements.footerRight = document.createElement('div');
    elements.footerRight.className = 'footer__right';

    elements.footerBorder = document.createElement('div');
    elements.footerBorder.className = 'footer__border';

    elements.footer.appendChild(elements.footerLeft);
    elements.footer.appendChild(elements.footerCenter);
    elements.footer.appendChild(elements.footerRight);

    setMenuFooterRanger();

    elements.footerIcons = document.createElement('div');
    elements.footerIcons.className = 'footer__icons';

    if (!is_mobile()) {
      elements.footerIcons.appendChild(elements.navFullScreen);
      elements.footerRight.appendChild(elements.footerIcons);
      elements.footer.appendChild(elements.footerBorder);
    } else {
      elements.footerRight.appendChild(elements.headerLogo);
      elements.footerRight.style.cssText = `width: 50%; align-content: flex-end;
      justify-content: flex-end;
      align-items: center;
      display: flex; z-index: 20;`;

      elements.navTOC.className = `Book__navTOC Book__navTOC-mobile ${BUTTON_ICON_TYPE}`;
      elements.footerLeft.appendChild(elements.navTOC);
      elements.footerLeft.style.width = '45%';

      elements.autoFlip.classList.add('Book_autoFlip-mobile')
      elements.footerLeft.appendChild(elements.autoFlip);

      elements.footer.style.zIndex = 20;
      elements.footer.style.padding = '0 7px';
      elements.footer.style.fontSize = '17px';
      elements.footer.style.backgroundColor = `#${this.customizeBookBackgroundColor__getCorrectAvgHex()}E6`;

      elements.footerBorder.style.opacity = '0.1';
      elements.footer.appendChild(elements.footerBorder);

      if (isLayoutBlocked()) {
        elements.footer.classList.add('hide');
      }
    }
  };

  const defineFullscreenModal = () => {
    if (elements.fullscreenModalElements) {
      return elements.fullscreenModalElements;
    }

    elements.fullscreenModalElements = {};

    if (!bookModel.settings.show_fullscreen_button) return;

    const ACTIVE_CLASSNAME = 'show';

    const handleFullscreenModalClick = () => {
      elements.fullscreenModalElements.modal.classList.remove(ACTIVE_CLASSNAME);
      fullScreenToggle();
    };

    const fullscreenModal = document.createElement('div');
    const fullscreenModalText = document.createElement('span');

    fullscreenModal.className = 'fs-modal';

    fullscreenModalText.textContent = APP._t('Elements Description Fullscreen');
    fullscreenModal.innerHTML = HELPER.buttonsIcons.fullscreenModalIcon;
    fullscreenModal.addEventListener('click', handleFullscreenModalClick);

    fullscreenModal.innerHTML = HELPER.buttonsIcons.fullscreenModalIcon;
    fullscreenModal.appendChild(fullscreenModalText);

    document.body.appendChild(fullscreenModal);

    elements.fullscreenModalElements = {
      modal: fullscreenModal
    };
  };

  const defineAbsoluteFullscreenButton = () => {
    if (!bookModel.settings.show_fullscreen_button) return;
    // if (!is_embed().embed || !transparent) return;

    const handleResize = () => {
      setAbsoluteFullscreenButtonClassName();

      if (shouldShowAbsoluteFullscreenButton()) {
        showAbsoluteFullscreenButton();
      } else {
        hideAbsoluteFullscreenButton();
      }
    };

    elements.absoluteFullscreen = document.createElement('button');
    elements.absoluteFullscreen.className = 'absolute-fs-btn path';
    elements.absoluteFullscreen.title = APP._t('Elements Description Fullscreen');
    elements.absoluteFullscreen.dataset.type = BUTTON_ICON_TYPES.INLINE;
    elements.absoluteFullscreen.innerHTML = HELPER.buttonsIcons.fullscreenOnIcon;
    elements.absoluteFullscreen.hidden = true;

    elements.absoluteFullscreen.addEventListener('click', fullScreenToggle);

    elements.innerChild.appendChild(elements.absoluteFullscreen);

    window.addEventListener('resize', handleResize);
    window.addEventListener('orientationchange', HELPER.debounce(setAbsoluteFullscreenButtonClassName, 600));
  };

  const setAbsoluteFullscreenButtonClassName = () => {
    if (!elements.innerChild || !elements.absoluteFullscreen) return;

    const BUTTON_WIDTH = 35;
    const WINDOW_OFFSET = 60;
    const BUTTON_ABSOLUTE_CLASSNAME = 'absolute';
    const bookWidth = elements.innerChild.getBoundingClientRect().width;
    const windowWidth = window.innerWidth;

    if (bookWidth + BUTTON_WIDTH + WINDOW_OFFSET <= windowWidth) {
      elements.absoluteFullscreen.classList.remove(BUTTON_ABSOLUTE_CLASSNAME);

      if (elements.absoluteFullscreen.parentNode !== elements.innerChild) {
        elements.innerChild.appendChild(elements.absoluteFullscreen);
      }

      return;
    }

    if (elements.absoluteFullscreen.parentNode !== root) {
      root.appendChild(elements.absoluteFullscreen);
    }

    elements.absoluteFullscreen.classList.add(BUTTON_ABSOLUTE_CLASSNAME);
  };

  const shouldShowAbsoluteFullscreenButton = () => {
    const isSmallEmbed = is_embed().embed && window.innerWidth <= 350;
    const isTransparentEmbed = is_embed().embed && transparent;
    const shouldShowFullscreenButton = isSmallEmbed || isTransparentEmbed || window.innerWidth <= 350 || isLayoutBlocked();

    return shouldShowFullscreenButton && scale <= 1 && !__is_full_screen;
  };

  const showAbsoluteFullscreenButton = () => {
    if (!elements.absoluteFullscreen) return;
    
    setAbsoluteFullscreenButtonClassName();
    elements.absoluteFullscreen.removeAttribute('hidden');
  };

  const hideAbsoluteFullscreenButton = () => {
    if (!elements.absoluteFullscreen) return;

    elements.absoluteFullscreen.setAttribute('hidden', '');
  };

  const calculateOffsetAndCenterTitle = () => {
    const offsetLeft = elements.headerLeft.clientWidth;
    const offsetRight = elements.headerRight.clientWidth;
    const offsetBetweenElements = 30;

    elements.headerCenter.style.width = `calc(100% - ${Math.max(offsetLeft, offsetRight) * 2}px - ${offsetBetweenElements}px)`;
    elements.headerCenter.style.marginRight = `${offsetLeft - offsetRight}px`;
  };

  const setMenuHeader = () => {
    elements.header = document.getElementById('header');

    elements.headerLeft = document.createElement('div');
    elements.headerLeft.className = 'header__left';

    elements.headerCenter = document.createElement('div');
    elements.headerCenter.className = 'header__center';

    elements.headerRight = document.createElement('div');
    elements.headerRight.className = 'header__right';

    elements.headerBorder = document.createElement('div');
    elements.headerBorder.className = 'header__border';

    elements.headerBookName = document.createElement('h1');
    elements.headerBookName.className = 'header__book-name';
    elements.headerBookName.innerText = bookModel.name;
    elements.headerBookName.ariaLabel = APP._t('Elements Description Book Name');
    elements.headerBookName.title = APP._t('Elements Description Book Name');
    elements.headerBookName.hidden = !bookModel.settings.show_title; 

    if (!is_mobile()) {
      elements.headerRight.className = 'header__right absolute';
      elements.topIcons = document.createElement('div');
      elements.topIcons.className = 'header__icons';
      elements.topIcons.appendChild(elements.navTOC);
      elements.topIcons.appendChild(elements.navThumbs);
      elements.topIcons.appendChild(elements.share);
      elements.topIcons.appendChild(elements.download);
      elements.topIcons.appendChild(elements.audioIcon);
      elements.topIcons.appendChild(elements.print);
      elements.topIcons.appendChild(elements.navScaler);
      elements.topIcons.appendChild(elements.zoomIn);
      elements.topIcons.appendChild(elements.note);
      elements.topIcons.appendChild(elements.autoFlip);
      elements.topIcons.appendChild(elements.wishlist);

      elements.headerLeft.appendChild(elements.headerBookName);
      elements.headerCenter.appendChild(elements.topIcons);
      elements.headerRight.appendChild(elements.headerLogo);
    } else {
      elements.topLeftIcons = document.createElement('div');
      elements.topLeftIcons.className = 'header__icons';

      elements.topLeftIcons.appendChild(elements.previewBack);
      elements.topLeftIcons.appendChild(elements.navThumbs);
      elements.topLeftIcons.appendChild(elements.wishlist);

      elements.topRightIcons = document.createElement('div');
      elements.topRightIcons.className = 'header__icons';
      elements.headerLogo.style.right = '0px';
      elements.headerLogo.style.textAlign = 'right';
      elements.headerLogo.style.height = '40px';

      elements.topRightIcons.appendChild(elements.search);
      elements.topRightIcons.appendChild(elements.share);
      elements.topRightIcons.appendChild(elements.download);

      if (!bookModel.settings.show_share_button && bookModel.settings.show_download_button) {
        elements.download.removeAttribute('hidden');        
      } else {
        elements.download.setAttribute('hidden', true);
      }

      setHeaderElementsOffsets();

      elements.header.style.zIndex = 20;
      elements.header.style.padding = '0 7px';
      elements.header.style.backgroundColor = `#${this.customizeBookBackgroundColor__getCorrectAvgHex()}E6`;

      if (isLayoutBlocked()) {
        elements.header.classList.add('hide');
      }
    }

    elements.header.appendChild(elements.headerLeft);
    elements.header.appendChild(elements.headerCenter);
    elements.header.appendChild(elements.headerRight);

    elements.header.appendChild(elements.headerBorder);

    if (is_mobile()) {
      calculateOffsetAndCenterTitle();
    }
  };

  const setHeaderElementsOffsets = () => {
    const iconSize = 35;
    const headerRightButtons = [elements.search, elements.share, elements.download].filter(item => item && !item.hasAttribute('hidden'));
    const headerLeftButtons = [elements.previewBack, elements.wishlist, elements.navThumbs].filter(item => item && !item.hasAttribute('hidden'));

    elements.navThumbs.style.margin = '0px';
    elements.share.style.margin = '0px';
    elements.download.style.margin = '0px';

    elements.headerCenter.appendChild(elements.headerBookName);
    // elements.headerCenter.style.width = 'calc(100% - 112px)';
    // elements.headerCenter.style.marginRight = '12px';
    // elements.headerCenter.style.left = '24px';

    elements.headerLeft.appendChild(elements.topLeftIcons);
    elements.headerRight.appendChild(elements.topRightIcons);
    
    elements.headerLeft.style.width = `${headerLeftButtons.length * iconSize}px`;
    elements.headerRight.style.width = `${headerRightButtons.length * iconSize}px`;
  };

  this.setHeaderOffsets = () => {
    setHeaderElementsOffsets();
    calculateOffsetAndCenterTitle();
  };

  const showMenuIfNeeded = function () {
    if (isLayoutBlocked()) return;

    if (elements.headerBorder && !elements.headerBorder.hasAttribute('hidden')) {
      elements.header.classList.remove('hide');
    }

    if (elements.footerBorder && !elements.footerBorder.hasAttribute('hidden')) {
      elements.footer.classList.remove('hide');
    }
  };

  this.isDoublePage = function () {
    if (forceOnePage) {
      return false;
    }
    return (
      is_horisontal() &&
      !settings.showSinglePageMode &&
      !(is_mobile() && !mobile_test_horizontal())
    );
  };

  this.showHideArrows__embed_not_customize = function (currentPage) {
    if (currentPage === 1) {
      elements.navNextPage.hidden = parseInt(bookModel.pages) === 1;
    } else {
      elements.navPrevPage.hidden = false;
      if (currentPage < bookModel.pages - (self.isDoublePage() ? 1 : 0)) {
        elements.navNextPage.hidden = false;
      }
    }
  };

  this.showHideArrows = function (currentPage) {
    elements.navPrevPage.hidden = true;
    elements.navNextPage.hidden = true;
    elements.navPrevPageAbsolute.hidden = true;
    elements.navNextPageAbsolute.hidden = true;

    if (!settings.nextPrevButtons) return;

    if (is_mobile()) {
      elements.navPrevPage.style.width = '0px';
      elements.navNextPage.style.width = '0px';

      if (currentPage === 1) {
        elements.navNextPageAbsolute.hidden = parseInt(bookModel.pages) === 1;
      } else {
        elements.navPrevPageAbsolute.hidden = false;

        if (currentPage < bookModel.pages) {
          elements.navNextPageAbsolute.hidden = false;
        }
      }
    } else {
      if (embedmode && !customizemode) {
        this.showHideArrows__embed_not_customize(currentPage);
      } else {
        if (currentPage === 1) {
          elements.navNextPage.hidden = parseInt(bookModel.pages) === 1;
        } else {
          elements.navPrevPage.hidden = false;

          if (currentPage < bookModel.pages) {
            elements.navNextPage.hidden = false;
          }

          if (this.isDoublePage() && parseInt(currentPage + 1) === parseInt(bookModel.pages)) {
            elements.navNextPage.hidden = true;
          }
        }
      }
    }
  };

  this.setPage = async function (v, initiator) {
    if (!isConverted && customizemode) return;

    if (mainInitiator && mainInitiator != initiator) {
      mainInitiator = null;
      return;
    }
    
    if (pageFlip && pageFlip.isBookZoomed()) {
      await resetScale();
    }

    setCorrectRangerColorWithPageNum(v, 'pageranger');

    v = Math.abs(parseInt(v)); // Only positive int's

    if (['pageClick', 'pageClickS', 'toc', 'ranger'].indexOf(initiator) !== -1) {
      HELPER.testLeadToShow(bookModel, v);
    }

    if (v === currentPage) return; // prevent double jobs
    if (v < 1) v = 1; // can't be less than 1
    if (v > bookModel.pages) v = bookModel.pages; // can't be more than bookModel.pages

    let prevV = v;

    if (v !== 1 && v % 2 === 1 && this.isDoublePage()) {
      v = v - 1;
    }

    mainInitiator = initiator;
    prevPage = currentPage;
    currentPage = v;
    
    // Show/Hide arrow before animation on first/last pages.
    if (self.isDoublePage()) {
      if (currentPage >= self.getModel().pages - 1) {
        this.showHideArrows(currentPage);
      }
    } else {
      if (currentPage >= self.getModel().pages || currentPage <= 1) {
        this.showHideArrows(currentPage);
      }
    }
    
    if (isFirstRender) {
      isFirstRender = false;
      this.showHideArrows(currentPage);
      this.setArrowsOffsetByThickness();
    } else {
      _global.page = currentPage;
    }

    blockNavAreaByPage(currentPage);
    
    // if we have pagesRanger, update it
    if (settings.pagesRanger && initiator !== 'ranger') {
      const curIndex =
        is_horisontal() &&
        !settings.showSinglePageMode &&
        !(is_mobile() && !mobile_test_horizontal())
          ? Math.floor(currentPage / 2) + 1
          : currentPage;

      elements.pagesRanger.setValue(curIndex, false, true);

      if (initiator !== 'resize') {
        self.setFooterPage(currentPage);
      }
    }

    elements.pagesRangerLeftNum.innerText = currentPage;
    renderPage(initiator);

    if (bookModel.settings.show_toc_button) {
      APP.Layout.setTocItemActiveByPageNum(prevV);
    }

    if (APP.Search && APP.Search.renderSearchResult) {
      APP.Search.renderSearchResult();
    }

    self.setCurrentThumbByPage(prevV);

    // dla zmiany audio container
    if (initiator == 'ranger') {
      self.removeHotspots();
    }

    APP.handleResponeToParent({ page: currentPage });
  };

  this.getThicknessWidth = (positive = false) => {
    const model = self.getModel();

    if (is_mobile() || !model.settings.show_book_thickness) {
      return [0, 0];
    }

    const thicknessMaxWidth = 10;
    const _currentPage = parseInt(currentPage, 10);
    const pagesCount = parseInt(model.pages, 10);

    let thicknessWidthStart = -Math.abs(_currentPage >= thicknessMaxWidth ? thicknessMaxWidth : _currentPage);
    let thicknessWidthEnd = -Math.abs(pagesCount - _currentPage >= thicknessMaxWidth ? thicknessMaxWidth : pagesCount - _currentPage);

    if (_currentPage <= 1) {
      thicknessWidthStart = 0;
      
      // if (!settings.showSinglePageMode && !model.settings.show_single_page_mode) {
        thicknessWidthEnd = 0;
      // }
    }

    if (_currentPage >= pagesCount || (self.isDoublePage() && pagesCount % 2 == 1 && _currentPage + 1 >= pagesCount)) {
      thicknessWidthStart = 0;
      thicknessWidthEnd = 0;
    }

    if (settings.showSinglePageMode || model.settings.show_single_page_mode) {
      thicknessWidthStart = 0;
    }

    if (is_vertical() && !is_mobile()) {
      thicknessWidthStart = 0;
    }

    if (positive) {
      return [Math.abs(thicknessWidthStart), Math.abs(thicknessWidthEnd)];
    }

    return [thicknessWidthStart, thicknessWidthEnd];
  };

  this.setArrowsOffsetByThickness = () => {
    let thickness = [0, 0];

    if (self.getModel().settings.show_book_thickness) {          
      thickness = self.getThicknessWidth(true);
    }

    const hardCoverSettings = this.getPageFlipHardCoverSettings();

    if (hardCoverSettings) {
      thickness[0] = thickness[0] - hardCoverSettings.hardWrapper.WRAPPER_EXTRA_WIDTH / 2;
      thickness[1] = thickness[1] - hardCoverSettings.hardWrapper.WRAPPER_EXTRA_WIDTH / 2;
    }

    const width = elements.innerChild.getBoundingClientRect().width;
    const persent = (width / 100) * 25;

    if (!settings.showSinglePageMode && !self.getModel().settings.show_single_page_mode) {
      if (currentPage === 1) {
        const hardCoverOffset = hardCoverSettings ? thickness[1] : 0;

        elements.navNextPage.style.right = is_vertical() ? `-${thickness[1]}px` : `${persent + hardCoverOffset}px`;
        elements.navNextPage.style.left = 'unset';
      } else if (currentPage == bookModel.pages && currentPage % 2 == 0) {
        const hardCoverOffset = hardCoverSettings ? thickness[0] : 0;

        elements.navPrevPage.style.left = is_vertical() ? `-${thickness[0]}px` : `${persent + hardCoverOffset}px`;
        elements.navPrevPage.style.right = 'unset';
      } else {
        elements.navNextPage.style.right = `-${thickness[1]}px`;
        elements.navPrevPage.style.left = `-${thickness[0]}px`;
      }
    } else {
      elements.navNextPage.style.right = `-${thickness[1]}px`;
      elements.navPrevPage.style.left = `0px`;
    }
  };

  this.setFooterPage = function (page) {
    if (pageFlip && pageFlip.isBookZoomed()) {
      self.scale(0);
    }

    if (!page) {
      page = self.getCurrentPage();
    }

    let currentPages = String(page);

    if (forceOnePage) {
      elements.pageNumInput.style.width = `${currentPages.length * (is_mobile() ? 15 : 14)}px`;
      elements.pageNumInput.value = currentPages;
    } else if (
      is_horisontal() &&
      page !== 1 &&
      page !== bookModel.pages &&
      !settings.showSinglePageMode
    ) {
      if (is_mobile() && !mobile_test_horizontal()) {
        elements.pageNumInput.style.width = `${currentPages.length * 10}px`;
        elements.pageNumInput.value = currentPages;
        return;
      } else {
        // console_log("setFooterPage", [page, bookModel.pages] );
        if (page === parseInt(bookModel.pages)) {
          elements.pageNumInput.style.width = `${currentPages.length * (is_mobile() ? 12 : 12)}px`;
        } else {
          if (page % 2 === 0) {
            currentPages = `${page}-${page + 1}`;
          } else {
            currentPages = `${page - 1}-${page}`;
          }
          elements.pageNumInput.style.width = `${currentPages.length * (is_mobile() ? 12 : 10)}px`;
        }

        elements.pageNumInput.value = currentPages;
      }
    } else {
      elements.pageNumInput.style.width = `${currentPages.length * (is_mobile() ? 15 : 12)}px`;
      elements.pageNumInput.value = currentPages;
    }
  };

  this.prevPage = function () {
    if (isSomePopupShowing() || currentPage <= 1) return;

    hideFSButtonIfNeeded();

    const bookViewType = getBookViewType();
    const newPage = Math.max(1, currentPage - (bookViewType === 'SINGLE' ? 1 : 2));

    blockNavAreaByPage(newPage);

    self.setPage(newPage, 'prevPage');
  };

  this.nextPage = function () {
    if (isSomePopupShowing() || currentPage >= bookModel.pages) return;

    hideFSButtonIfNeeded();

    const bookPagesCount = +bookModel.pages;
    const bookViewType = getBookViewType();
    const newPage = Math.min(bookPagesCount, currentPage + (bookViewType === 'SINGLE' ? 1 : 2));

    blockNavAreaByPage(newPage);

    self.setPage(newPage, 'nextPage');
  };

  const getAllModals = () => {
    const shareModal = elements.shareModalElements ? elements.shareModalElements.modal : null;
    const leadFormModal = elements.leadFormModalElements ? elements.leadFormModalElements.modal : null;
    const alertModal = elements.alertModalElements ? elements.alertModalElements.modal : null;
    const galleryModal = elements.galleryModalElements ? elements.galleryModalElements.modal : null;
    const videoModal = elements.videoModalElements ? elements.videoModalElements.modal : null;
    const printModal = elements.printModalElements ? elements.printModalElements.modal : null;
    const sourceModal = elements.sourceModalElements ? elements.sourceModalElements.modal : null;
    const productModal = document.querySelector('.product-modal');

    const existingsModals = [shareModal, leadFormModal, alertModal, galleryModal, videoModal, productModal, printModal, sourceModal].filter(Boolean);

    return existingsModals;
  };

  const isSomePopupShowing = () => {
    const existingsModals = getAllModals();
    const isSomeShowed = existingsModals.some(modal => !modal.hidden);

    return isSomeShowed;
  };

  this.hideAllModals = () => {
    const existingsModals = getAllModals();

    existingsModals.forEach(modal => modal.hide && typeof(modal.hide) === 'function' && (modal.hide()));
  };

  const getBookViewType = () => {
    const BOOK_VIEW_TYPE = {
      SINGLE: 'SINGLE',
      DOUBLE: 'DOUBLE'
    };

    const orientation = pageFlip && (pageFlip.getOrientation());

    if (forceOnePage || orientation === 'portrait') {
      return BOOK_VIEW_TYPE.SINGLE;
    }

    const bookPagesCount = +bookModel.pages;
    const oddNumberOfPages = bookPagesCount % 2 === 1;
    const isFirstPage = currentPage <= 1;
    const isLastPage = currentPage >= (oddNumberOfPages ? bookPagesCount - 1 : bookPagesCount);
    
    if (isFirstPage || isLastPage) {
      return BOOK_VIEW_TYPE.SINGLE;
    }

    if (is_mobile()) {
      if (orientation === 'landscape') {
        return BOOK_VIEW_TYPE.DOUBLE;
      }

      return BOOK_VIEW_TYPE.SINGLE;
    }

    return BOOK_VIEW_TYPE.DOUBLE;
  };

  const blockNavAreaByPage = (page) => {
    const isBookScaled = scale > 0;

    if (isBookScaled) {
      elements.navAreaNextPage.hidden = isBookScaled;
      elements.navAreaPrevPage.hidden = isBookScaled;

      return;
    }

    if (page >= bookModel.pages || (self.isDoublePage() && Number(bookModel.pages) % 2 == 1 && page + 1 >= bookModel.pages)) {
      elements.navAreaNextPage.hidden = true;
    } else if (elements.navAreaNextPage.hidden) {
      elements.navAreaNextPage.hidden = false;
    }

    if (page <= 1) {
      elements.navAreaPrevPage.hidden = true;
    } else if (elements.navAreaPrevPage.hidden) {
      elements.navAreaPrevPage.hidden = false;
    }
  };

  const canBeZoomedOnThisElement = e => {
    const path = e.composedPath() || e.path;

    if (e.touches.length >= 2 && !path.filter(item => item.classList && item.classList.contains('Book__innerChild')).length) {
      return false;
    }

    return true;
  };

  this.getBookSize = () => {
    const { width, height } = elements.innerChild.style;

    return {
      bookWidth:
        is_horisontal() &&
        !settings.showSinglePageMode &&
        !(is_mobile() && !mobile_test_horizontal())
          ? parseFloat(width) / 2
          : parseFloat(width),
      bookHeight: parseFloat(height),
    };
  };

  this.getBookModelSettings = () => {
    return bookModel.settings;
  };

  const updateOriginalLink = (arr, firstLoadingPage = null) => {
    const protocol = arr[0];
    const domain = arr[1];

    return `${protocol}//${domain}/flip-book/${model.publisherId}/${model.bookId}/page/${firstLoadingPage || currentPage}`;
  };

  const updateDevLink = (arr, firstLoadingPage = null) => {
    const protocol = arr[0];
    const domain = arr[1];

    return `${protocol}//${domain}/publuudev/${model.publisherId}/${model.bookId}/page/${firstLoadingPage || currentPage}`;
  };

  const updateOnlineLink = (arr, isFlipBookLink, firstLoadingPage = null) => {
    const protocol = arr[0];
    const domain = arr[1];

    if (!isFlipBookLink) {
      return `${protocol}//${domain}/${model.publisherId}/${model.bookId}/page/${firstLoadingPage || currentPage}`;
    }

    return `${protocol}//${domain}/flip-book/${model.publisherId}/${model.bookId}/page/${firstLoadingPage || currentPage}`;
  };

  const updateCustomDomainLink = (arr, isFlipBookLink, isCustomUrl, firstLoadingPage = null) => {
    const protocol = arr[0];
    const domain = arr[1];

    if (isFlipBookLink) {
      return `${protocol}//${domain}/flip-book/${model.publisherId}/${model.bookId}/page/${firstLoadingPage || currentPage}`;
    }

    if (isCustomUrl) {
      return `${protocol}//${domain}/${arr[2]}/page/${firstLoadingPage || currentPage}`;
    }

    return `${protocol}//${domain}/flip-book/${model.publisherId}/${model.bookId}/page/${firstLoadingPage || currentPage}`; 
  };

  this.updateURL = () => {
    const arr = LOCATION_HREF.split('?')[0].split('/').filter(item => item && !!item.length);
    const isFlipBookLink = LOCATION_HREF.indexOf('/flip-book/') !== -1;
    const isCustomUrl = window.customUrl && !isFlipBookLink && LOCATION_HREF.indexOf('flip-book') === -1;
    const isLinkWithPage = !URLChanged && LOCATION_HREF.indexOf(`/page/`) !== -1;
    
    let url = null;
    let firstLoadingPage = isLinkWithPage && !URLChanged ? +LOCATION_HREF.split('?')[0].split('/page/')[1] : null;

    if (firstLoadingPage !== null && (firstLoadingPage < 1 || firstLoadingPage > +bookModel.pages)) {
      firstLoadingPage = null;
    }

    switch (document.location.host) {
      case 'publuu.com':
        url = updateOriginalLink(arr, firstLoadingPage);
        break;
      
      case 'online.publuu.com':
        url = updateOnlineLink(arr, isFlipBookLink, firstLoadingPage);
        break;
      
      case 'va.publuu.com':
        url = updateDevLink(arr, firstLoadingPage);
        break;

      default:
        url = updateCustomDomainLink(arr, isFlipBookLink, isCustomUrl, firstLoadingPage);
        break;
    }

    if (currentPage == 1) {
      if (!isLinkWithPage && url.indexOf(`/page/${currentPage}`) !== -1) {
        url = url.replace(`/page/${currentPage}`, '');
      }
    }

    URLChanged = true;

    window.history.replaceState('', '', url);
    return;

    if (startPage) {
      let arr = LOCATION_HREF.split('/');
      let arrayItemIndex = (arr[5] && !isNaN(parseInt(arr[5]))) ? 5 : 4;

      if (arr[arrayItemIndex].includes('?')) {
        let arr4 = arr[arrayItemIndex].split('?');
        arr[arrayItemIndex] = arr4[0];
      }
      
      arr[arrayItemIndex + 1] = 'page';
      
      arr[arrayItemIndex + 2] = currentPage;
      let newArr = arr.filter((_, index) => index <= arrayItemIndex + 2);
      let link = newArr.join('/');
      window.history.replaceState('', '', link);
    } else {
      if (LOCATION_HREF.includes(APP.ROOT_PATH)) {
        //dont update local
        let url = `${APP.ROOT_PATH}${model.publisherId}/${model.bookId}/page/${currentPage}`;
        window.history.replaceState('', '', url);
      }
    }
  };

  this.pageFlipGetState = () => {
    return pageFlip.getState();
  };

  const initAudio = () => {
    if (audio) return;

    const audioElements = Array.from(document.querySelectorAll('.flip-audio'));

    if (audioElements.length > 0) {
      audio = audioElements[0];
    }
  };

  this.playPageFlipSound = () => {
    if (!audio) {
      initAudio();
    }

    if (audio && APP.someUserIteraction) {
      if (!is_safari()) {
        audio.currentTime && audio.pause(), (audio.currentTime = 0); // restetujemy jak cos leci
        audio.play();
      } else {
        if (pageFlip.getState() == 'flipping') {
        } else {
          audio.currentTime && audio.pause(), (audio.currentTime = 0); // restetujemy jak cos leci

          if (prevPage !== 0) {
            audio.play();
          }
        }
      }
    }
  };

  this.playPageFlipSoundSafari = () => {
    if (!audio) {
      initAudio();
    }

    if (!elements.audioIcon.classList.contains('sound-off')) {
      audio && audio.currentTime && audio.pause(), (audio.currentTime = 0), audio.play();
    }
  };

  var loadedPages = [],
    pagesToLoad = [],
    pages = {},
    timerGoodQualityImages,
    timerBadQualityImages,
    observer;

  var renderPage_setPagesToLoad = function (currentPage) {
    pagesToLoad = [];
    pagesToLoad.push(currentPage);
    var orientation = pageFlip.getOrientation();

    if (forceOnePage || !self.isDoublePage()) {
      //Pierwszza strona i sa kolejne
      if (Number(bookModel.pages) > currentPage) {
        pagesToLoad.push(currentPage + 1);
      }
      if (currentPage > 1) {
        pagesToLoad.push(currentPage - 1);
      }

      return pagesToLoad;
    }

    if (currentPage === 1 || !self.isDoublePage()) {
      if (Number(bookModel.pages) > 1) {
        pagesToLoad.push(currentPage + 1);

        if (orientation === 'landscape' && Number(bookModel.pages) > 2) {
          pagesToLoad.push(currentPage + 2);
        }
      }
    } else {
      if (currentPage === bookModel.pages) {
        pagesToLoad.push(currentPage - 1);
        if (orientation === 'landscape' && currentPage - 2 > 0) {
          pagesToLoad.push(currentPage - 2);
        }
      } else {
        if (orientation === 'landscape') {
          if (currentPage % 2 == 0) {
            pagesToLoad.push(currentPage + 1);
            if (currentPage + 2 <= bookModel.pages) pagesToLoad.push(currentPage + 2);
            if (currentPage + 3 <= bookModel.pages) pagesToLoad.push(currentPage + 3);
            if (currentPage - 1 > 0) pagesToLoad.push(currentPage - 1);
            if (currentPage - 2 > 0) pagesToLoad.push(currentPage - 2);
          } else {
            pagesToLoad.unshift(currentPage - 1);
            if (currentPage + 1 <= bookModel.pages) pagesToLoad.push(currentPage + 1);
            if (currentPage + 2 <= bookModel.pages) pagesToLoad.push(currentPage + 2);
            if (currentPage - 2 > 0) pagesToLoad.push(currentPage - 2);
            if (currentPage - 3 > 0) pagesToLoad.push(currentPage - 3);
          }
        } else {
          pagesToLoad.push(currentPage - 1, currentPage + 1);
        }
      }
    }

    return pagesToLoad;
  };

  const setZIndexToSideElements = (state) => {
    const isLastPage = currentPage === bookModel.pages;
    const isEvenPageCount = bookModel.pages % 2 === 0;
    const newZIndex = state === 'read' && isLastPage && isEvenPageCount ? '2' : '1';

    if (elements.absoluteFullscreen) {
      elements.absoluteFullscreen.style.zIndex = newZIndex;
    }
    
    if (elements.noteBtnBottom) {
      elements.noteBtnBottom.style.zIndex = newZIndex;
    }
    
    if (elements.search) {
      elements.search.style.zIndex = newZIndex;
    }
  };

  var renderPage = function (initiator) {
    clearTimeout(timerGoodQualityImages);
    clearTimeout(timerBadQualityImages);
    
    const book = document.getElementById('book');

    if (!book) return;

    if (
      !elements.audioIcon.classList.contains('sound-off') &&
      initiator !== 'resize' &&
      initiator !== 'orientationchange' 
      // && !elements.audioIcon.hidden
      && bookModel.settings.show_audio_button
    ) {
      if (pageFlip && !is_mobile()) {
        self.playPageFlipSound();
      }
    }

    if (!pageFlip) {
      pageFlip = new St.PageFlip(document.getElementById('book'), {
        width: pageWidth,
        height: pageHeight,
        minWidth: pageWidth,
        maxWidth: bookModel.pages_info[0].width,
        minHeight: pageHeight,
        maxHeight: bookModel.pages_info[0].height,
        size: 'fixed',
        autoSize: false,
        usePortraite: true,
        flippingTime: forceOnePage ? 500 : 400, //480
        maxShadowOpacity: is_mobile() ? 0.3 : 0.6,
        showShadowOnMobile: true,
        showCover: true,
        bookPages: pagesNum,
        startPage: startPage > Number(bookModel.pages) ? Number(bookModel.pages) : startPage,
        hardCoverMode: !!bookModel.settings.hard_cover_enabled,
        hardCoverSettings: {
          color: bookModel.settings.hard_cover_color || '#fff'
        },
        hardFlipMode: APP.bookModel.settings.flip_type === 'hard'
      });
      pageFlip.loadFromHTML(document.querySelectorAll('.Page'));
      setBookInnerSize();
      self.connectObserver();

      pageFlip.on('pageChange', function (e) {
        APP.EMBED.clearPlayersOnChangePage();
      });

      pageFlip.on('flip', function (e) {
        if (is_mobile() && embedmode) {
          hideFSButtonIfNeeded();
        }
        if (startPage) {
          self.setPage(e.data + 1, 'pageClickS');
        } else {
          // raczej nie potrzebne jest, funkcja setPage ma w sobie warunki na to.
          if (pageFlip.getOrientation() == 'landscape') {
            if (
              e.data !== 0 &&
              e.data + 2 !== elements.pagesRanger.getValue() &&
              initiator !== 'pageClick'
            ) {
              let currPage = e.data + 2;
              if (e.data + 2 > bookModel.pages) currPage = bookModel.pages;

              self.setPage(currPage, 'pageClick');
            }
            if (e.data == 0 && currentPage !== 1) self.setPage(1, 'pageClick');
          } else {
            if (e.data + 1 !== elements.pagesRanger.getValue() && initiator !== 'pageClick') {
              self.setPage(e.data + 1, 'pageClick');
            }
          }
        }
        mainInitiator = null;
      });

      pageFlip.on('changeState', (e) => {
        setZIndexToSideElements(e.data);

        if (e.data === 'fold_corner') {
          bookNode.classList.add('book-action--fold-corner');
        }

        if (e.data !== 'fold_corner') {
          bookNode.classList.remove('book-action--fold-corner');
        } else {
          fsToggleShowedTimes++;
        }

        for (let i of pagesToLoad) {
          if (e.data !== 'read') {
            // if (pages[i]) {
            //   pages[i].hotspotsOff();
            // }
          } else {
            if (pages[i] && pages[i].getPage().style.display === 'none') {
              pages[i].removeHotspots();
            }
            setTimeout(() => {
              if (pages[i] && pages[i].getPage().style.display === 'none') {
                pages[i].removeHotspots();
              }
              if (pages[i] && pages[i].getPage().style.display === 'block') {
                pages[i].hotspotsRender(pageWidth, pageHeight);
                pages[i].hotspotsOn();
              }
            }, 200);
          }
        }
      });
    }

    pagesToLoad = [];
    pages = {};
    pagesToLoad = renderPage_setPagesToLoad(currentPage);

    for (let i in loadedPages) {
      if (!pagesToLoad.includes(loadedPages[i].getPageNumber())) {
        if (
          loadedPages[i].getPage().dataset.preload === 'start' &&
          loadedPages[i].isAbortController()
        ) {
          loadedPages[i].stopPreload();
        }
        if (
          loadedPages[i].getPage().dataset.loading === 'start' &&
          loadedPages[i].isAbortController()
        ) {
          loadedPages[i].cancelImageLoading();
        }
      }
    }

    self.getPage(pagesToLoad, function (PagesObj) {
      pages = PagesObj;
    });

    for (let i of pagesToLoad) {
      if (!loadedPages.includes(pages[i]) && pages[i] && !pages[i].getPage()) {
        pages[i].render();
      }
    }
    timerBadQualityImages = setTimeout(() => {
      for (let i of pagesToLoad) {
        if (pages[i] && !loadedPages.includes(pages[i]) && !pages[i].getPage().dataset.preload) {
          pages[i].loadBluredImage();
          pages[i].visibility();
          loadedPages.push(pages[i]);
        }
      }
    }, 150);
    timerGoodQualityImages = setTimeout(() => {
      for (var i of pagesToLoad) {
        if (pages[i] && !pages[i].getPage().dataset.loading) {
          pages[i].setPageImages();
        }
      }
    }, 500);

    if (initiator == 'nextPage') {
      // APP.EMBED.clearPlayersOnChangePage();
      pageFlip.flipNext();
      normalizeInnerPageTransform('changePage');
    }
    if (initiator == 'prevPage') {
      // APP.EMBED.clearPlayersOnChangePage();
      pageFlip.flipPrev();
      normalizeInnerPageTransform('changePage');
    }

    if (
      initiator == 'sidebar.click' ||
      initiator == 'ranger' ||
      initiator == 'footerInput' ||
      initiator == 'hotspot' ||
      initiator == 'sidebar.toc' ||
      initiator == 'customize' ||
      initiator == 'product'
    ) {
      // APP.EMBED.clearPlayersOnChangePage();
      pageFlip.flip(currentPage - 1);
      normalizeInnerPageTransform('other');
    }

    if (initiator === 'pageClickS') {
      // APP.EMBED.clearPlayersOnChangePage();
    }

    if (initiator == 'ranger') {
      // self.updateURL();//troche hack
    }
    self.updateURL();
  };

  this.removeHotspots = () => {
    for (let i of pagesToLoad) {
      if (pages[i] && pages[i].getPage().style.display === 'none') {
        pages[i].removeHotspots();
      }
      setTimeout(() => {
        if (pages[i] && pages[i].getPage().style.display === 'none') {
          pages[i].removeHotspots();
        }
        if (pages[i] && pages[i].getPage().style.display === 'block') {
          pages[i].hotspotsRender(pageWidth, pageHeight);
          pages[i].hotspotsOn();
        }
      }, 200);
    }
  }

  const countMenuHeight = function () {
    const windowHeight = window.innerHeight;

    if (!isSmallDesktopMode()) {
      let menuHeight = windowHeight * (settings.menuHProc / 100);
      menuHeight = menuHeight < settings.menuMinH ? settings.menuMinH : menuHeight;
      menuHeight = menuHeight > settings.menuMaxH ? settings.menuMaxH : menuHeight;
    } else {
      menuHeight = 0;
    }

    return menuHeight;
  };

  var countRightFreeSpace = function () {
    let windowWidth = window.innerWidth;
    var spaceW = (windowWidth - parseInt(elements.innerChild.style.width, 10)) / 2;

    return spaceW;
  };
  var countTopFreeSpace = function () {
    let windowHeight = window.innerHeight;
    var spaceH = (windowHeight - parseInt(elements.innerChild.style.height, 10)) / 2;

    return spaceH;
  };

  var normalizeLogoSize = function () {
    if (transparent) {
      elements.header.style.display = 'none';
    }
    let freeSpaceH = countTopFreeSpace() - 20;

    if (is_mobile()) {
    } else {
      let minWidth = 125;
      elements.headerLogo.style.minHeight = `${countMenuHeight()}px`;
      elements.headerLogo.style.minWidth = `${minWidth}px`;
      elements.headerLogo.style.maxHeight = `45px`;
      elements.headerLogo.style.maxWidth = `175px`;

      let freeSpaceW = countRightFreeSpace() - 20;
      if (freeSpaceW > minWidth) {
        elements.header.style.height = `${countMenuHeight()}px`;
        elements.headerLogo.style.height = null;
        elements.headerLogo.style.width = `${freeSpaceW}px`;
      } else {
        elements.headerLogo.style.width = null;
        elements.headerLogo.style.height = `${freeSpaceH}px`;
      }
    }
  };

  var setMarcinShadowHalfSize = function (firstPage = true) {
    document.querySelector('.stf__marcinBookShadow').style.display = 'block';
    document.querySelector('.stf__marcinBookShadow').classList.remove('doubleWidth');
    document.querySelector('.stf__marcinBookShadow').classList.add('singleWidth');
    document.querySelector('.stf__marcinBookShadow').style.right = firstPage ? '0px' : null;
    document.querySelector('.stf__marcinBookShadow').style.left = firstPage ? null : '0px';
    document.querySelector('.stf__marcinBookShadow').style.width = null;
  };

  var normalizeInnerPageTransform = function (sender) {
    requestAnimationFrame(function () {
      let innerOffsetInHardCoverMode = 0;
      const hardCoverSettings = bookModel.settings.hard_cover_enabled && self.getPageFlipHardCoverSettings();
      
      if (hardCoverSettings) {
        // innerOffsetInHardCoverMode = Math.ceil(hardCoverSettings.thickness.totalLines / 2);
        // innerOffsetInHardCoverMode = -hardCoverSettings.hardWrapper.WRAPPER_EXTRA_WIDTH / 4;
      }

      if (is_vertical() || (is_mobile() && !mobile_test_horizontal())) {
        elements.innerChild.style.transform = `translateX(${innerOffsetInHardCoverMode}px)`;

        const width = parseInt(document.querySelector('.Book__innerChild').style.width);

        if (document.querySelector('.stf__marcinBookShadow')) {
          document.querySelector('.stf__marcinBookShadow').style.width = `${width}px`;
          document.querySelector('.stf__marcinBookShadow').style.right = 'unset';
        }
      } //landscape
      else {
        if (!settings.showSinglePageMode) {
          if (currentPage === 1) {
            setMarcinShadowHalfSize(true);

            if (sender === 'setBookInnerSize') {
              elements.innerChild.style.transform = `translateX(calc(-25% - ${innerOffsetInHardCoverMode}px))`;
            }
          } else if (currentPage == bookModel.pages && currentPage % 2 == 0) {
            elements.innerChild.style.transform = `translateX(calc(25% + ${innerOffsetInHardCoverMode}px))`;

            setMarcinShadowHalfSize(false);
          }
        }
      }
    });
  };
  this.connectObserver = function () {
    var target = document.querySelector('.stf__block');
    observer = new MutationObserver(function (mutations) {
      if (is_vertical() || settings.showSinglePageMode) {
        mutations.forEach(function (mutation) {
          if (mutation.addedNodes.length > 0) {
            let page = mutation.addedNodes[0];
            if (page) page.innerHTML = '';
          }
        });
      }
    });

    var config = { attributes: true, childList: true, characterData: true };
    observer.observe(target, config);
  };

  const getPageRotation = (pageNumber) => {
    const bookModel = self.getModel();

    if (!bookModel.hasOwnProperty('rotations')) {
      return 0;
    }

    const pageRotation = bookModel.rotations.find(item => item.p == pageNumber);

    if (!pageRotation) {
      return 0;
    }

    return pageRotation.d;
  };

  this.getPage = function (page, callback) {
    if (is_array(page)) {
      var i,
        _pages = [],
        __pages = {};
      for (i = page.length - 1; i >= 0; i--) {
        if (page[i] >= 1 && page[i] <= bookModel.pages) {
          _pages.push(page[i]);
        }
      }
      for (i = _pages.length - 1; i >= 0; i--) {
        self.getPage(_pages[i], function (pageObject) {
          __pages[_pages[i]] = pageObject;
        });
      }
      if (is_function(callback)) {
        var f = function () {
          if (Object.keys(__pages).length == _pages.length) {
            callback(__pages);
          } else {
            setTimeout(f, 50);
          }
        };
        f();
      }
    } else {
      if (page == 0) return;

      if (!isset(Pages[page])) {
        const pageRotation = getPageRotation(page);

        Pages[page] = new Page({
          publisherId: bookModel.publisher_id,
          bookId: bookModel.ref_id ? bookModel.ref_id : bookModel.id,
          page: page,
          width: bookModel.pages_info[page - 1].width,
          height: bookModel.pages_info[page - 1].height,
          pageReady: bookModel.pages_info[page - 1].pageReady,
          svg: bookModel.pages_info[page - 1].svg,
          publishedAt: bookModel.published_at,
          Book: self,
          version: bookModel.version || 0,
          rotation: pageRotation
        });
        preloadCover(Pages[page].getCover());
      }
      if (is_function(callback)) callback(Pages[page]);
    }
  };

  this.getPageNoPreload = function (page, callback) {
    if (is_array(page)) {
      var i,
        _pages = [],
        __pages = {};
      for (i = page.length - 1; i >= 0; i--) {
        if (page[i] >= 1 && page[i] <= bookModel.pages) {
          _pages.push(page[i]);
        }
      }
      for (i = _pages.length - 1; i >= 0; i--) {
        self.getPageNoPreload(_pages[i], function (pageObject) {
          __pages[_pages[i]] = pageObject;
        });
      }
      if (is_function(callback)) {
        var f = function () {
          if (Object.keys(__pages).length == _pages.length) {
            callback(__pages);
          } else {
            setTimeout(f, 50);
          }
        };
        f();
      }
    } else {
      if (page == 0) return;
      if (!isset(Pages[page])) {
        const pageRotation = getPageRotation(page);

        Pages[page] = new Page({
          publisherId: bookModel.publisher_id,
          bookId: bookModel.ref_id ? bookModel.ref_id : bookModel.id,
          page: page,
          width: bookModel.pages_info[page - 1].width,
          height: bookModel.pages_info[page - 1].height,
          pageReady: bookModel.pages_info[page - 1].pageReady,
          svg: bookModel.pages_info[page - 1].svg,
          publishedAt: bookModel.published_at,
          Book: self,
          version: bookModel.version || 0,
          rotation: pageRotation
        });
      }
      if (is_function(callback)) callback(Pages[page]);
    }
  };

  this.getCurrentPage = function (callback) {
    return currentPage;
  };

  const getCurrentPageIfPageIsNegative = () => {
    if (currentPage == -1) {
      if (document.location.pathname.indexOf('/page/') !== -1) {
        return isNaN(Math.ceil(document.location.pathname.split('/page/')[1])) ? 1 : Math.ceil(document.location.pathname.split('/page/')[1]);
      }

      return 1;
    }

    return currentPage;
  };

  // features
  var __is_full_screen = false;

  this.getFullscreenState = function () {
    return __is_full_screen;
  };

  const fullscreenAnimationFirstTime = async (state, isBookLoaded = false) => {
    if (!show_fs_modal_button() || fsToggleShowedTimes !== 0) return;

    if (isBookLoaded || ['init', 'read', 'fold_corner'].includes(state)) {
      fsAnimationStarted = false;
      hideFSButtonIfNeeded();
      await fullscreenAnimation();

      fsTimeoutTime = 0;
      fsToggleShowedTimes++;
    } else {
      hideFSButtonIfNeeded();
    }
  };

  const fullscreenAnimation = function () {
    if (!show_fs_modal_button() || fsAnimationStarted) return;

    const offsetShowTime = fsTimeoutTime;
    const offsetHideTime = 1500;

    fsAnimationStarted = true;
    clearTimeout(APP.fsShowTimeout);
    clearTimeout(fsAnimationTimeout);

    return new Promise(resolve => {
      APP.fsShowTimeout = setTimeout(() => {
        showFSButtonIfNeeded();
  
        fsAnimationTimeout = setTimeout(() => {
          fsAnimationStarted = false;
  
          hideFSButtonIfNeeded();

          if (shouldShowAbsoluteFullscreenButton()) {
            showAbsoluteFullscreenButton();
          }

          resolve();
        }, offsetHideTime);
      }, offsetShowTime);
    });
  };

  const requestBrowserFullscreen = () => {
    if (document.documentElement.requestFullscreen) {
      document.documentElement.requestFullscreen();
    } else if (document.documentElement.webkitRequestFullscreen) {
      document.documentElement.webkitRequestFullscreen(); // WebKit-based browsers (older versions of Chrome, Safari)
    } else if (document.documentElement.mozRequestFullScreen) {
      document.documentElement.mozRequestFullScreen(); // Firefox (older versions)
    } else if (document.documentElement.msRequestFullscreen) {
      document.documentElement.msRequestFullscreen(); // Internet Explorer, Microsoft Edge
    }
  };

  const exitBrowserFullscreen = () => {
    if (document.exitFullscreen) {
      document.exitFullscreen(); // Standard method
    } else if (document.webkitExitFullscreen) {
      document.webkitExitFullscreen(); // WebKit-based browsers (older Chrome, Safari)
    } else if (document.mozCancelFullScreen) {
      document.mozCancelFullScreen(); // Firefox (older versions)
    } else if (document.msExitFullscreen) {
      document.msExitFullscreen(); // Internet Explorer, Microsoft Edge
    }
  };

  const getBrowserFullscreenElement = () => {
    const fullscreenElement = 
      document.fullscreenElement || 
      document.webkitFullscreenElement || 
      document.mozFullScreenElement || 
      document.msFullscreenElement;

    return fullscreenElement || null;
  };

  const handleFullscreenChange = () => {
    const fullscreenElement = getBrowserFullscreenElement();
    isVideoFullscreen = APP.EMBED.isPlayerFullscreen() || APP.EMBED.isMediaViewerFullscreen();

    if (!fullscreenElement && __is_full_screen) {
      __is_full_screen = false;

      changeFullscreenButtonsAppearance();
      self.hideSidebar();

      hideMenuIfNeeded();
      hideInnerBookButtons();
    } else {
      hideMenuIfNeeded();

      if (is_safari()) {
        const resizeEvent = window.document.createEvent('UIEvents');
        
        resizeEvent.initUIEvent('resize', true, false, window, 0);
        window.dispatchEvent(resizeEvent);
      }
    }

    const fullscreenTimeout = setTimeout(() => {
      createEvent('adjust-scrollbar', { fullScreenChange: true }, true);
      setAbsoluteFullscreenButtonClassName();
      
      if (is_small_embed_not_mobile()) {
        HELPER.sidebar_setStylesForEmbedFullscreen(__is_full_screen);
      }

      clearTimeout(fullscreenTimeout);
    }, 300);
  };

  const handleBeforeUnloadInFullscreen = () => {
    if (!getBrowserFullscreenElement()) return;

    exitBrowserFullscreen();
    __is_full_screen = false;
    handleFullscreenChange();
  };

  const changeFullscreenButtonsAppearance = () => {
    const fullscreenButtons = [elements.navFullScreen, elements.absoluteFullscreen].filter(Boolean);

    fullscreenButtons.forEach(btn => {
      btn.classList.toggle('Book__navFullScreen--on', __is_full_screen);

      if (btn.dataset && btn.dataset.type === BUTTON_ICON_TYPES.INLINE) {
        btn.innerHTML = HELPER.buttonsIcons[__is_full_screen ? 'fullscreenOffIcon' : 'fullscreenOnIcon'];
      }

      if (__is_full_screen) {
        btn.hidden = btn === elements.absoluteFullscreen;
      } else {
        if (!bookModel.settings.show_fullscreen_button) {
          btn.setAttribute('hidden', true);
          return;
        }

        if (Boolean(transparent)) {
          btn.hidden = btn === elements.navFullScreen;
        } else {
          btn.hidden = btn === elements.absoluteFullscreen;
        }
      }
    });
  };

  const fullscreenMobileAction = () => {
    let url = window.location.href;

    if (bookModel.settings.is_password_protected || bookModel.settings.enable_protected_domain == "1") {
      const protectedParameters = HELPER.cryptString((Date.now() + 30000).toString());
      const searchParams = new URLSearchParams({
        pp: protectedParameters
      });

      url = `${url}?${searchParams.toString()}`;
    }

    window.open(url, '_blank');
  };

  const fullScreenToggle = function () {
    const body = APP.Layout.getBody();

    self.hideSidebar();

    if (__is_full_screen) {
      exitBrowserFullscreen();
      __is_full_screen = false;
      changeFullscreenButtonsAppearance();

      if (body.classList.contains('is_embed_fullscreen')) {
        body.classList.remove('is_embed_fullscreen');
        HELPER.sidebar_setStylesForEmbedFullscreen(false);
      }

      hideMenuIfNeeded();
      hideInnerBookButtons();
    } else {
      if (is_mobile()) {
        fullscreenMobileAction();
        return;
      }

      requestBrowserFullscreen();
      __is_full_screen = true;
      changeFullscreenButtonsAppearance();

      if (is_small_embed_not_mobile() && body.classList.contains('is_embed')) {
        body.classList.add('is_embed_fullscreen');
        HELPER.sidebar_setStylesForEmbedFullscreen(true);
      }

      fsAnimationStarted = false;
      hideFSButtonIfNeeded();
      showMenuIfNeeded();
      showInnerBookButtons();
    }

    for (const i of pagesToLoad) {
      if (!pages[i]) return;

      // pages[i].removeHotspots();
      const hotspotsContainer = pages[i].getHotspotsContainer();

      if (hotspotsContainer) {
        hotspotsContainer.style.display = 'none';
        hotspotsContainer.style.opacity = '0';
        hotspotsContainer.style.visibility = 'hidden';
      }

      setTimeout(() => {
        if (!pages[i]) return;

        if (pages[i].getPage().style.display === 'block') {
          pages[i].adjustHotspotsSize(pageWidth, pageHeight);
          // pages[i].hotspotsRender(pageWidth, pageHeight);
          // pages[i].hotspotsOn();
        } 
      }, 300);
    }
    
    APP.Note.showNotesByDependsOnWidth();
  };

  this.hideSidebar = function (e) {
    if (!settings.sidebar) return;

    APP.Layout.getSidebar().hide();
    APP.Layout.getSidebarTOC().hide();
    APP.Layout.getSidebarSearch().hide();
    APP.Layout.getSidebarWishlist().hide();

    scaleRoot();

    elements.headerBookName.style.opacity = 0.8;
  };

  const previewBackToggle = () => {
    if (window.history && window.history.length && window.history.length > 1) {
      return window.history.go(-1);
    }

    window.close();
  };

  const scaleRoot = () => {
    const root = APP.Layout.getRoot();
    const newRootWidth = APP.Layout.getRootWidthDependOnSidebarState();

    if (root.clientWidth === newRootWidth && root.clientHeight === window.innerHeight) return;

    root.style.width = `${newRootWidth}px`;
    root.style.height = `${window.innerHeight}px`;

    setBookInnerSize(null, 'root.scale');
  };

  this.showSidebar = function (e) {
    if (!settings.sidebar) return;

    APP.Layout.getSidebarTOC().hide();
    APP.Layout.getSidebarSearch().hide();
    APP.Layout.getSidebarWishlist().hide();

    if (!isThumbsLoaded || !APP.Layout.getSidebar().querySelectorAll('img').length) {
      self.renderSidebarThumbs();
    } 
    
    self.animateScrollToThumb();

    if (APP.Layout.getSidebar().isActive()) {
      elements.headerBookName.style.opacity = 0.8;
      APP.Layout.getSidebar().hide();
    } else {
      self.refreshSidebarUnconvertedThumbs();
      APP.Layout.getSidebar().show();
    }

    scaleRoot();

    APP.Layout.getFooter().hide();
    return stopEvent(e);
  };

  this.setCurrentThumbByPage = function (page) {
    if (page % 2 === 1) {
      page -= 1;
    }

    let className = 'active-full';
    const thumbs = APP.Layout.getSidebar().querySelectorAll('.thumbnail-item');

    if (!thumbs || !thumbs.length) {
      return;
    }

    if (
      (!(is_mobile() && !mobile_test_horizontal()) || is_small_embed_not_mobile()) &&
      !(settings.showSinglePageMode || self.getModel().settings.show_single_page_mode)
    ) {
      className = 'active';
    }

    if ((is_horisontal() && self.isDoublePage()) && (page == 1 || page == self.getModel().pages)) {
      className = 'active-full';
    }

    thumbs.forEach(thumb => (thumb.classList && thumb.classList.contains(`bookSpread-${page}`)) ? thumb.classList.add(className) : (thumb.classList.remove('active-full'), thumb.classList.remove('active')));
  
    const sidebarInner = elements.sidebar.querySelector('.Book__sidebar');
    const activeThumbnailItem = elements.sidebar.querySelector('.thumbnail-item:is(.active, .active-full)');

    if (APP.Layout.getSidebar().isActive() && sidebarInner && activeThumbnailItem) {
      sidebarInner.scroll({
        top: activeThumbnailItem.offsetTop - 15,
        behavior: "smooth"
      });
    }
  }

  this.toggleSidebarTOC = function (e) {
    APP.Layout.getSidebar().hide();
    APP.Layout.getSidebarSearch().hide();
    APP.Layout.getSidebarWishlist().hide();

    if (APP.Layout.getSidebarTOC().isActive()) {
      elements.headerBookName.style.opacity = 0.8;
      APP.Layout.getSidebarTOC().hide();
    } else {
      APP.Layout.getSidebarTOC().show();
    }

    scaleRoot();

    APP.Layout.getFooter().hide();
    return stopEvent(e);
  };

  this.showSidebarTOC = function (e) {
    APP.Layout.getSidebar().hide();
    APP.Layout.getSidebarSearch().hide();
    APP.Layout.getSidebarWishlist().hide();

    if (!APP.Layout.getSidebarTOC().isActive()) {
      APP.Layout.getSidebarTOC().show();
    }

    scaleRoot();

    APP.Layout.getFooter().hide();
    return stopEvent(e);
  };

  this.showSidebarSearch = function (e) {
    APP.Layout.getSidebar().hide();
    APP.Layout.getSidebarTOC().hide();
    APP.Layout.getSidebarWishlist().hide();

    if (APP.Layout.getSidebarSearch().isActive()) {
      elements.headerBookName.style.opacity = 0.8;
      APP.Layout.getSidebarSearch().hide();
    } else {
      APP.Layout.getSidebarSearch().show();
      APP.Search.sidebarShowHandler();
    }

    scaleRoot();

    APP.Layout.getFooter().hide();
    return stopEvent(e);
  };

  this.showSidebarWishlist = function (e) {
    APP.Layout.getSidebar().hide();
    APP.Layout.getSidebarTOC().hide();
    APP.Layout.getSidebarSearch().hide();

    if (APP.Layout.getSidebarWishlist().isActive()) {
      elements.headerBookName.style.opacity = 0.8;
      APP.Layout.getSidebarWishlist().hide();
      APP.Products.sidebarHideHandler();
    } else {
      APP.Products.sidebarShowHandler();
      APP.Layout.getSidebarWishlist().show();
    }

    scaleRoot();

    APP.Layout.getFooter().hide();
    return stopEvent(e);
  }

  this.refreshSidebarUnconvertedThumbs = function () {
    const thumbs = document.getElementsByClassName('thumb-img');
    if (thumbs.length) {
      Array.prototype.forEach.call(thumbs, function (el) {
        // Do stuff here
        if (el.getAttribute('pageReady') == 'false') {
          if (bookModel.pages_info[el.getAttribute('pagenum') - 1].pageReady) {
            const WEBP = `&webp=${self.useWebP() ? 1 : 0}`;
            var src =
              APP.PATH_V2IMAGE +
              `?thumb=1&size=400&hidetext=0&iid=${model.bookId}&page=${el.getAttribute(
                'pagenum',
              )}&pid=${model.publisherId}${is_retina() ? '&retina' : ''}${WEBP}&x=update`;

            el.src = src;
          }
        }
      });
    }
  };

  this.isBookView = function (page) {
    let isBookView = false;
    if (
      page !== 1 &&
      page !== bookModel.pages &&
      !(is_mobile() && !mobile_test_horizontal()) &&
      !is_vertical() &&
      !settings.showSinglePageMode
    ) {
      isBookView = true;
    } else {
      isBookView = false;
    }
    return isBookView;
  }

  this.getSidebarThumbClassName = function (key, isBookView) {
    let keyNum = Number(key);
    let className = 'right-thumbnail';

    if (isBookView) {
      if (Number(key) !== 1 && Number(key) !== bookModel.pages) {
        if (keyNum % 2 == 0) {
          className = 'left-thumbnail';
        } else {
          className = 'right-thumbnail';
        }
      }
    }
    return className;
  }

  this.renderSidebarThumbs = function () {
    isThumbsLoaded = true;
    const thumbs = document.querySelectorAll('.thumbnail-item .thumb');
    let firstThumb = null;
    let isOnePage = false;
    if ((!is_mobile() || !is_small_embed_not_mobile()) && (!self.isDoublePage() || forceOnePage)) {
      isOnePage = true;
    }

    if (thumbs) {
      thumbs.forEach((thumb, index) => {
        let thumbWidth = thumb.dataset.w;
        let thumbHeight = thumb.dataset.h;
        const maxThumbWidth = 200;
        const thumbInConverting = thumb.classList && thumb.classList.contains('converting');

        if (thumbInConverting) {
          thumbWidth = firstThumb.dataset.w;
          thumbHeight = firstThumb.dataset.h;
        } else if (thumb.dataset.w > maxThumbWidth) {
          thumbWidth = maxThumbWidth;
          thumbHeight = parseInt((maxThumbWidth * thumb.dataset.h) / thumb.dataset.w);
        }

        thumb.parentNode.parentNode.style = isOnePage 
          ? `--thumb-width: 100%; --thumb-height: 100%; --thumb-width-or: ${thumbWidth}px; --thumb-height-or: ${thumbHeight}px` 
          : `--thumb-width: ${thumbWidth}px; --thumb-height: ${thumbHeight}px; --thumb-width-or: ${thumbWidth}px; --thumb-height-or: ${thumbHeight}px;`;

        const page = Pages[thumb.dataset.page]
        const isBookView = self.isBookView(page);
        const className = self.getSidebarThumbClassName(thumb.dataset.page, isBookView);
        const thumbImage = HELPER.getThumbDiv(isOnePage ? '100%' : thumbWidth + 'px', isOnePage ? '100%' : thumbHeight + 'px', page, className, thumbInConverting);
        thumb.innerHTML = '';
        thumb.insertAdjacentHTML('beforebegin', thumbImage);

        if (index === 0) {
          firstThumb = thumb;
        }
      })

      if (is_mobile() || is_small_embed_not_mobile()) {
        HELPER.setThumbSizeAndCountMobile();
      }

      document.querySelectorAll('.thumbnail-item').forEach(item => {
        if (item.querySelectorAll('.thumb').length == 2 || isOnePage) {
          item.classList.add('single')
        }
      })

      self.animateScrollToThumb();
    }
  }

  this.animateScrollToThumb = function () {
    const currentPage = self.getCurrentPage();

    if (self.isDoublePage()) {
      if (currentPage == 1) {
        self.setCurrentThumbByPage(currentPage);
      } else if (currentPage % 2 == 0) {
        self.setCurrentThumbByPage(currentPage);
      } else if (currentPage % 2 == 1) {
        self.setCurrentThumbByPage(currentPage - 1);
      }
    } else {
      self.setCurrentThumbByPage(currentPage);
    }

    APP.Layout.getSidebar().querySelector(`.bookSpread-${currentPage}`).scrollIntoView({
      behavior: 'auto',
      block: 'center'
    })
  }

  const createThumbnailItem = (page, w, h) => {
    let insert = '';
    const loadPages = [page];
    const loadedThumbnails = [];

    const isSidePage = [1, +bookModel.pages].includes(page);
    const isBookView = self.isBookView(page);

    const thumbnailItem = document.createElement('div');
    const thumbnailItemWrapper = document.createElement('div');

    thumbnailItem.className = `thumbnail-item bookSpread-${page}`;
    thumbnailItemWrapper.className = 'thumbnail-item__wrapper';

    if (
      (!(is_mobile() && !mobile_test_horizontal()) || is_small_embed_not_mobile()) &&
      !(settings.showSinglePageMode || self.getModel().settings.show_single_page_mode)
    ) {
      thumbnailItem.style.width = `100%`;
    }

    if (
      !isSidePage &&
      (!(is_mobile() && !mobile_test_horizontal()) || is_small_embed_not_mobile())  &&
      !(settings.showSinglePageMode || self.getModel().settings.show_single_page_mode)
    ) {
      loadPages.push(page + 1);
      thumbnailItem.page = page;
      page += 2;
    } else {
      thumbnailItem.page = page;
      page++;
    }

    self.getPageNoPreload(loadPages, function (PageObj) {
      for (const [key, value] of Object.entries(PageObj)) {
        let className = self.getSidebarThumbClassName(key, isBookView);
        
        if (value.getPageReady() || page < 10) {
          insert += `
            <div class="thumb pageNumb-${value.getPageNumber()}" data-page="${value.getPageNumber()}" data-w="${w}" data-h="${h}">
              <div class="thumb-img-ph" style="width: ${w}px; height: ${h}px"></div>
              <div class="thumbnail-number">${value.getPageNumber()}</div>
            </div>`;
        } else {
          insert += `
            <div class="thumb pageNumb-${value.getPageNumber()} converting" data-page="${value.getPageNumber()}">
              <img class="thumb-img ${className}" style="width: ${w}px; height: ${h}px;object-fit: contain;" loading="lazy" 
              pagenum='${value.getPageNumber()}' pageReady='false' src='${APP.PATH_CF_FLIP}assets/images/thumb_ph.png'/>
              <div class="thumbnail-number">${value.getPageNumber()}</div>
            </div>`;
        } 
      }

      thumbnailItemWrapper.insertAdjacentHTML('beforeend', insert);
      thumbnailItem.addEventListener('click', function () {
        self.setCurrentThumbByPage(this.page);
        self.setPage(this.page, 'sidebar.click');

        if (is_mobile() || is_small_embed_not_mobile()) {
          self.hideSidebar();
        }
      });

      thumbnailItem.appendChild(thumbnailItemWrapper);
      loadedThumbnails.push(thumbnailItem);
    });

    return { loadedThumbnails, page };
  };

  this.renderSidebarDima = () => {
    const MAX_WIDTH = 612;
    let fragmentThumbnails = document.createDocumentFragment();
    let imagePercent = is_mobile() && mobile_test_horizontal() ? 7 : 9;
    let pageWidth = parseInt(bookModel.pages_info[0].width) > MAX_WIDTH ? MAX_WIDTH : parseInt(bookModel.pages_info[0].width);
    let pageHeight = parseInt(pageWidth) * (parseInt(bookModel.pages_info[0].height) / parseInt(bookModel.pages_info[0].width));

    let imgWidth = Math.floor((pageWidth / 100) * imagePercent);
    let imgHeight = Math.floor((pageHeight / 100) * imagePercent);
    let prop = imgHeight / imgWidth;
    let w = imgWidth < 95 ? 95 : imgWidth;
    let h = w * prop;
    let page = 1;

    while (page <= +bookModel.pages) {
      const { loadedThumbnails, page: pageNumber } = createThumbnailItem(+page, w, h);
      page = pageNumber;

      loadedThumbnails.forEach(element => fragmentThumbnails.appendChild(element));
    }

    const thumbnailsSibebarInner = elements.sidebar.querySelector('.Book__sidebar');

    if (thumbnailsSibebarInner) {
      thumbnailsSibebarInner.innerHTML = '';
      thumbnailsSibebarInner.appendChild(fragmentThumbnails);
    }

    if (is_mobile()) {
      HELPER.setThumbSizeAndCountMobile(w, h);
    } else if (is_small_embed_not_mobile()) {
      HELPER.setThumbSizeAndCountMobile(w, h);
      HELPER.renderCustomScroll(APP.Layout.getSidebar());
    } else {
      HELPER.renderCustomScroll(APP.Layout.getSidebar());
    }
  };

  this.renderSidebar = function (sidebarWillOpen = false) {
    const MAX_WIDTH = 612;
    let fragmentThumbnails = document.createDocumentFragment();
    let imagePercent = is_mobile() && mobile_test_horizontal() ? 7 : 9;
    let pageWidth = parseInt(bookModel.pages_info[0].width) > MAX_WIDTH ? MAX_WIDTH : parseInt(bookModel.pages_info[0].width);
    let pageHeight = parseInt(pageWidth) * (parseInt(bookModel.pages_info[0].height) / parseInt(bookModel.pages_info[0].width));

    // if (parseInt(bookModel.pages_info[0].width) > parseInt(bookModel.pages_info[0].height)) {
    //   pageWidth = parseInt(bookModel.pages_info[0].height) > MAX_HEIGHT ? MAX_HEIGHT : parseInt(bookModel.pages_info[0].height);
    //   pageHeight = parseInt(bookModel.pages_info[0].width) > MAX_WIDTH ? MAX_WIDTH : parseInt(bookModel.pages_info[0].width);
    // }

    let imgWidth = Math.floor((pageWidth / 100) * imagePercent);
    let imgHeight = Math.floor((pageHeight / 100) * imagePercent);
    let prop = imgHeight / imgWidth;
    let w = imgWidth < 95 ? 95 : imgWidth;
    let h = w * prop;
    let isBookView;
    let thumbnailItem;
    let thumbnailItemWrapper;
    let page = 1;

    while (page <= bookModel.pages) {
      let insert = [],
        loadPages = [page];

      thumbnailItemWrapper = document.createElement('div');
      thumbnailItem = document.createElement('div');
      thumbnailItemWrapper.className = 'thumbnail-item__wrapper';
      thumbnailItem.className = `thumbnail-item bookSpread-${page}`;
      if (
        (!(is_mobile() && !mobile_test_horizontal()) || is_small_embed_not_mobile()) &&
        !(settings.showSinglePageMode || self.getModel().settings.show_single_page_mode)
      ) {
        thumbnailItem.style.width = `100%`;
      }
      if (
        page !== 1 &&
        page !== bookModel.pages &&
        (!(is_mobile() && !mobile_test_horizontal()) || is_small_embed_not_mobile())  &&
        !(settings.showSinglePageMode || self.getModel().settings.show_single_page_mode)
      ) {
        loadPages.push(page + 1);
        thumbnailItem.page = page;
        page += 2;
      } else {
        thumbnailItem.page = page;
        page++;
      }
      isBookView = self.isBookView(page);

      self.getPageNoPreload(loadPages, function (PageObj) {
        for (const [key, value] of Object.entries(PageObj)) {
          let className = self.getSidebarThumbClassName(key, isBookView);
          
          if (value.getPageReady() || page < 10) {
            insert.push(`<div class="thumb pageNumb-${value.getPageNumber()}" data-page="${value.getPageNumber()}" data-w="${w}" data-h="${h}">
                    <div class="thumb-img-ph" style="width: ${w}px; height: ${h}px"></div>
                    <div class="thumbnail-number">${value.getPageNumber()}</div>
                  </div>`);
          } else {
            insert.push(`<div class="thumb pageNumb-${value.getPageNumber()} converting" data-page="${value.getPageNumber()}">
                          <img class="thumb-img ${className}" style="width: ${w}px; height: ${h}px;object-fit: contain;" loading="lazy" 
                          pagenum='${value.getPageNumber()}' pageReady='false' src='${APP.PATH_CF_FLIP}assets/images/thumb_ph.png'/>
                          <div class="thumbnail-number">${value.getPageNumber()}</div>
                        </div>`);
          } 
        }

        if (bookModel.settings.show_right_to_left) {
          insert = insert.reverse();
        }

        insert = insert.join('');

        thumbnailItemWrapper.insertAdjacentHTML('beforeend', insert);
        thumbnailItem.addEventListener('click', function () {
          self.setCurrentThumbByPage(this.page);
          self.setPage(this.page, 'sidebar.click');

          if (is_mobile() || is_small_embed_not_mobile()) {
            self.hideSidebar();
          }
        });
        thumbnailItem.appendChild(thumbnailItemWrapper);
        fragmentThumbnails.appendChild(thumbnailItem);
      });
    }

    const thumbnailsSibebarInner = elements.sidebar.querySelector('.Book__sidebar');

    if (thumbnailsSibebarInner) {
      thumbnailsSibebarInner.innerHTML = '';
      thumbnailsSibebarInner.appendChild(fragmentThumbnails);
    }

    if (is_mobile()) {
      HELPER.setThumbSizeAndCountMobile(w, h);
    } else if (is_small_embed_not_mobile()) {
      HELPER.setThumbSizeAndCountMobile(w, h);
      HELPER.renderCustomScroll(APP.Layout.getSidebar());
    } else {
      HELPER.renderCustomScroll(APP.Layout.getSidebar());
    }
  };

  this.showFooter = function () {
    APP.Layout.getFooter().show();
  };

  const toggleMobileLayout = () => {
    if (!is_mobile()) return;

    if (shouldShowAbsoluteFullscreenButton()) {
      showAbsoluteFullscreenButton();
      return;
    }

    hideAbsoluteFullscreenButton();

    const shouldShowLayout = elements.header.classList.contains('hide') && elements.footer.classList.contains('hide');
    const shouldShowFullscreen = shouldShowLayout && scale <= 1;
    const shouldShowArrows = shouldShowLayout;

    if (elements.headerBorder && !elements.headerBorder.hasAttribute('hidden')) {
      elements.header.classList.toggle('hide', !shouldShowLayout);
    }

    if (elements.footerBorder && !elements.footerBorder.hasAttribute('hidden')) {
      elements.footer.classList.toggle('hide', !shouldShowLayout);
    }

    if (shouldShowFullscreen) {
      fullscreenAnimation();
    }

    elements.navNextPageAbsolute.style.opacity = shouldShowArrows ? '1' : '.5';
    elements.navPrevPageAbsolute.style.opacity = shouldShowArrows ? '1' : '.5';
  };

  const getBookTopOffset = () => {
    return is_mobile() || transparent ? 0 : settings.padding + 5;
  };

  const setElementCursor = (element, cursorType) => {
    element.style.cursor = cursorType;
  };

  let startCoordinates = '';
  let moveCoordinates = { x: 0, y: 0 };
  let xMove = 0;
  let yMove = 0;
  let mousePressedMoveInZoomTime = 0;
  let mouseMoveLeftRightOutside = 0;
  let mouseMoveTopBottomOutside = 0;

  const step = function (startPos, velocity, direction) {
    //Uses trig to work out the next positon of the circle
    return {
      x: startPos.x + velocity * Math.cos(direction),
      y: startPos.y + velocity * Math.sin(direction),
    };
  };

  const tick = function (startPos, velocity, direction) {
    const finalPosMouseSim = step(startPos, velocity, direction);

    xMove = startPos.x - finalPosMouseSim.x + moveCoordinates.x;
    yMove = startPos.y - finalPosMouseSim.y + moveCoordinates.y;

    const coordinates = self.scaledTransformWithTrans(xMove, yMove, `transform 400ms 0s ease-out`);

    xMove = coordinates.x;
    yMove = coordinates.y;
    moveCoordinates = { x: xMove, y: yMove };
  };
  
  //mouseHandlers
  const mouseDownHandler = function (e) {
    setElementCursor(elements.innerChild, 'grabbing');

    if (pageFlip.isBookZoomed() && (e.target.classList && !e.target.classList.contains('Note__marker'))) {
      e.preventDefault();
      mousePressedMoveInZoomTime = Date.now();
      startCoordinates = { x: e.x, y: e.y };
      if (elements.bookWrapper.classList.contains('Book__wrapper--transition'))
        elements.bookWrapper.classList.remove('Book__wrapper--transition');

      const root = APP.Layout.getRoot();
      const rootRect = root.getBoundingClientRect();

      //moved from move
      let countOfPages =
        (currentPage === 1 || currentPage == bookModel.pages) &&
        pageFlip.getOrientation() == 'landscape'
          ? 2
          : 1;
      mouseMoveLeftRightOutside = scaledWidth / countOfPages - rootRect.width;
      mouseMoveLeftRightOutside =
        mouseMoveLeftRightOutside > 0 ? Math.round(mouseMoveLeftRightOutside / 2) : 0;

      mouseMoveTopBottomOutside = scaledHeight - rootRect.height + settings.menuHeight;
      if (is_mobile()) {
        mouseMoveTopBottomOutside = scaledHeight - rootRect.height + 0;
      }
      mouseMoveTopBottomOutside =
        mouseMoveTopBottomOutside > 0 ? Math.round(mouseMoveTopBottomOutside / 2) : 0;
    }
  };

  const mouseMoveHandlerWhenZoomed = function (e) {
    xMove = -1 * (startCoordinates.x - e.x) + moveCoordinates.x;
    yMove = -1 * (startCoordinates.y - e.y) + moveCoordinates.y;
    (Math.abs(e.movementX) > 1 || Math.abs(e.movementY) > 1) &&
      (mousePressedMoveInZoomTime = Date.now());
    Math.abs(xMove) > mouseMoveLeftRightOutside &&
      (xMove = xMove > 0 ? mouseMoveLeftRightOutside : -1 * mouseMoveLeftRightOutside);
    Math.abs(yMove) > mouseMoveTopBottomOutside &&
      (yMove = yMove > 0 ? mouseMoveTopBottomOutside : -1 * mouseMoveTopBottomOutside);
    elements.bookWrapper.style.transform = `translate(${xMove}px,${yMove}px)`;
    elements.bookWrapper.style.transition = '';
  };

  const mouseMoveHandler = function (e) {
    //  e.preventDefault();
    startCoordinates && pageFlip.isBookZoomed() && mouseMoveHandlerWhenZoomed(e);
    oLogs.lastActionTime = Date.now();
  };

  const mouseUpHandler = function (e) {
    if (audio && audio.paused) {
      audio.pause();
    }

    if (!startCoordinates) {
      setElementCursor(elements.innerChild, 'grab');
      return;
    }

    e.preventDefault();

    const tmpStartCoordinates = startCoordinates;
    const deltaTime = Date.now() - mousePressedMoveInZoomTime;

    startCoordinates = '';
    moveCoordinates = { x: xMove, y: yMove };

    if (deltaTime > 30) {
      setElementCursor(elements.innerChild, 'grab');
    } else {
      elements.bookWrapper.style.transition = 'transform 400ms 0s ease-out';

      setTimeout(function () {
        setElementCursor(elements.innerChild, 'grab');
      }, 400);

      const velocity = (Math.sqrt((tmpStartCoordinates.x - e.x + 10) ** 2 + (tmpStartCoordinates.y - e.y + 10) ** 2)) * 2;
      const direction = Math.atan2(tmpStartCoordinates.y - e.y + 10, tmpStartCoordinates.x - e.x + 10); //Uses atan2 to find the angle between the circle and the mouse pointer
      
      tick(tmpStartCoordinates, velocity, direction);
    }
  };

  const doubleClickHandler = function() {
    return;
    if (is_mobile() || !bookModel.settings.show_zoom_button) return;

    const isZoomed = scale > 0;
    
    changeMenuButtonsOnZoom(isZoomed);
    
    self.scale(isZoomed ? 0 : 2);
  };

  //Touch handlers
  let isMoving = false;
  let canBeTap = false;
  let isZooming = false;
  let startCoordinatesTime = false;

  let zoomPoint = { x: 0, y: 0 };
  let targetCoordinates = { x: 0, y: 0 };
  let offsetCoordinates = { x: 0, y: 0 };
  
  let scaleFactor = 1;
  let initialScale = 1;
  let currentScale = 1;
  let pointDistance = 0;
  let initialDistance = 0;
  let lastTapTime = 0;
  let lastTapEvent = null;

  const getTouchDistance = (event) => {
    if (event.touches.length === 1) {
      return 0;
      const touch = event.touches[0];

      return { x: touch.pageX, y: touch.pageY };
    }

    if (event.touches.length !== 2) return 0;
  
    const touch1 = event.touches[0];
    const touch2 = event.touches[1];
  
    const dx = touch2.pageX - touch1.pageX;
    const dy = touch2.pageY - touch1.pageY;
  
    return Math.sqrt(Math.pow(dx, 2) + Math.pow(dy, 2));
  };

  const getTouchMidpoint = (event) => {
    if (event.touches.length === 1) {
      const touch = event.touches[0];

      return {
        x: Math.round(touch.pageX),
        y: Math.round(touch.pageY),
      };
    }

    if (event.touches.length !== 2) return { x: 0, y: 0 };
  
    const touch1 = event.touches[0];
    const touch2 = event.touches[1];
  
    return {
      x: Math.round((touch1.pageX + touch2.pageX) / 2),
      y: Math.round((touch1.pageY + touch2.pageY) / 2),
    };
  };

  const isDoubleTapEvent = () => {
    if (lastTapTime === false) {
      lastTapTime = 0;
      return false;
    }

    let currentTime = new Date().getTime();
    let tapLength = currentTime - lastTapTime;
      
    if (tapLength < 300 && tapLength > 0) {
      return true;
    }

    lastTapTime = currentTime;
    return false;
  };

  const touchStartHandler = function (e) {
    const isButtonClick = e.target.localName === 'button';
    const isNavAreaCkick = [elements.navAreaNextPage, elements.navAreaPrevPage].indexOf(e.target) !== -1;
    const isHotspotAreaClick = (e.path || e.composedPath()).find(el => el && el.classList && el.classList.contains('hotspot-area'));

    if (e.touches.length === 1 && (isButtonClick || isNavAreaCkick || isHotspotAreaClick)) {
      canBeTap = false;
      return;
    }

    lastTapEvent = e;
    if (e.touches.length != 1) {
      lastTapTime = false;
    }

    if (!canBeZoomedOnThisElement(e)) return stopEvent(e);

    if (isZooming && e.touches.length === 2) return;

    if (e.touches.length === 2) {
      canBeTap = false;
      isZooming = true;
      startCoordinates = '';

      initialDistance = getTouchDistance(e);
      zoomPoint = getTouchMidpoint(e);

      initialScale = currentScale;
      pointDistance = initialDistance;
    } else {
      setElementCursor(elements.inner, 'grabbing');
      canBeTap = true;

      if (pageFlip.isBookZoomed()) {
        startCoordinatesTime = performance.now(); 
        startCoordinates = { x: e.touches[0].pageX, y: e.touches[0].pageY };
      }
    }
  };

  const touchMoveHandler = function (event) {
    isMoving = true;
    canBeTap = false;

    if (pointDistance) {
      timerForSVGLoading = null;

      if (event.touches.length !== 2) return;

      event.preventDefault();
      event.stopPropagation();
      
      const currentDistance = getTouchDistance(event);

      scaleFactor = currentDistance / initialDistance;

      if (currentDistance < pointDistance) {
        if (currentScale < settings.MOBILE_MIN_SCALE) return;

        pointDistance = currentDistance;
        self.scaleOutPoint(event);
      }
      
      if (currentDistance > pointDistance) {
        if (currentScale >= settings.MOBILE_MAX_SCALE) return;

        pointDistance = currentDistance;
        self.scaleInPoint(event);
      }
    } else {
      if (!startCoordinates) return;
      
      elements.bookWrapper.classList.remove('Book__wrapper--transition');
      
      xMove = -1 * (startCoordinates.x - event.touches[0].pageX) + moveCoordinates.x;
      yMove = -1 * (startCoordinates.y - event.touches[0].pageY) + moveCoordinates.y;
      
      const coordinates = self.scaledTransform(xMove, yMove);
      
      xMove = coordinates.x;
      yMove = coordinates.y;
    }
  };

  const getDistanceBetweenTwoPoint = function (t, e) {
    return null === t || null === e
      ? 1 / 0
      : Math.sqrt(Math.pow(e.x - t.x, 2) + Math.pow(e.y - t.y, 2));
  };

  const touchEndHandler_akcjaLazyStop = function (e) {
    let endP = { x: e.changedTouches[0].clientX, y: e.changedTouches[0].clientY };
    let difT = performance.now() - startCoordinatesTime;
    let difX = endP.x - startCoordinates.x;
    let difY = endP.y - startCoordinates.y;
    let szybkosc = getDistanceBetweenTwoPoint(startCoordinates, endP) / difT;

    let finalEndP = { x: endP.x + (szybkosc / 10) * difX, y: endP.y + (szybkosc / 10) * difY };

    xMove = -1 * (startCoordinates.x - finalEndP.x) + moveCoordinates.x;
    yMove = -1 * (startCoordinates.y - finalEndP.y) + moveCoordinates.y;

    if (szybkosc > 0.78) {
      let timeAnim = szybkosc / 2 > 0.5 ? 0.5 : szybkosc / 2;

      let coordinates = self.scaledTransformAnimated(xMove, yMove, timeAnim);
      xMove = coordinates.x;
      yMove = coordinates.y;
      moveCoordinates = { x: xMove, y: yMove };
    }
  };

  const touchEndHandler = function (e) {
    // TODO: zoom do punktu po double tap.
    
    // if (isDoubleTapEvent()) {
    //   const innerBookRect = elements.inner.getBoundingClientRect();
      
    //   console.log(scaleFactor);
    //   scaleFactor = scaleFactor <= 1 ? 1.5 : 1;
    //   console.log(scaleFactor);
    //   zoomPoint = { 
    //     x: innerBookRect.width / 2, 
    //     y: innerBookRect.height / 2 
    //   };
    //   moveCoordinates = { x: 0, y: 0 };

    //   // elements.inner.style.transformOrigin = `${zoomPoint.x}px ${zoomPoint.y}px`;
    //   elements.inner.style.transition = `transform 600ms cubic-bezier(0.2, 0.6, 0.35, 1) 0s`;

    //   if (scaleFactor === 1) {
    //     self.scaleOutPoint();
    //   } else {
    //     self.scaleInPoint();
    //   }
      
    //   return new Promise(resolve => {
    //     const interval = setInterval(() => {
    //       if (!isBookInnerTransitionEnd) return; 
          
    //       if (is_mobile()) {
    //         elements.inner.style.transformOrigin = '';
    //         elements.inner.style.transition = 'none';
            
    //         // zoomPoint = { x: 0, y: 0 };
    //       }
  
    //       clearInterval(interval);
    //       resolve();
    //     }, 10);
    //   });
    // }

    if (e.touches.length == 0 && canBeTap && !transparent) {
      toggleMobileLayout();
    }

    if (startCoordinates) {
      moveCoordinates = { x: xMove, y: yMove };
      touchEndHandler_akcjaLazyStop(e);
      startCoordinates = '';
    }

    if (pointDistance) {
      pointDistance = 0;
    }

    if (isZooming) {
      const scaledPages = getScaledPages();
      const scaleMin = is_mobile() ? settings.MOBILE_MIN_SCALE : settings.MIN_SCALE;

      if (scale > scaleMin) {
        for (const i in scaledPages) {
          scaledPages[i].hotspotsOff();
          scaledPages[i].setVectorForZoom();
          scaledPages[i].removeShadow();
        }
      } else {
        for (let i in scaledPages) {
          scaledPages[i].hotspotsOn();
          scaledPages[i].clearVectorImage();
        }
      }
    }

    isZooming = false;
    isMoving = false;
  };

  // scaling
  let scale = 0;
  let prevScale = 0;
  let scaledWidth = 0;
  let scaledHeight = 0;
  let prevScaledWidth = 0;
  let timerForSVGLoading = null;
  let scaleMobileStartInitialized = false;

  this.isBookZoomed = function () {
    return pageFlip.isBookZoomed();
  }

  const handleScaleMobileStart = () => {
    scaleMobileStartInitialized = true;

    clearTimeout(timerBadQualityImages);

    const scaledPages = getScaledPages();

    for (const i in scaledPages) {
      scaledPages[i].hotspotsOff();
      scaledPages[i].clearVectorImage();
      scaledPages[i].removeShadow();
    }

    bookNode.style = `width: ${window.innerWidth}; height: ${window.innerHeight}px; margin-top: unset;`;

    self.changeArrows(true);
    pageFlip.bookZoom(true);
    hideAbsoluteFullscreenButton();
  };

  const handleScaleMobileEnd = () => {
    scaleMobileStartInitialized = false;

    clearTimeout(timerBadQualityImages);

    bookNode.style = `width: ${window.innerWidth}; height: ${window.innerHeight}px; margin-top: unset;`;
    self.changeArrows(false);

    // setTimeout(() => {
    //   if (pageFlip) pageFlip.bookZoom(false);
    // }, 500);
    pageFlip.bookZoom(false);

    if (shouldShowAbsoluteFullscreenButton()) {
      showAbsoluteFullscreenButton();
    } else {
      hideAbsoluteFullscreenButton();
    }

    setElementCursor(elements.inner, 'default');

    moveCoordinates = { x: 0, y: 0 };
    elements.bookWrapper.style.transform = 'unset';

    const scaledPages = getScaledPages();

    for (const i in scaledPages) {
      scaledPages[i].hotspotsOn();

      if (settings.loadSvgOnZoom) {
        scaledPages[i].clearVectorImage();
      }

      if (!settings.shadowsInZoom) {
        scaledPages[i].addShadow();
      }
    }
  };

  const scaleMobile = function (value) {
    const scaleValue = Math.min(Math.max(initialScale * scaleFactor, settings.MOBILE_MIN_SCALE), settings.MOBILE_MAX_SCALE);
    prevScale = currentScale;

    if (scaleValue !== prevScale) {
      scale = scaleValue;
      currentScale = scaleValue;

      elements.inner.style.transform = `scale(${scaleValue})`;
      
      scaledWidth = parseInt(elements.innerChild.style.width) * scaleValue;
      scaledHeight = parseInt(elements.innerChild.style.height) * scaleValue;
    }

    if (value > settings.MIN_SCALE) {
      if (!scaleMobileStartInitialized) handleScaleMobileStart();
      return;
    }

    if (prevScale > settings.MIN_SCALE) {
      if (scaleMobileStartInitialized) handleScaleMobileEnd();
    }
  };

  this.svgWrapperSetSizes = function (v) {
    if (!v) v = scale;

    const svgWrapper = document.querySelector('.svg__wrapper');
    
    if (!svgWrapper) {
      this.svgWrapperHandler(v);
      return;
    }

    const innerChildRect = elements.innerChild.getBoundingClientRect();
    const bookWrapperRect = elements.bookWrapper.getBoundingClientRect();
    let scaleValue = v / 4 + 1;
    scaleValue = scaleValue > 6 ? 6 : scaleValue;
    scaleValue = scaleValue < 1 ? 1 : scaleValue;
    // svgWrapper.hidden = true;
    let countOfPages = (currentPage === 1 || currentPage == bookModel.pages) && pageFlip.getOrientation() == 'landscape'
      ? 2
      : 1;

    const newWidth = scaleValue == 1 ? parseInt(elements.innerChild.clientWidth) : parseInt(elements.innerChild.clientWidth) * scaleValue;
    const newHeight = scaleValue == 1 ? parseInt(elements.innerChild.clientHeight) : parseInt(elements.innerChild.clientHeight) * scaleValue;
    const scaledWidth = parseInt(elements.innerChild.clientWidth) * scaleValue;
    const scaledHeight = parseInt(elements.innerChild.clientHeight) * scaleValue;

    mouseMoveLeftRightOutside = scaledWidth / countOfPages - window.innerWidth;
    mouseMoveLeftRightOutside = mouseMoveLeftRightOutside > 0 ? Math.round(mouseMoveLeftRightOutside / 2) : 0;

    mouseMoveTopBottomOutside = scaledHeight - window.innerHeight + settings.menuHeight;
    mouseMoveTopBottomOutside = mouseMoveTopBottomOutside > 0 ? Math.round(mouseMoveTopBottomOutside / 2) : 0;
    
    let newTopPos = mouseMoveTopBottomOutside == 0 ? innerChildRect.top - bookWrapperRect.top : -mouseMoveTopBottomOutside;
    let newLeftPos = mouseMoveLeftRightOutside == 0 ? innerChildRect.left - bookWrapperRect.left : -mouseMoveLeftRightOutside;
    newTopPos = !this.isBookView(currentPage) ? innerChildRect.top - bookWrapperRect.top : newTopPos;
    newLeftPos = !this.isBookView(currentPage) ? innerChildRect.left - bookWrapperRect.left : newLeftPos;

    svgWrapper.style.width = `${newWidth}px`;
    svgWrapper.style.height = `${newHeight}px`;
    svgWrapper.style.top = `${newTopPos}px`;
    svgWrapper.style.left = `${newLeftPos}px`;

    svgWrapper.removeAttribute('hidden');
  }

  this.svgWrapperHandler = function (v) {
    let svgWrapper = document.querySelector('.svg__wrapper');
    const innerChildRect = elements.innerChild.getBoundingClientRect();
    let scaleValue = v / 4 + 1;

    if (!svgWrapper) {
      svgWrapper = document.createElement('div');
      const svgRightPage = document.createElement('div');
      const svgLeftPage = document.createElement('div');

      svgWrapper.className = 'svg__wrapper';
      svgRightPage.className = 'svg__page svg__page--right';
      svgLeftPage.className = 'svg__page svg__page--left';

      svgWrapper.style = `
        width: ${elements.innerChild.clientWidth * (v == 0 ? 1 : scaleValue)}px;
        height: ${elements.innerChild.clientHeight * (v == 0 ? 1 : scaleValue)}px;
        position: absolute;
        top: ${innerChildRect.top || 0}px;
        bottom: 0;
        left: ${innerChildRect.left || 0}px;
        right: 0;
        pointer-events: none;
      `;

      svgWrapper.appendChild(svgLeftPage);
      svgWrapper.appendChild(svgRightPage);
      elements.bookWrapper.insertAdjacentElement('beforeend', svgWrapper);

      elements.inner.addEventListener('transitionstart', () => {
        // svgWrapper.hidden = true;
      });

      elements.inner.addEventListener('transitionend', () => {
        this.svgWrapperSetSizes(v);
      });
    }

    const scaledPages = getScaledPages();
    svgWrapper.classList.remove('show');

    Object.values(scaledPages).forEach(page => {
      page.getPage().querySelector('.front').classList.remove('hide');
    });

    setTimeout(() => {
      Object.values(scaledPages).forEach(page => {
        page.getPage().querySelector('.front').classList.add('hide');
      });
      
      svgWrapper.classList.add('show');
    }, 100);

    this.svgWrapperSetSizes(v);
  }

  const resetScale = () => {
    isBookInnerTransitionEnd = false;

    scale = 0;
    // initialScale = 0;
    currentScale = 0;
    initialDistance = 0;
    scaleFactor = 1;

    if (is_mobile()) {
      elements.inner.style.transformOrigin = `${zoomPoint.x}px ${zoomPoint.y}px`;
      elements.inner.style.transition = `transform 600ms cubic-bezier(0.2, 0.6, 0.35, 1) 0s`;
      
      self.scaleOutPoint();
      scaleMobile(0);
    } else {
      self.scale(0);
    }

    return new Promise(resolve => {
      const interval = setInterval(() => {
        if (!isBookInnerTransitionEnd) return; 
        
        if (is_mobile()) {
          elements.inner.style.transformOrigin = '';
          elements.inner.style.transition = 'none';
          
          zoomPoint = { x: 0, y: 0 };
        }

        clearInterval(interval);
        resolve();
      }, 10);
    });
  };

  this.scale = function (v, e) {
    v = Math.max(0, v);

    if (is_safari() && !is_mobile()) {
      if (v >= settings.MIN_SCALE && v <= settings.MAX_SCALE) {
        this.svgWrapperHandler(v);
      }
    }

    if (is_mobile()) {
      return scaleMobile(v, e);
    }
    
    prevScale = scale;

    if (v >= settings.MIN_SCALE && v <= settings.MAX_SCALE) {
      scale = v;
      const scaleToVal = scale / 4 + 1;

      if (settings.Scaller) {
        elements.navScaler.navScalerR.setValue(v * 5);
      }

      prevScaledWidth = scaledWidth;
      scaledWidth = parseInt(elements.innerChild.style.width) * scaleToVal;
      scaledHeight = parseInt(elements.innerChild.style.height) * scaleToVal;

      if (moveCoordinates.x !== 0 || moveCoordinates.y !== 0) {
        const x = Math.round(moveCoordinates.x * (scaledWidth / prevScaledWidth));
        const y = Math.round(moveCoordinates.y * (scaledWidth / prevScaledWidth));

        moveCoordinates = self.scaledTransformMarcin(x, y);
        //scaledTransform <- tu bylo
      }

      if (APP.Note && APP.Note.scaleNote) {
        APP.Note.scaleNote(scaleToVal);
      }

      const scalestyle = is_safari()
        ? `scale(${scaleToVal}, ${scaleToVal})`
        : `scale3d(${scaleToVal}, ${scaleToVal}, ${scaleToVal})`;

      elements.inner.style.transform = scalestyle;
      elements.inner.style.transition = `transform 600ms cubic-bezier(0.2, 0.6, 0.35, 1) 0s`;
    }

    clearTimeout(timerBadQualityImages);

    if (v > 0) {
      elements.inner.style.paddingTop = '0px';

      if (transparent) {
        bookNode.style = `width: ${window.innerWidth}; height: ${window.innerHeight}px; margin-top: unset; `;
      } else {
        bookNode.style = `width: ${window.innerWidth}; height: ${window.innerHeight - settings.menuHeight}px; margin-top: ${settings.menuHeight / 2}px;`;
      }

      self.changeArrows(true);
      pageFlip.bookZoom(true);
      hideAbsoluteFullscreenButton();

      if (!is_touchscreen() && elements.inner.style.cursor !== 'grab') {
        setElementCursor(elements.inner, 'grab');
      }

      if (elements.bookWrapper.style.transform == 'unset') {
        elements.bookWrapper.style.transform = 'translate(0px, 0px)';
      }

      const scaledPages = getScaledPages();

      for (const i in scaledPages) {
        if (settings.loadSvgOnZoom) {
          if (is_mobile()) {
            scaledPages[i].hotspotsOff();

            timerForSVGLoading = setTimeout(() => {
              scaledPages[i].hotspotsOn();
              scaledPages[i].setVectorForZoom();
            }, 300);
          } else {
            scaledPages[i].setVectorForZoom();
          }
        }

        if (!settings.shadowsInZoom) {
          scaledPages[i].removeShadow();
        }
      }

      if (elements.zoomIn) {
        //@Marcin - for mobile
        elements.zoomIn.classList.add('Book__zoom--out');

        if (elements.zoomIn.dataset && elements.zoomIn.dataset.type === BUTTON_ICON_TYPES.INLINE) {
          elements.zoomIn.innerHTML = HELPER.buttonsIcons.closeIcon;
        }
      }

      elements.navScaler.hidden = false;
    } else {
      v = 0;
      elements.inner.style.paddingTop = `${getBookTopOffset()}px`;

      const bookSizes = self.getBookSize();
      self.setPageImageByBookSize(bookSizes.bookHeight, true);
      self.changeArrows(false);

      if (shouldShowAbsoluteFullscreenButton()) {
        showAbsoluteFullscreenButton();
      } else {
        hideAbsoluteFullscreenButton();
      }

      setTimeout(() => {
        bookNode.style = `width: ${window.innerWidth}; height: ${window.innerHeight}px; margin-top: unset;`;
        if (pageFlip) pageFlip.bookZoom(false);
      }, 500);
      
      setElementCursor(elements.inner, 'default');

      moveCoordinates = { x: 0, y: 0 };
      elements.bookWrapper.style.transform = 'unset';

      const scaledPages = getScaledPages();

      for (const i in scaledPages) {
        scaledPages[i].hotspotsOn();

        if (settings.loadSvgOnZoom) {
          scaledPages[i].clearVectorImage();
        }

        if (!settings.shadowsInZoom) {
          scaledPages[i].addShadow();
        }
      }
    }
  };

  this.changeArrows = function (relative) {
    let _currentPage = currentPage;

    if (_currentPage == -1) {
      _currentPage = getCurrentPageIfPageIsNegative();
    }

    if (_currentPage === 0) {
      _currentPage = 1;
    }

    if (!is_horisontal() ||(is_horisontal() && !forceOnePage)) { // zamiast pierwszego is_hor() pageFlip && pageFlip.getOrientation() === 'landscape'
      if (_currentPage === 1) {
        if (_currentPage != bookModel.pages) {
          elements.navNextPage.hidden = relative;
          elements.navAreaNextPage.hidden = relative;

          if (is_mobile()) {
            elements.navNextPageAbsolute.hidden = false;
          } else {
            elements.navNextPageAbsolute.hidden = !relative;
          }
        }
      } else if (_currentPage == bookModel.pages || (self.isDoublePage() && _currentPage + 1 == bookModel.pages)) {
        elements.navPrevPage.hidden = relative;
        elements.navAreaPrevPage.hidden = relative;

        if (is_mobile()) {
          elements.navPrevPageAbsolute.hidden = false;
        } else {
          elements.navPrevPageAbsolute.hidden = !relative;
        }
      } else {
        elements.navAreaNextPage.hidden = relative;
        elements.navAreaPrevPage.hidden = relative;
        elements.navNextPage.hidden = relative;
        elements.navPrevPage.hidden = relative;

        if (is_mobile()) {
          elements.navNextPageAbsolute.hidden = false;
          elements.navPrevPageAbsolute.hidden = false;
        } else {
          elements.navNextPageAbsolute.hidden = !relative;
          elements.navPrevPageAbsolute.hidden = !relative;
        }
      }
    }

    if (!is_horisontal() && forceOnePage) {
      elements.navAreaNextPage.hidden = relative;
      elements.navAreaPrevPage.hidden = relative;
    }

    const waitForCurrentPageInit = setInterval(() => {
      if (_currentPage != 0) {
        clearInterval(waitForCurrentPageInit);
        blockNavAreaByPage(_currentPage);
      }
    }, 100);
  };

  const getScaledPages = function () {
    const pages = [currentPage];
    let scaledPages;

    if (pageFlip && pageFlip.getOrientation() == 'landscape') {
      if (currentPage !== 1 && currentPage !== bookModel.pages) {
        if (currentPage % 2 == 0) {
          pages.push(currentPage + 1);
        } else {
          pages.push(currentPage - 1);
        }
      }
    }

    self.getPage(pages, function (PagesObj) {
      scaledPages = PagesObj;
    });

    return scaledPages;
  };

  this.scaledTransformMarcin = function (x, y) {
    let leftRightOutside = scaledWidth - window.innerWidth;
    leftRightOutside = leftRightOutside > 0 ? Math.round(leftRightOutside / 2) : 0;
    let topBottomOutside = scaledHeight - window.innerHeight + settings.menuHeight;
    topBottomOutside = topBottomOutside > 0 ? Math.round(topBottomOutside / 2) : 0;
    // console_log([scaledWidth, window.innerWidth]);
    if (Math.abs(x) > leftRightOutside) {
      x = x > 0 ? leftRightOutside : -1 * leftRightOutside;
    }
    if (Math.abs(y) > topBottomOutside) {
      y = y > 0 ? topBottomOutside : -1 * topBottomOutside;
    }

    let style = `translate(${x}px,${y}px)`;
    elements.bookWrapper.style.transform = style;
    elements.bookWrapper.style.transition = `transform 600ms cubic-bezier(0.2, 0.6, 0.35, 1) 0s`;

    return { x, y };
  };

  this.scaledTransform = function (x, y) {
    const root = APP.Layout.getRoot();
    const rootRect = root.getBoundingClientRect();

    let countOfPages =
      (currentPage === 1 || currentPage == bookModel.pages) &&
      pageFlip.getOrientation() == 'landscape'
        ? 2
        : 1;

    let leftRightOutside = scaledWidth / countOfPages - rootRect.width;
    leftRightOutside = leftRightOutside > 0 ? Math.round(leftRightOutside / 2) : 0;
    //o.005

    let topBottomOutside = scaledHeight - rootRect.height + settings.menuHeight;

    if (is_mobile()) {
      topBottomOutside = scaledHeight - rootRect.height + 0;
    }

    topBottomOutside = topBottomOutside > 0 ? Math.round(topBottomOutside / 2) : 0;

    if (Math.abs(x) > leftRightOutside) {
      x = x > 0 ? leftRightOutside : -1 * leftRightOutside;
    }

    if (Math.abs(y) > topBottomOutside) {
      y = y > 0 ? topBottomOutside : -1 * topBottomOutside;
    }

    x = Math.round(x);
    y = Math.round(y);
    //0.005 -> 0.009

    elements.bookWrapper.style.transform = `translate(${x}px,${y}px)`;
    elements.bookWrapper.style.transition = '';
    //25-35

    return { x, y };
  };

  this.scaledTransformWhileZoomMoving = function (x, y, leftRightOutside, topBottomOutside) {
    Math.abs(x) > leftRightOutside && (x = x > 0 ? leftRightOutside : -1 * leftRightOutside);
    Math.abs(y) > topBottomOutside && (y = y > 0 ? topBottomOutside : -1 * topBottomOutside);
    elements.bookWrapper.style.transform = `translate(${x}px,${y}px)`;
    elements.bookWrapper.style.transition = '';
    return { x, y };
  };

  this.scaledTransformWithTrans = function (x, y, trans) {
    const root = APP.Layout.getRoot();
    const rootRect = root.getBoundingClientRect();

    let countOfPages =
      (currentPage === 1 || currentPage == bookModel.pages) &&
      pageFlip.getOrientation() == 'landscape'
        ? 2
        : 1;

    let leftRightOutside = scaledWidth / countOfPages - rootRect.width;
    leftRightOutside = leftRightOutside > 0 ? Math.round(leftRightOutside / 2) : 0;

    let topBottomOutside = scaledHeight - rootRect.height + settings.menuHeight;
    if (is_mobile()) {
      topBottomOutside = scaledHeight - rootRect.height + 0;
    }
    topBottomOutside = topBottomOutside > 0 ? Math.round(topBottomOutside / 2) : 0;

    if (Math.abs(x) > leftRightOutside) {
      x = x > 0 ? leftRightOutside : -1 * leftRightOutside;
    }
    if (Math.abs(y) > topBottomOutside) {
      y = y > 0 ? topBottomOutside : -1 * topBottomOutside;
    }
    x = Math.round(x);
    y = Math.round(y);
    let style = `translate(${x}px,${y}px)`;

    elements.bookWrapper.style.transform = style;
    elements.bookWrapper.style.transition = trans;
    return { x, y };
  };

  this.scaledTransformAnimated = function (x, y, time) {
    let countOfPages =
      (currentPage === 1 || currentPage == bookModel.pages) &&
      pageFlip.getOrientation() == 'landscape'
        ? 2
        : 1;

    let leftRightOutside = scaledWidth / countOfPages - window.innerWidth;
    leftRightOutside = leftRightOutside > 0 ? Math.round(leftRightOutside / 2) : 0;

    let topBottomOutside = scaledHeight - window.innerHeight + settings.menuHeight;
    if (is_mobile()) {
      topBottomOutside = scaledHeight - window.innerHeight + 0;
    }
    topBottomOutside = topBottomOutside > 0 ? Math.round(topBottomOutside / 2) : 0;

    if (Math.abs(x) > leftRightOutside) {
      x = x > 0 ? leftRightOutside : -1 * leftRightOutside;
    }
    if (Math.abs(y) > topBottomOutside) {
      y = y > 0 ? topBottomOutside : -1 * topBottomOutside;
    }
    x = Math.round(x);
    y = Math.round(y);
    let style = `translate(${x}px,${y}px)`;

    elements.bookWrapper.style.transform = style;
    elements.bookWrapper.style.transition = `transform ${time}s`;
    elements.bookWrapper.style.transitionTimingFunction = 'ease-out';

    return { x, y };
  };

  this.scaleInPoint = function (e) {
    scale = Math.max(1, scale);

    let x = (window.innerWidth * scale - window.innerWidth) / 2;
    let y = (window.innerHeight * scale - window.innerHeight) / 2;
    const a = x + zoomPoint.x - moveCoordinates.x;
    const b = y + zoomPoint.y - moveCoordinates.y;

    self.scaleIn();

    const sc = parseFloat(scale.toFixed(3));
    const pr = parseFloat(prevScale.toFixed(3));

    targetCoordinates.x = (a / pr) * sc;
    targetCoordinates.y = (b / pr) * sc;

    x = (window.innerWidth * sc - window.innerWidth) / 2;
    y = (window.innerHeight * sc - window.innerHeight) / 2;

    offsetCoordinates.x = x + zoomPoint.x - targetCoordinates.x;
    offsetCoordinates.y = y + zoomPoint.y - targetCoordinates.y;

    // let leftRightOutside = scaledWidth - window.innerWidth;
    // leftRightOutside = leftRightOutside > 0 ? Math.round(leftRightOutside / 2) : 0;
    // let topBottomOutside = scaledHeight - window.innerHeight + settings.menuHeight;
    // topBottomOutside = topBottomOutside > 0 ? Math.round(topBottomOutside / 2) : 0;

    elements.bookWrapper.style.transform = `translate(${offsetCoordinates.x}px, ${offsetCoordinates.y}px)`;
    elements.bookWrapper.style.transition = ``; //transform 600ms cubic-bezier(0.2, 0.6, 0.35, 1) 0s`;

    moveCoordinates.x = offsetCoordinates.x;
    moveCoordinates.y = offsetCoordinates.y;
  };

  this.scaleOutPoint = function () {
    scale = Math.max(1, scale);

    let x = (window.innerWidth * scale - window.innerWidth) / 2;
    let y = (window.innerHeight * scale - window.innerHeight) / 2;
    const a = x + zoomPoint.x - moveCoordinates.x;
    const b = y + zoomPoint.y - moveCoordinates.y;

    self.scaleOut();

    const sc = parseFloat(scale.toFixed(3));
    const pr = parseFloat(prevScale.toFixed(3));

    targetCoordinates.x = (a / pr) * sc;
    targetCoordinates.y = (b / pr) * sc;

    x = (window.innerWidth * sc - window.innerWidth) / 2;
    y = (window.innerHeight * sc - window.innerHeight) / 2;

    offsetCoordinates.x = x + zoomPoint.x - targetCoordinates.x;
    offsetCoordinates.y = y + zoomPoint.y - targetCoordinates.y;

    let leftRightOutside = scaledWidth - window.innerWidth;
    leftRightOutside = leftRightOutside > 0 ? Math.round(leftRightOutside / 2) : 0;

    let topBottomOutside = scaledHeight - window.innerHeight + settings.menuHeight;
    topBottomOutside = topBottomOutside > 0 ? Math.round(topBottomOutside / 2) : 0;

    if (Math.abs(offsetCoordinates.x) > leftRightOutside) {
      offsetCoordinates.x = offsetCoordinates.x > 0 ? leftRightOutside : -1 * leftRightOutside;
    }

    if (Math.abs(offsetCoordinates.y) > topBottomOutside) {
      offsetCoordinates.y = offsetCoordinates.y > 0 ? topBottomOutside : -1 * topBottomOutside;
    }

    elements.bookWrapper.style.transform = `translate(${offsetCoordinates.x}px, ${offsetCoordinates.y}px)`;
    elements.bookWrapper.style.transition = ``; //transform 600ms cubic-bezier(0.2, 0.6, 0.35, 1) 0s`;

    moveCoordinates.x = offsetCoordinates.x;
    moveCoordinates.y = offsetCoordinates.y;
  };

  this.scaleOut = function (e, step = 1) {
    if (scale === 1) {
      step = 1;
    }

    self.scale(scale - step, e);
    return stopEvent(e);
  };

  this.scaleIn = function (e, step = 1) {
    self.scale(scale + step, e);
    return stopEvent(e);
  };

  this.scaleInStep = function (e) {
    self.scaleIn(e, 4);
  };

  this.scaleOutStep = function (e) {
    self.scaleOut(e, 4);
  };

  this.getModel = function () {
    return bookModel;
  };

  var _worker = null,
    URL = window.URL || window.webkitURL,
    _preloadImagesByWebWorkerCache = {};
  this.getFromWebWorkerCache = function (url) {
    //APP.log('getFromWebWorkerCache', url);
    if (!settings.PreloadImages || !window.Worker) return url;
    if (isset(_preloadImagesByWebWorkerCache[url])) return _preloadImagesByWebWorkerCache[url];
    if (!_worker)
      _worker = new Worker(['./assets/scripts/workers/preloadImages.js?v=', APP.v].join(''));
    _preloadImagesByWebWorkerCache[url] = false;
    _worker.postMessage({
      url: url,
    });
    _worker.addEventListener('message', function (event) {
      _preloadImagesByWebWorkerCache[event.data.url] = URL.createObjectURL(event.data.blob);
    });
    return false;
  };
  var preloaders = [],
    createPreloader = function (page) {
      //APP.log('createPreloader', page);
      if (preloaders.indexOf(page) != -1) return;

      preloaders.push(page);
      self.getPage(page, function (PageObj) {
        var _model = PageObj.getModel(),
          el;
        if (!settings.PreloadImages || !window.Worker) {
          if (
            elements.preloaders.querySelector(
              '[src="',
              _model.raster[Object.keys(_model.raster)[0]],
              '"]',
            )
          )
            return;
          for (var i in _model.raster) {
            el = document.createElement('img');
            el.setAttribute('src', _model.raster[i]);
            elements.preloaders.appendChild(el);
            break;
          }
        }
      });
    },
    preloadCover = function (url) {
      try {
        if (elements.preloaders.querySelector('[src="', url, '"]')) return;
      } catch (error) {
        console_log('safari old version error');
      }
      var el = document.createElement('img');
      el.setAttribute('src', url);
      elements.preloaders.appendChild(el);
    };
  this.preloadThumbnails = function () {
    let x = 1;
    while (x <= bookModel.pages) {
      createPreloader(x);
      x++;
    }
  };
  this.useVector = function () {
    return settings.useVector;
  };
  this.useWebP = function () {
    return settings.webpSupport;
  };
  this.getPagesToLoad = function () {
    return pagesToLoad;
  };
  this.getOrientation = function () {
    return pageFlip.getOrientation();
  };

  this.getLoaderStatus = function () {
    return bookLoader != null && bookLoader.classList && bookLoader.classList.contains('roothidden');
  }

  this.preload250Blured = () => {
    const PRELOAD_COUNT = 2;
    const pagesIndexesToPreload = [Math.max(0, currentPage - PRELOAD_COUNT), Math.min(currentPage + PRELOAD_COUNT, Number(bookModel.pages))];

    Object.values(Pages).filter((page, index) => index >= pagesIndexesToPreload[0] && index < pagesIndexesToPreload[1]).forEach(page => preloadCover(page.getBlured()));
  }

  const turnOnCornerAnimation = () => {
    if (is_mobile()) return;
    if (!pageFlip) return;
    if (!bookModel.settings.show_corner_fold) return;
    if (bookModel.settings.hard_cover_enabled) return;

    if (scale > 1) return;
    if (APP.Note.getState() !== 'waiting') return;
    if ((currentPage !== 1 && currentPage !== +startPage)) return;
    if (document.getElementsByName('frame1').length > 0 && document.querySelectorAll('.embedded-player').length > 0) return;

    pageFlip.showCorner();
  };

  this.bookLoaded = function () {
    if (!isBookLoaded) {
      isBookLoaded = true;

      APP.showCookies();
      APP.hideHeaderDependsOnVisibleElements();
      APP.hideFooterDependsOnVisibleElements();
      APP.handleResponeToParent({
        'v_book_loaded': true
      });

      if (!window.disablelogs) {
        oLogs.setLogViewerStart(this, this.getCurrentPageForLogs());
      }

      if (self.getModel().settings.hasOwnProperty('auto_flip_enabled') && self.getModel().settings.auto_flip_enabled) {
        self.initializeBookAutoFlip({
          state: self.getModel().settings.auto_flip_enabled || false,
          duration: self.getModel().settings.auto_flip_duration || 0,
          loop: self.getModel().settings.auto_flip_loop_continously || false,
          times: self.getModel().settings.auto_flip_loop_times || 0
        });
      }

      if (is_mobile() && self.getModel().hasOwnProperty('is250BluredExists') && self.getModel().is250BluredExists) {
        this.preload250Blured();
      }
    }
    
    //loader min 2sek
    //todo
    const now = performance.now();
    const timeLeft = transparent ? 0 : (settings.minLoaderTime - (now - this.timerStart));
      console.log(startPage, currentPage);
    setTimeout(() => {
      HELPER.testLeadToShow(bookModel, currentPage);

      bookLoader.classList.add('roothidden');

      setTimeout(() => {
        root.hidden = false;
        root.classList.remove('roothidden');

        fullscreenAnimationFirstTime(['init'], true);
        openProductPopup();
        
        if (parseInt(bookModel.pages) > 2) {
          const firstCornerAnimationTimeout = setTimeout(() => {
            turnOnCornerAnimation();
            
            this.cornerTimeout = setInterval(() => {
              turnOnCornerAnimation();
              clearInterval(this.cornerTimeout);
            }, 5000);

            clearTimeout(firstCornerAnimationTimeout);
          }, 200);
        }

        setTimeout(() => {
          const gallerySliderIcons = document.querySelectorAll('.hotspot-gallery__slider-icon');

          if (gallerySliderIcons.length === 0) return;

          Array.from(gallerySliderIcons).forEach(item => {
            if (item && item.classList) {
              item.classList.add('hotspot-gallery__slider-animation');
            }
          });
        }, 1000);
      }, 100);
    }, timeLeft);
  };

  this.getInteractivities = function (pageNumber) {
    let interactivities = bookModel.interactivites || [];
    //binary search
    let firstIndex = 0;
    let lastIndex = interactivities.length - 1;

    while (firstIndex <= lastIndex) {
      let middleIndex = Math.round((firstIndex + lastIndex) / 2);

      if (isNaN(parseInt(interactivities[middleIndex].page))) {
        return null;
      }

      if (interactivities[middleIndex].page === pageNumber - 1) {
        return interactivities[middleIndex];
      } else if (interactivities[middleIndex].page < pageNumber - 1) {
        firstIndex = middleIndex + 1;
      } else if (interactivities[middleIndex].page > pageNumber - 1) {
        lastIndex = middleIndex - 1;
      }
    }
    return null;
  };

  this.setPageImageByBookSize = function (bookHeight, isZooming = false) {
    if (pagesToLoad && pagesToLoad.length) {
      const nodeSelector = this.showPageLikeIMG() ? 'img' : 'div';

      const pagesToNewLoad = pagesToLoad.filter(page => pages[page] && (!pages[page].getPage().querySelector(`${nodeSelector}.back`).getAttribute('size') || pages[page].getPage().querySelector(`${nodeSelector}.back`).getAttribute('size') != HELPER.getSizeOfTxtImg2()));

      if (prevBookSize && (isZooming || Math.abs(bookHeight - prevBookSize.bookHeight) >= 200) && Boolean(pagesToNewLoad.length)) {
        for (var i of pagesToNewLoad) {
          if (pages[i]) {
            pages[i].setPageImages();
          }
        }
        prevBookSize = self.getBookSize();
      }
    }

    if (!prevBookSize) {
      prevBookSize = self.getBookSize();
    }
  }

  // const setProportionalSizesToMediaViewer = async (image, video = null) => {
  //   let widthBef = 0; 
  //   let heightBef = 0;
    
  //   if (image) {
  //     let imageSizes = getImageSizesFromUrl(image);

  //     if (!imageSizes) {
  //       imageSizes = await addImageProcess(image);
  //     }

  //     widthBef = imageSizes.width;
  //     heightBef = imageSizes.height;
  //   } else if (video) {
  //     widthBef = video.thumbnail_w || 1280;
  //     heightBef = video.thumbnail_h || 720;
  //   } else {
  //     widthBef = elements.innerChild.style.width;
  //     heightBef = elements.innerChild.style.height;
  //   }
    
  //   widthBef = parseInt(widthBef);
  //   heightBef = parseInt(heightBef);
  //   const mediaViewer = document.querySelector('.media-viewer__content');
  //   let percentage = Math.max(window.innerWidth, window.innerHeight) > 768 ? 0.7 : 0.9;
  //   const sizes = HELPER.calculateAspectRatioFit(widthBef, heightBef, window.innerWidth * percentage, window.innerHeight * percentage);
  //   mediaViewer.style.width = `${sizes.width - 1}px`;
  //   mediaViewer.style.height = `${sizes.height - 1}px`;
  // }

  //Media viewer functions
  this.pauseAllEmbededVideos = function () {
    const iframes = document.getElementsByName('frame1');

    if (iframes && iframes.length) {
      iframes.forEach((iframe) => iframe.remove());
    }
  };

  this.setDataToMediaViewer = function (data, type, model, area = null) {
    let mediaContainer = '';

    switch (type) {
      case 'video':
        if (data.includes('https://player.vimeo.com/video') || data.includes('youtube.com/')) {
          if (data.startsWith('https://youtube.com')) {
            data = data.replace('https://', 'https://www.');
          }

          mediaContainer = document.createElement('iframe');
          mediaContainer.name = 'frame1';
          mediaContainer.frameborder = '0';
          mediaContainer.allow = 'autoplay; encrypted-media';
          mediaContainer.setAttribute('allowfullscreen', '');
        } else {
          mediaContainer = document.createElement('video');
          mediaContainer.setAttribute('autoplay', '');
          mediaContainer.setAttribute('disablePictureInPicture', '');
          mediaContainer.controlsList = 'nodownload noremoteplayback noplaybackrate';

          if (model.data.controls) mediaContainer.setAttribute('controls', '');
          if (model.data.muted) mediaContainer.setAttribute('muted', '');

          if (Math.max(0, +model.data.start) > 0) mediaContainer.currentTime = +model.data.start;
        }

        data = data.replace('/shorts/', '/embed/');
        
        const videoUrl = new URL(data);
        videoUrl.searchParams.set('playsinline', 1);
        videoUrl.searchParams.set('autoplay', 1);
        videoUrl.searchParams.set('mute', 0);

        mediaContainer.setAttribute('src', videoUrl.toString());

        this.pauseAllEmbededVideos();
        this.handleAudioPause();
        APP.EMBED.pauseAllPlayersOnPage(model.page);

        videoModalHandler(mediaContainer, model);

        break;

      case 'audio':
        audioModalHandler(model, area);
        break;
      
      case 'gallery':
        if (data.length) {
          APP.EMBED.pauseAllPlayersOnPage(model.page);
          gallerySliderHandler(model);
        }
        
        break;

      default:
        break;
    }
  };

  const defineAudioModal = () => {
    if (elements.audioModalElements) {
      return elements.audioModalElements;
    }

    elements.audioModalElements = {};

    const closeAudioModal = () => {
      const callback = audioModal.handleClose && typeof(audioModal.handleClose) === 'function' ? audioModal.handleClose : null;

      HELPER.fadeOutEffect(audioModal, null, 100, () => {
        HELPER.sendPostMessageToParent('publuu.viewer.popup.audio.close');

        if (callback) {
          callback();
        }
      });
    };

    const audioModal = document.createElement('div');
    const audioModalContainer = document.createElement('div');
    const audioModalCloseBtn = document.createElement('button');
    const audioModalBody = document.createElement('div');
    const audioModalBodyAudioWrapper = document.createElement('div');

    audioModal.className = 'modal modal--audio';
    audioModalContainer.className = 'modal__container';
    audioModalCloseBtn.className = 'modal__close';
    audioModalBody.className = 'modal__body';
    audioModalBodyAudioWrapper.className = 'modal__body__audio__wrapper';

    audioModal.hidden = true;
    audioModal.hide = closeAudioModal;

    audioModalCloseBtn.innerHTML = HELPER.buttonsIcons.popupCloseIcon;
    
    audioModalCloseBtn.addEventListener('click', closeAudioModal);

    window.addEventListener('keydown', (event) => {
      if (!audioModal || audioModal.hasAttribute('hidden')) return;
      if (event.code !== 'Escape') return;

      closeAudioModal();
    });
  
    audioModalBody.appendChild(audioModalBodyAudioWrapper);
    audioModalBody.appendChild(audioModalCloseBtn);

    audioModalContainer.appendChild(audioModalBody);
    audioModal.appendChild(audioModalContainer);

    document.body.appendChild(audioModal);

    elements.audioModalElements = {
      modal: audioModal,
      audioWrapper: audioModalBodyAudioWrapper,
      closeBtn: audioModalCloseBtn,
      container: audioModalContainer
    };
  };

  this.handleAudioPlay = () => {
    if (!elements.audioModalElements || !elements.audioModalElements.audio) return;

    elements.audioModalElements.audio.play();
  };

  this.handleAudioPause = () => {
    if (!elements.audioModalElements || !elements.audioModalElements.audio) return;

    elements.audioModalElements.audio.pause();
  };

  const audioModalHandler = async (hotspotModel, hotspotArea) => {
    if (elements.audioModalElements && elements.audioModalElements.audio) {
      const hotspotId = hotspotModel.uid || hotspotModel.area_number;

      if (elements.audioModalElements.audio.id != hotspotId) {
        elements.audioModalElements.audio.instance.destroy();
        delete elements.audioModalElements.audio;
      } else {
        if (hotspotModel._action === 'RENDER') {
          elements.audioModalElements.audio.instance.updateOnRender(hotspotArea);
          return;
        }

        const actualState = elements.audioModalElements.audio.instance.getState();

        if (actualState === 'PLAYING') {
          elements.audioModalElements.audio.instance.pause();
        } else {
          elements.audioModalElements.audio.instance.play();
        }

        return;
      }
    }
    
    defineAudioModal();

    const audioModal = elements.audioModalElements.modal;

    const hotspotOptions = hotspotModel.data;
    const mediaData = {
      url: hotspotOptions.value
    };

    const options = {
      id: hotspotModel.uid || hotspotModel.area_number,
      page: hotspotModel.page,
      start: hotspotOptions.start || 0,
      loop: !!hotspotOptions.loop,
      muted: hotspotOptions.hasOwnProperty('muted') ? !!hotspotOptions.muted : false,
      controls: hotspotOptions.hasOwnProperty('controls') ? !!hotspotOptions.controls : true,
      autoplay: true,
      showPlayer: hotspotOptions.hasOwnProperty('show_player') ? !!hotspotOptions.show_player : true,
    };

    await HELPER.loadScript(`${APP.ROOT_PATH}assets/scripts/hotspots/Audio.js?v=${window._config.VERSION}`, {
      id: 'Audio',
      type: "text/javascript",
      crossorigin: "anonymous",
    });

    elements.audioModalElements.hotspot = hotspotArea;

    const audio = new APP.Audio(elements.audioModalElements, mediaData, options);

    HELPER.fadeInEffect(audioModal, null, 100, () => HELPER.sendPostMessageToParent('publuu.viewer.popup.audio.open'));

    elements.audioModalElements.audio = audio.getElement();
    elements.audioModalElements.audio.instance = audio;

    audioModal.handleClose = () => {
      audio.destroy();
      delete elements.audioModalElements.audio;
    };

    audioModal.errorCallbacks = {
      NotAllowedError: () => setTimeout(alertModalHandler, 500)
    };
  };

  const defineVideoModal = () => {
    if (elements.videoModalElements) {
      return elements.videoModalElements;
    }

    elements.videoModalElements = {};

    const initializeVideoHandlers = (videoElement) => {
      videoElement.addEventListener('webkitendfullscreen', closeVideoModal);
    };

    const closeVideoModal = () => {
      const cb = videoModal.handleClose && typeof(videoModal.handleClose) === 'function' ? videoModal.handleClose : null;

      HELPER.fadeOutEffect(videoModal, videoModalContainer, MODAL_FADE_TIME, () => {
        HELPER.sendPostMessageToParent('publuu.viewer.popup.video.close');

        if (cb) {
          cb();
        }
      });
    };

    const closeVideoModalOnClickOutside = (event) => {
      const shouldCloseModal = !(event.path || event.composedPath()).find(el => el === videoModalBody || el === videoModalFooterContent);

      if (!shouldCloseModal) return;

      closeVideoModal();
    };

    const videoModal = document.createElement('div');
    const videoModalContainer = document.createElement('div');
    const videoModalCloseBtn = document.createElement('button');
    const videoModalBody = document.createElement('div');
    const videoModalBodyContent = document.createElement('div');

    videoModal.className = 'modal modal--video';
    videoModalContainer.className = 'modal__container';
    videoModalCloseBtn.className = 'modal__close';
    videoModalBody.className = 'modal__body';
    videoModalBodyContent.className = 'modal__body__content';

    videoModal.hidden = true;
    videoModal.hide = closeVideoModal;
    videoModal.initializeVideoHandlers = initializeVideoHandlers;

    videoModalCloseBtn.innerHTML = HELPER.buttonsIcons.popupCloseIcon;
    
    videoModalCloseBtn.addEventListener('click', closeVideoModal);
    // videoModal.addEventListener('click', closeVideoModalOnClickOutside); 

    window.addEventListener('keydown', (event) => {
      if (!videoModal || videoModal.hasAttribute('hidden')) return;
      if (event.code !== 'Escape') return;

      closeVideoModal();
    });
  
    videoModalBody.appendChild(videoModalCloseBtn);
    videoModalBody.appendChild(videoModalBodyContent);

    videoModalContainer.appendChild(videoModalBody);
    videoModal.appendChild(videoModalContainer);

    document.body.appendChild(videoModal);

    elements.videoModalElements = {
      modal: videoModal,
      container: videoModalContainer,
      bodyContent: videoModalBodyContent,
      closeBtn: videoModalCloseBtn
    };
  };

  const defineGalleryModal = () => {
    if (elements.galleryModalElements) {
      return elements.galleryModalElements;
    }

    elements.galleryModalElements = {};

    const closeGalleryModal = () => {
      const cb = galleryModal.handleClose && typeof(galleryModal.handleClose) === 'function' ? galleryModal.handleClose : null;

      HELPER.fadeOutEffect(galleryModal, galleryModalContainer, MODAL_FADE_TIME, () => {
        HELPER.sendPostMessageToParent('publuu.viewer.popup.gallery.close');

        if (cb) {
          cb();
        }
      });
    };

    const closeGalleryModalOnClickOutside = (event) => {
      const shouldCloseModal = !(event.path || event.composedPath()).find(el => el === galleryModalBody || el === galleryModalFooterContent);

      if (!shouldCloseModal) return;

      closeGalleryModal();
    };

    const galleryModal = document.createElement('div');
    const galleryModalContainer = document.createElement('div');
    const galleryModalControlsLeft = document.createElement('div');
    const galleryModalControlsRight = document.createElement('div');
    const galleryModalArrowLeft = document.createElement('button');
    const galleryModalArrowRight = document.createElement('button');
    const galleryModalCloseBtn = document.createElement('button');
    const galleryModalBody = document.createElement('div');
    const galleryModalBodyContent = document.createElement('div');
    const galleryModalFooter = document.createElement('div');
    const galleryModalFooterContent = document.createElement('div');

    galleryModal.className = 'modal modal--gallery';
    galleryModalContainer.className = 'modal__container';
    galleryModalControlsLeft.className = 'modal__controls modal__controls--left slider__btn--prev';
    galleryModalControlsRight.className = 'modal__controls modal__controls--right slider__btn--next';
    galleryModalArrowLeft.className = 'modal__controls__arrow';
    galleryModalArrowRight.className = 'modal__controls__arrow';
    galleryModalCloseBtn.className = 'modal__close';
    galleryModalBody.className = 'modal__body';
    galleryModalBodyContent.className = 'modal__body__content';
    galleryModalFooter.className = 'modal__footer';
    galleryModalFooterContent.className = 'modal__footer__content';

    galleryModal.hidden = true;
    galleryModal.hide = closeGalleryModal;

    galleryModalArrowLeft.innerHTML = HELPER.buttonsIcons.sliderControlArrowLeft;
    galleryModalArrowRight.innerHTML = HELPER.buttonsIcons.sliderControlArrowRight;
    galleryModalCloseBtn.innerHTML = HELPER.buttonsIcons.popupCloseIcon;
    
    galleryModalCloseBtn.addEventListener('click', closeGalleryModal);
    // galleryModal.addEventListener('click', closeGalleryModalOnClickOutside); 

    window.addEventListener('keydown', (event) => {
      if (!galleryModal || galleryModal.hasAttribute('hidden')) return;
      if (event.code !== 'Escape') return;

      closeGalleryModal();
    });
  
    galleryModalControlsLeft.appendChild(galleryModalArrowLeft);
    galleryModalControlsRight.appendChild(galleryModalArrowRight);
    galleryModalBody.appendChild(galleryModalCloseBtn);
    galleryModalBody.appendChild(galleryModalControlsLeft);
    galleryModalBody.appendChild(galleryModalBodyContent);
    galleryModalBody.appendChild(galleryModalControlsRight);

    galleryModalFooter.appendChild(galleryModalFooterContent);
    galleryModalContainer.appendChild(galleryModalBody);
    galleryModalContainer.appendChild(galleryModalFooter);
    galleryModal.appendChild(galleryModalContainer);

    document.body.appendChild(galleryModal);

    elements.galleryModalElements = {
      modal: galleryModal,
      container: galleryModalContainer,
      bodyContent: galleryModalBodyContent,
      footerContent: galleryModalFooterContent,
      footer: galleryModalFooter,
      controls: {
        arrowLeft: galleryModalArrowLeft,
        arrowRight: galleryModalArrowRight,
      },
      closeBtn: galleryModalCloseBtn
    };
  };

  const videoModalHandler = (mediaContainer, hotspotModel) => {
    defineVideoModal();

    const videoModal = elements.videoModalElements.modal;
    const videoModalContainer = elements.videoModalElements.container;
    const videoModalBodyContent = elements.videoModalElements.bodyContent;

    videoModalBodyContent.appendChild(mediaContainer);
    videoModal.initializeVideoHandlers(mediaContainer);

    videoModal.handleClose = () => videoModalBodyContent.innerHTML = '';

    HELPER.fadeInEffect(videoModal, videoModalContainer, MODAL_FADE_TIME, () => HELPER.sendPostMessageToParent('publuu.viewer.popup.video.open'));
  };

  const prepareHotspotData = (hotspot) => {
    const hotspotData = [];

    hotspot.data.forEach((item, index) => {
      if (item && item.length > 0) {
        hotspotData.push({
          src: item,
          thumb: hotspot.data_thumbs && hotspot.data_thumbs.length > index ? hotspot.data_thumbs[index] : null,
          sizes: hotspot.data_sizes && hotspot.data_sizes.length > index ? hotspot.data_sizes[index] : null,
          type: 'IMAGE',
        });
      }
    });

    return hotspotData;
  };

  const gallerySliderHandler = async (hotspotModel) => {
    defineGalleryModal();

    const galleryModal = elements.galleryModalElements.modal;
    const galleryModalContainer = elements.galleryModalElements.container;

    const galleryData = prepareHotspotData(hotspotModel);
    const galleryElements = {
      mainSliderWrapper: elements.galleryModalElements.bodyContent,
      thumbsSliderWrapper: elements.galleryModalElements.footerContent,
      controls: elements.galleryModalElements.controls
    };

    HELPER.fadeInEffect(galleryModal, galleryModalContainer, MODAL_FADE_TIME, () => HELPER.sendPostMessageToParent('publuu.viewer.popup.gallery.open'));

    await HELPER.loadScript(`${APP.ROOT_PATH}assets/scripts/hotspots/Gallery.js?v=${window._config.VERSION}`, {
      id: 'Gallery',
      type: "text/javascript",
      crossorigin: "anonymous",
    });

    const gallery = new APP.Gallery(galleryElements, galleryData, {});

    galleryModal.handleClose = () => gallery.destroy();
  };

  const defineAlertModal = () => {
    if (elements.alertModalElements) {
      return elements.alertModalElements;
    }

    elements.alertModalElements = {};

    const closeAlertModal = () => {
      const callbackFn = alertModal.handleClose && typeof(alertModal.handleClose) === 'function' ? alertModal.handleClose : null;

      HELPER.fadeOutEffect(alertModal, alertModalContainer, MODAL_FADE_TIME, () => {
        HELPER.sendPostMessageToParent('publuu.viewer.popup.alert.close');

        if (callbackFn) {
          callbackFn();
        }
      });
    };

    const closeAlertModalOnClickOutside = (event) => {
      const shouldCloseModal = !(event.path || event.composedPath()).find(el => el === alertModalContainer);

      if (!shouldCloseModal) return;

      closeAlertModal();
    };

    const handleAlertModalAccept = () => {
      this.handleAudioPlay();
      closeAlertModal();
    };

    const alertModal = document.createElement('div');
    const alertModalContainer = document.createElement('div');
    const alertModalCloseBtn = document.createElement('button');

    const alertModalHeader = document.createElement('div');
    const alertModalHeaderIconWrapper = document.createElement('div');
    const alertModalHeaderTitle = document.createElement('h2');

    const alertModalBody = document.createElement('div');
    const alertModalBodyContent = document.createElement('div');
    const alertModalBodyContentText = document.createElement('p');
    
    const alertModalFooter = document.createElement('div');
    const alertModalFooterBtn = document.createElement('button');

    alertModal.className = 'modal modal--alert';
    alertModalContainer.className = 'modal__container';
    alertModalCloseBtn.className = 'modal__close';

    alertModalHeaderIconWrapper.className = 'modal__header__icon';
    alertModalHeader.className = 'modal__header';
    alertModalHeaderTitle.className = 'modal__title';
    
    alertModalBody.className = 'modal__body';
    alertModalBodyContent.className = 'modal__body__content';
    alertModalBodyContentText.className = 'modal__body__text';

    alertModalFooter.className = 'modal__footer';
    alertModalFooterBtn.className = 'modal__footer__button';

    alertModalFooterBtn.dataset.type ='accept';
    alertModalFooterBtn.textContent = APP._t('Alert Form Submit');
    alertModalHeaderTitle.textContent = APP._t('Alert Form Title');
    alertModalBodyContentText.textContent = APP._t('Alert Form Text');
    alertModalHeaderIconWrapper.innerHTML = HELPER.buttonsIcons.alertIcon;
    
    alertModal.hidden = true;
    alertModal.hide = closeAlertModal;

    alertModalCloseBtn.innerHTML = HELPER.buttonsIcons.popupCloseIcon;
    
    alertModalFooterBtn.addEventListener('click', handleAlertModalAccept);
    alertModalCloseBtn.addEventListener('click', closeAlertModal);
    alertModal.addEventListener('click', closeAlertModalOnClickOutside); 

    window.addEventListener('keydown', (event) => {
      if (!alertModal || alertModal.hasAttribute('hidden')) return;
      if (event.code !== 'Escape') return;

      closeAlertModal();
    });

    alertModalHeader.appendChild(alertModalHeaderIconWrapper);
    alertModalHeader.appendChild(alertModalHeaderTitle);

    alertModalFooter.appendChild(alertModalFooterBtn);
  
    alertModalBodyContent.appendChild(alertModalBodyContentText);
    alertModalBody.appendChild(alertModalCloseBtn);
    alertModalBody.appendChild(alertModalBodyContent);
    alertModalBody.appendChild(alertModalFooter);

    alertModalContainer.appendChild(alertModalHeader);
    alertModalContainer.appendChild(alertModalBody);
    alertModal.appendChild(alertModalContainer);

    document.body.appendChild(alertModal);

    elements.alertModalElements = {
      modal: alertModal,
      container: alertModalContainer,
      bodyContent: alertModalBodyContent,
      closeBtn: alertModalCloseBtn
    };
  };

  const alertModalHandler = () => {
    defineAlertModal();

    const alertModal = elements.alertModalElements.modal;
    const alertModalContainer = elements.alertModalElements.container;

    HELPER.fadeInEffect(alertModal, alertModalContainer, MODAL_FADE_TIME, () => HELPER.sendPostMessageToParent('publuu.viewer.popup.alert.open'));
  };

  const defineLeadFormModal = () => {
    if (elements.leadFormModalElements) {
      return elements.leadFormModalElements;
    }

    const leadFormErrors = [];
    const settings = bookModel.settings;
    elements.leadFormModalElements = {};

    const closeLeadFormModal = () => {
      const callbackFn = leadFormModal.handleClose && typeof(leadFormModal.handleClose) === 'function' ? leadFormModal.handleClose : null;

      HELPER.fadeOutEffect(leadFormModal, leadFormModalContainer, MODAL_FADE_TIME, () => {
        HELPER.sendPostMessageToParent('publuu.viewer.popup.leadform.close');

        if (callbackFn) {
          callbackFn();
        }
      });
    };

    const getPrivacyContent = () => {
      if (settings.lead_form_custom_policy && settings.lead_form_custom_policy_link !== '') {
        return APP._t('Lead Form Privacy With Link', [settings.lead_form_custom_policy_link, 'https://publuu.com/privacy-statement/']);
      } 
  
      return APP._t('Lead Form Privacy', ['https://publuu.com/privacy-statement/']);
    };

    const getLeadFormFields = (fields) => {
      const defaultFields = [{
        'name': 'E-mail',
        'type': 'email',
        'required': true
      }, {
        'name': 'Phone Number',
        'type': 'phone', 
        'required':false
      }];
  
      try {
        if (fields) return JSON.parse(fields);
  
        return defaultFields;
      } catch (err) {
        console.log(`Error while parsing lead form data: ${err}`);
        return defaultFields;
      }
    };

    const createFormField = (field) => {
      const formInputWrapper = document.createElement('div');
      const formInputErrorElement = document.createElement('span');
      const formInput = document.createElement('input');
      const formInputName = !!field.name ? field.name : '';
      const formInputType = !!field.type ? field.type : 'text';
      const isFieldRequired = !!field.required;
      const formInputPlaceholder = `${formInputName} ${isFieldRequired ? '*' : ''}`;

      formInputWrapper.className = 'modal__body__form__input__wrapper';
      formInputErrorElement.className = 'modal__body__form__input__error';
      
      formInput.value = '';
      formInput.type = 'text';
      formInput.name = formInputName;
      formInput.required = isFieldRequired;
      formInput.dataset.type = formInputType;
      formInput.placeholder = formInputPlaceholder;
      formInput.className = 'modal__body__form__input';
      
      formInput.addEventListener('input', function () {
        if (!validateLeadFormFields(this, formInputErrorElement)) {
          elements.leadFormModalElements.submitButton.disabled = true;
          return;
        } 

        const invalidInputs = elements.leadFormModalElements.modal.querySelectorAll('.modal__body__form__input:required:invalid');

        if (invalidInputs.length > 0) {
          elements.leadFormModalElements.submitButton.disabled = true;
        } else {
          elements.leadFormModalElements.submitButton.disabled = false;
        }
      });

      formInputWrapper.appendChild(formInput);
      formInputWrapper.appendChild(formInputErrorElement);

      return formInputWrapper;
    };

    const createValidationError = (obj, errorType) => {
      const errorIndex = leadFormErrors.findIndex(item => item.element == obj);

      obj.classList.add('invalid')
  
      if (errorIndex === -1) {
        leadFormErrors.push({ element: obj, errorType });
        return;
      }
  
      leadFormErrors[errorIndex].errorType = errorType;
    };
  
    const removeValidationError = (obj) => {
      const errorIndex = leadFormErrors.findIndex(item => item.element == obj);
  
      if (errorIndex === -1) return false;
  
      leadFormErrors.splice(errorIndex, 1);
      obj.classList.remove('invalid');
  
      return true;
    };

    const validateLeadFormFields = function (obj, errorElement) {
      const type = obj.dataset && obj.dataset.hasOwnProperty('type') ? obj.dataset.type : 'text';
      const REGEX = {
        email: /^[\w\-\.]+@([\w-]+\.)+[\w-]{2,}$/,
        phone: /^[+]{0,1}[0-9-\s()]+$/
      };
  
      if (!!obj.required && !obj.value.length) {
        createValidationError(obj, 'required');
        errorElement.textContent = APP._t('Lead Form Validation Required');
      } else if (!!obj.value.length && !obj.value.match(REGEX[type])) {
        createValidationError(obj, 'regex');
        errorElement.textContent = APP._t('Lead Form Validation Format');
      } else {
        removeValidationError(obj);
        errorElement.textContent = '';
      }
  
      return !leadFormErrors.length;
    };

    const handleSubmitLeadForm = async (skipped = false) => {
      const publisherId = bookModel.publisher_id;
      const issueId = bookModel.id;
      const dlid = window.dlid ? window.dlid.toString() : null;

      if (!skipped) {
        const leadFormFields = elements.leadFormModalElements.modal.querySelectorAll('.modal__body__form__input');
        
        if (leadFormFields.length > 0) {
          leadFormErrors.length = 0;
      
          Array.from(leadFormFields).forEach(obj => validateLeadFormFields(obj, obj.nextElementSibling));
      
          if (!!leadFormErrors.length) return;
        }
      }
      
      const fields = Array.from(elements.leadFormModalElements.modal.querySelectorAll('.modal__body__form__input')).map(formInput => ({
        name: formInput.name, 
        value: formInput.value
      }));

      const data = `{"uid":"${HELPER.uid()}", "app_type":1, "fields":${JSON.stringify(fields)}, "skipped":${skipped}, "dlid": ${dlid}}`;

      try {
        const response = await fetch(`${APP.LEADS_API}/${publisherId}/${issueId}`, {
          method: 'POST',
          body: data,
          headers: {
            'Accept': 'application/json',
          },
        });

        if (response.status !== 200 && response.status !== 201) {
          throw new Error('Invalid response');
        }

        closeLeadFormModal();
        HELPER.setLocalStorageData('leadform', ',' + issueId, 365);

        return true;
      } catch (err) {
        console.log(`Lead form error. Error: ${err}`);
        return false;
      }
    };

    const leadFormModal = document.createElement('div');
    const leadFormModalContainer = document.createElement('div');

    const leadFormModalBody = document.createElement('div');
    const leadFormModalBodyContent = document.createElement('div');
    const leadFormModalBodyImageWrapper = document.createElement('div');
    const leadFormModalBodyImage = document.createElement('img');

    const leadFormModalBodyForm = document.createElement('form');
    const leadFormModalBodyFormFields = document.createElement('div');
    const leadFormModalBodyTitle = document.createElement('h2');
    const leadFormModalBodyPrivacy = document.createElement('div');
    const leadFormModalBodyButtonsWrapper = document.createElement('div');
    const leadFormModalBodySubmitButton = document.createElement('button');
    const leadFormModalBodySkipButton = document.createElement('button');
    
    leadFormModal.className = 'modal modal--lead-form';
    leadFormModalContainer.className = 'modal__container';

    leadFormModalBody.className = 'modal__body';
    leadFormModalBodyContent.className = 'modal__body__content';
    leadFormModalBodyImageWrapper.className = 'modal__body__image__wrapper';
    leadFormModalBodyImage.className = 'modal__body__image';
    leadFormModalBodyForm.className = 'modal__body__form';
    leadFormModalBodyFormFields.className = 'modal__body__form__inputs';
    leadFormModalBodyTitle.className = 'modal__title';
    leadFormModalBodyPrivacy.className = 'modal__body__privacy';
    leadFormModalBodyButtonsWrapper.className = 'modal__body__buttons';
    leadFormModalBodySubmitButton.className = 'modal__body__button modal__body__button--submit';
    leadFormModalBodySkipButton.className = 'modal__body__button modal__body__button--skip';

    leadFormModalBodyImage.src = `${APP.PATH_CF_FLIP}assets/images/lead/lead_bg.svg`;
    leadFormModalBodyImage.alt = 'Lead form image';

    leadFormModalBodySkipButton.type = 'button';
    leadFormModalBodySubmitButton.type = 'submit';
    leadFormModalBodySubmitButton.setAttribute('disabled', '');
    leadFormModalBodySubmitButton.textContent = APP._t('Lead Form Submit');
    leadFormModalBodySkipButton.textContent = APP._t('Lead Form Skip');
    leadFormModalBodySkipButton.hidden = !settings.lead_form_allow_skip;

    leadFormModalBodyTitle.textContent = settings.lead_form_caption || APP._t('Lead Form Caption Default');
    leadFormModalBodyPrivacy.innerHTML = getPrivacyContent();

    if (!settings.lead_form_image_enabled) {
      leadFormModalContainer.classList.add('modal__container--simple');
    }

    leadFormModalBodySkipButton.addEventListener('click', async () => {
      if (customizemode) return;

      const responseState = await handleSubmitLeadForm(true);

      if (responseState) {
        closeLeadFormModal();
      }
    });
    
    leadFormModalBodySubmitButton.addEventListener('click', async (event) => {
      event.preventDefault();

      if (customizemode) return;

      await handleSubmitLeadForm();
    });

    leadFormModal.hidden = true;
    leadFormModal.hide = closeLeadFormModal;
    leadFormModal.createFormField = createFormField;

    const fields = getLeadFormFields(settings.lead_form_fields);

    if (fields.length > 0) {
      const hasRequiredField = fields.some(field => !!field.required);

      if (hasRequiredField) {
        leadFormModalBodySubmitButton.disabled = true;
      }

      fields.forEach(field => {
        const formField = createFormField(field);

        leadFormModalBodyFormFields.appendChild(formField);
      });
    }
  
    leadFormModalBodyImageWrapper.appendChild(leadFormModalBodyImage);
    leadFormModalBody.appendChild(leadFormModalBodyImageWrapper);
    leadFormModalBodyButtonsWrapper.appendChild(leadFormModalBodySubmitButton);
    leadFormModalBodyButtonsWrapper.appendChild(leadFormModalBodySkipButton);
    leadFormModalBodyForm.appendChild(leadFormModalBodyFormFields);
    leadFormModalBodyForm.appendChild(leadFormModalBodyPrivacy);
    leadFormModalBodyForm.appendChild(leadFormModalBodyButtonsWrapper);
    leadFormModalBodyContent.appendChild(leadFormModalBodyTitle);
    leadFormModalBodyContent.appendChild(leadFormModalBodyForm);
    leadFormModalBody.appendChild(leadFormModalBodyContent);

    leadFormModalContainer.appendChild(leadFormModalBody);
    leadFormModal.appendChild(leadFormModalContainer);

    if (customizemode) {
      const leadFormModalTooltip = document.createElement('div');
      
      leadFormModalTooltip.className = 'modal__body__tooltip';
      leadFormModalTooltip.innerHTML = `<span>${APP._t('Lead Form Tooltip')}</span>`;
      
      leadFormModalBodyButtonsWrapper.appendChild(leadFormModalTooltip);
    }

    document.body.appendChild(leadFormModal);

    elements.leadFormModalElements = {
      modal: leadFormModal,
      container: leadFormModalContainer,
      title: leadFormModalBodyTitle,
      bodyContent: leadFormModalBodyContent,
      form: leadFormModalBodyForm,
      formFieldsWrapper: leadFormModalBodyFormFields,
      skipButton: leadFormModalBodySkipButton,
      submitButton: leadFormModalBodySubmitButton,
      imageWrapper: leadFormModalBodyImageWrapper,
      privacy: leadFormModalBodyPrivacy
    };
  };

  const leadFormHandler = () => {
    defineLeadFormModal();

    const leadFormModal = elements.leadFormModalElements.modal;
    const leadFormModalContainer = elements.leadFormModalElements.container;

    HELPER.fadeInEffect(leadFormModal, leadFormModalContainer, MODAL_FADE_TIME, () => HELPER.sendPostMessageToParent('publuu.viewer.popup.leadform.open'));
  };

  this.showLeadFormModal = () => {
    leadFormHandler();
  };

  this.hideLeadFormModal = () => {
    if (!elements.leadFormModalElements) return;

    elements.leadFormModalElements.modal.hide();
  };

  this.customizeLeadFormModal = (eventData) => {
    defineLeadFormModal();

    const settings = bookModel.settings;

    if (eventData.hasOwnProperty('lead_form_image_enabled')) {
      elements.leadFormModalElements.container.classList.toggle('modal__container--simple', !eventData.lead_form_image_enabled);
    }

    if (eventData.hasOwnProperty('lead_form_allow_skip')) {
      if (!!eventData.lead_form_allow_skip) {
        elements.leadFormModalElements.skipButton.removeAttribute('hidden');
      } else {
        elements.leadFormModalElements.skipButton.setAttribute('hidden', '');
      }
    }

    if (eventData.hasOwnProperty('lead_form_caption')) {
      elements.leadFormModalElements.title.textContent = eventData.lead_form_caption;
    }
    
    if (eventData.hasOwnProperty('lead_form_custom_policy') || eventData.hasOwnProperty('lead_form_custom_policy_link')) {
      let newPrivacy = '';
      
      if (!!settings.lead_form_custom_policy && settings.lead_form_custom_policy_link !== '') {
        newPrivacy = APP._t('Lead Form Privacy With Link', [settings.lead_form_custom_policy_link, 'https://publuu.com/privacy-statement/']);
      } else {
        newPrivacy = APP._t('Lead Form Privacy', ['https://publuu.com/privacy-statement/']);
      }

      elements.leadFormModalElements.privacy.innerHTML = newPrivacy;
    }

    if (eventData.hasOwnProperty('lead_form_fields')) {
      const fields = eventData.lead_form_fields;

      if (!fields) return;

      const hasRequiredField = fields.some(field => !!field.required);

      if (hasRequiredField) {
        elements.leadFormModalElements.submitButton.setAttribute('disabled', '');
      } else {
        elements.leadFormModalElements.submitButton.removeAttribute('disabled');
      }

      const existingsFormFields = elements.leadFormModalElements.formFieldsWrapper.querySelectorAll('.modal__body__form__input__wrapper');
      
      Array.from(existingsFormFields).forEach(field => field.remove());

      fields.forEach(field => {
        const formField = elements.leadFormModalElements.modal.createFormField(field);

        elements.leadFormModalElements.formFieldsWrapper.appendChild(formField);
      });
    }
  };

  const definePrintModal = () => {
    if (elements.printModalElements) {
      return elements.printModalElements;
    }

    const bookId = bookModel.ref_id || bookModel.id;
    const FROM = 1;
    const TO = +bookModel.pages;
    const PATH_FROM = HELPER.getPageImageWithTxtUrl(FROM, 400).replaceAll(bookModel.id, bookId);
    const PATH_TO = HELPER.getPageImageWithTxtUrl(TO, 400).replaceAll(bookModel.id, bookId);

    elements.printModalElements = {};

    const closePrintModal = () => {
      const callbackFn = printModal.handleClose && typeof(printModal.handleClose) === 'function' ? printModal.handleClose : null;

      HELPER.fadeOutEffect(printModal, printModalContainer, MODAL_FADE_TIME, () => {
        HELPER.sendPostMessageToParent('publuu.viewer.popup.print.close');

        if (callbackFn) {
          callbackFn();
        }
      });
    };

    const closePrintModalOnClickOutside = (event) => {
      const shouldCloseModal = !(event.path || event.composedPath()).find(el => el === printModalContainer);

      if (!shouldCloseModal) return;

      closePrintModal();
    };

    const setPreview = (FROM, TO) => {
      elements.printModalElements.pageStart.textContent = FROM;
      elements.printModalElements.pageEnd.textContent = TO;

      const bookId = bookModel.ref_id || bookModel.id;
      const PATH_FROM = HELPER.getPageImageWithTxtUrl(FROM, 400).replaceAll(bookModel.id, bookId);
      const PATH_TO = HELPER.getPageImageWithTxtUrl(TO, 400).replaceAll(bookModel.id, bookId);

      elements.printModalElements.coverFrom.src = PATH_FROM;
      elements.printModalElements.coverTo.src = PATH_TO;

      elements.printModalElements.previewFrom.classList.remove('current-full');

      if (FROM === TO) {
        elements.printModalElements.previewTo.setAttribute('hidden', '');

        const { width, height } = Pages[self.getCurrentPage()].getModel().model;

        if (width >= height) {
          elements.printModalElements.previewFrom.classList.add('current-full');
        } 
      } else {
        elements.printModalElements.previewTo.removeAttribute('hidden');
      }
    };

    const handleChangePrintOption = function() {
      const optionValue = +this.value;

      let FROM = 1;
      let TO = +bookModel.pages;

      elements.printModalElements.rangeFrom.classList.remove('active');
      elements.printModalElements.rangeTo.classList.remove('active');

      switch (optionValue) {
        case 1:
          FROM = 1;
          TO = +bookModel.pages;

          break;
        
        case 2:
          FROM = currentPage > 1 ? currentPage : 1;
          TO = currentPage === 1 ? 1 : forceOnePage ? currentPage : currentPage + 1;

          break;
        
        case 3:
          FROM = +elements.printModalElements.rangeFrom.value;
          TO = +elements.printModalElements.rangeTo.value;

          if (FROM > TO) {
            FROM = TO;

            elements.printModalElements.rangeFrom.value = FROM;
          }

          elements.printModalElements.rangeFrom.classList.add('active');
          elements.printModalElements.rangeTo.classList.add('active');
          break;

        default:
          break;
      }

     setPreview(FROM, TO);
    };

    const handleKeyUpPrintRange = (event) => {
      if (event.target.value === '') {
        event.preventDefault();
        event.stopPropagation();
        return;
      }
    };

    const handleChangePrintRange = () => {
      let shouldUpdatePreview = false;

      const rangeFromValue = +elements.printModalElements.rangeFrom.value;
      const rangeToValue = +elements.printModalElements.rangeTo.value;

      if (isNaN(rangeFromValue)) {
        elements.printModalElements.rangeFrom.value = 1;
        shouldUpdatePreview = true;
      }

      if (isNaN(rangeToValue)) {
        elements.printModalElements.rangeTo.value = bookModel.pages;
        shouldUpdatePreview = true;
      }

      if (rangeFromValue < 1) {
        elements.printModalElements.rangeFrom.value = 1;
        shouldUpdatePreview = true;
      }

      if (rangeToValue > +bookModel.pages) {
        elements.printModalElements.rangeTo.value = bookModel.pages;
        shouldUpdatePreview = true;
      }

      if (rangeFromValue > rangeToValue) {
        elements.printModalElements.rangeFrom.value = elements.printModalElements.rangeTo.value;
        shouldUpdatePreview = true;
      }

      if (rangeToValue < rangeToValue) {
        elements.printModalElements.rangeTo.value = elements.printModalElements.rangeFrom.value;
        shouldUpdatePreview = true;
      }

      if (shouldUpdatePreview) {
        const FROM = +elements.printModalElements.rangeFrom.value;
        const TO = +elements.printModalElements.rangeTo.value;
        
        setPreview(FROM, TO);
      }
    };

    const handleInputPrintRange = (event) => {
      const target = event.target;
      let rangeValue = +target.value;

      if (isNaN(rangeValue)) return;

      if (rangeValue < 1) rangeValue = 1;
      if (rangeValue > +bookModel.pages) rangeValue = +bookModel.pages;
      if (target.value != rangeValue) target.value = rangeValue;

      const FROM = elements.printModalElements.rangeFrom.value;
      const TO = elements.printModalElements.rangeTo.value;

      if (isNaN(FROM) || isNaN(TO)) return;

      setPreview(FROM, TO);
    };

    const handlePrint = async () => {
      const printBtn = elements.printModalElements.printBtn;

      elements.printModalElements.errorElement.setAttribute('hidden', '');

      const bookId = bookModel.ref_id || bookModel.id;
      const publisherId = bookModel.publisher_id;
      const version = +bookModel.version;

      const PDF_URL = new URL(`${APP.VIEWER_API}/${publisherId}/${bookId}/pdf`);

      let FROM = 1;
      let TO = +bookModel.pages;

      if (elements.printModalElements.options.all.checked) {
        FROM = 1;
        TO = +bookModel.pages;
      } else if (elements.printModalElements.options.current.checked) {
        if (
          pageFlip.getOrientation() === 'landscape' &&
          currentPage !== 1 &&
          currentPage !== bookModel.pages
        ) {
          FROM = +currentPage;
          TO = +currentPage + 1;
        } else {
          FROM = +currentPage;
          TO = +currentPage;
        }
      } else if (elements.printModalElements.options.range.checked) {
        FROM = +elements.printModalElements.rangeFrom.value || 1;
        TO = +elements.printModalElements.rangeTo.value || +bookModel.pages;
      }

      PDF_URL.searchParams.set('pageFrom', FROM);
      PDF_URL.searchParams.set('pageTo', TO);

      if (bookModel.settings.is_password_protected) {
        PDF_URL.searchParams.set('password', self.password);
      }

      if (version > 0) {
        PDF_URL.searchParams.set('version', version);
      }

      if (customizemode === 1) {
        PDF_URL.searchParams.set('cm', 1);
      }
    
      printBtn.classList.add('loading');

      try {
        const response = await fetch(PDF_URL.toString(), { method: 'GET' });
        const data = await response.json();

        if (!data.url) {
          throw 'No data url provided';
        }

        printJS({ 
          printable: data.url, 
          type: 'pdf', 
          showModal: true,
          modalMessage: APP._t('Print Form Modal Message'), 
          onLoadingEnd: () => {
            printBtn.classList.remove('loading');
            elements.printModalElements.closeBtn.click();
          }
        });
      } catch (err) {
        console.log(`Error while fetching pdf: ${err}`);
      } finally {
        printBtn.classList.remove('loading');
      }
    };

    const handleFocusPrintRange = () => {
      if (!elements.printModalElements.options.range.checked) {
        elements.printModalElements.options.range.click();
      }
    };

    const printModal = document.createElement('div');
    const printModalContainer = document.createElement('div');
    const printModalCloseBtn = document.createElement('button');

    const printModalHeader = document.createElement('div');
    const printModalHeaderIconWrapper = document.createElement('div');
    const printModalHeaderTitle = document.createElement('h2');

    const printModalBody = document.createElement('div');
    const printModalBodyContent = document.createElement('div');

    const printModalBodyContentMain = document.createElement('div');
    const printModalBodyContentMainContent = document.createElement('div');
    const printModalBodyContentMainPrintAll = document.createElement('div');
    const printModalBodyContentMainPrintAllInput = document.createElement('input');
    const printModalBodyContentMainPrintAllLabel = document.createElement('label');

    const printModalBodyContentMainPrintCurrent = document.createElement('div');
    const printModalBodyContentMainPrintCurrentInput = document.createElement('input');
    const printModalBodyContentMainPrintCurrentLabel = document.createElement('label');

    const printModalBodyContentMainPrintRange = document.createElement('div');
    const printModalBodyContentMainPrintRangeInput = document.createElement('input');
    const printModalBodyContentMainPrintRangeLabel = document.createElement('label');
    const printModalBodyContentMainPrintRangeLabelSpan = document.createElement('span');
    const printModalBodyContentMainPrintRangeInputsWrapper = document.createElement('div');
    const printModalBodyContentMainPrintRangeInputFrom = document.createElement('input');
    const printModalBodyContentMainPrintRangeInputTo = document.createElement('input');

    const printModalBodyContentMainPrintRangeError = document.createElement('div');
    
    const printModalBodyContentMainButton = document.createElement('button');
    const printModalBodyContentMainButtonLoader = document.createElement('div');

    const printModalBodyContentPreview = document.createElement('div');
    const printModalBodyContentPreviewLeftPage = document.createElement('div');
    const printModalBodyContentPreviewRightPage = document.createElement('div');

    const printModalBodyContentPreviewLeftPageImage = document.createElement('img');
    const printModalBodyContentPreviewRightPageImage = document.createElement('img');
    const printModalBodyContentPreviewLeftPageNumber = document.createElement('span');
    const printModalBodyContentPreviewRightPageNumber = document.createElement('span');

    printModalBodyContentMain.className = 'modal__body__main';
    printModalBodyContentMainContent.className = 'modal__body__main__content';

    printModalBodyContentMainPrintAll.className = 'modal__body__control modal__body__control--print-all';
    printModalBodyContentMainPrintAllInput.name = 'print';
    printModalBodyContentMainPrintAllInput.type = 'radio';
    printModalBodyContentMainPrintAllInput.value = '1';
    printModalBodyContentMainPrintAllInput.id = 'radio-print-all';
    printModalBodyContentMainPrintAllInput.setAttribute('checked', '');
    printModalBodyContentMainPrintAllLabel.htmlFor = 'radio-print-all';
    printModalBodyContentMainPrintAllLabel.innerText = APP._t('Print Form Option All');

    printModalBodyContentMainPrintCurrent.className = 'modal__body__control modal__body__control--print-current';
    printModalBodyContentMainPrintCurrentInput.name = 'print';
    printModalBodyContentMainPrintCurrentInput.type = 'radio';
    printModalBodyContentMainPrintCurrentInput.value = '2';
    printModalBodyContentMainPrintCurrentInput.id = 'radio-print-current';
    printModalBodyContentMainPrintCurrentLabel.htmlFor = 'radio-print-current';
    printModalBodyContentMainPrintCurrentLabel.innerText = APP._t('Print Form Option Current');

    printModalBodyContentMainPrintRange.className = 'modal__body__control modal__body__control--print-range';
    printModalBodyContentMainPrintRangeInput.name = 'print';
    printModalBodyContentMainPrintRangeInput.type = 'radio';
    printModalBodyContentMainPrintRangeInput.value = '3';
    printModalBodyContentMainPrintRangeInput.id = 'radio-print-range';
    printModalBodyContentMainPrintRangeLabel.htmlFor = 'radio-print-range';
    
    printModalBodyContentMainPrintRangeLabelSpan.innerText = APP._t('Print Form Option Range');
    printModalBodyContentMainPrintRangeInputsWrapper.className = 'modal__body__controls__wrapper';

    printModalBodyContentMainPrintRangeInputFrom.type = 'text';
    printModalBodyContentMainPrintRangeInputFrom.id = 'print-range-start';
    printModalBodyContentMainPrintRangeInputFrom.value = FROM.toString();
    
    printModalBodyContentMainPrintRangeInputTo.type = 'text';
    printModalBodyContentMainPrintRangeInputTo.id = 'print-range-end';
    printModalBodyContentMainPrintRangeInputTo.value = TO.toString();

    printModalBodyContentMainPrintRangeError.className = 'modal__body__control__error';
    printModalBodyContentMainPrintRangeError.setAttribute('hidden', '');
    printModalBodyContentMainPrintRangeError.textContent = APP._t('Print Form Option Range Error');

    printModalBodyContentMainButton.className = 'modal__button';
    printModalBodyContentMainButton.innerHTML = `<span>${APP._t('Print Form Submit')}</span>`;

    printModalBodyContentMainButtonLoader.className = 'loader';
    printModalBodyContentMainButtonLoader.innerHTML = `<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" style="margin: auto; background: none; display: block; shape-rendering: auto;" width="50px" height="50px" viewBox="0 0 100 100" preserveAspectRatio="xMidYMid"><circle cx="50" cy="50" fill="none" stroke="#3461c9" stroke-width="6" r="32" stroke-dasharray="150.79644737231007 52.26548245743669"><animateTransform attributeName="transform" type="rotate" repeatCount="indefinite" dur="1s" values="0 50 50;360 50 50" keyTimes="0;1"></animateTransform></circle></svg>`;

    printModalBodyContentPreview.className = 'modal__body__preview';
    printModalBodyContentPreviewLeftPage.id = 'print_left_preview';
    printModalBodyContentPreviewRightPage.id = 'print_right_preview';
    printModalBodyContentPreviewLeftPage.className = 'modal__body__preview__page modal__body__preview__page--left';
    printModalBodyContentPreviewRightPage.className = 'modal__body__preview__page modal__body__preview__page--right';
    printModalBodyContentPreviewLeftPageImage.id = 'cover_from';
    printModalBodyContentPreviewRightPageImage.id = 'cover_to';

    printModalBodyContentMainPrintAllInput.addEventListener('change', handleChangePrintOption);
    printModalBodyContentMainPrintCurrentInput.addEventListener('change', handleChangePrintOption);
    printModalBodyContentMainPrintRangeInput.addEventListener('change', handleChangePrintOption);

    printModalBodyContentMainPrintRangeInputFrom.addEventListener('keyup', handleKeyUpPrintRange);
    printModalBodyContentMainPrintRangeInputTo.addEventListener('keyup', handleKeyUpPrintRange);
    printModalBodyContentMainPrintRangeInputFrom.addEventListener('input', handleInputPrintRange);
    printModalBodyContentMainPrintRangeInputTo.addEventListener('input', handleInputPrintRange);
    printModalBodyContentMainPrintRangeInputFrom.addEventListener('focus', handleFocusPrintRange);
    printModalBodyContentMainPrintRangeInputTo.addEventListener('focus', handleFocusPrintRange);
    printModalBodyContentMainPrintRangeInputFrom.addEventListener('change', handleChangePrintRange);
    printModalBodyContentMainPrintRangeInputTo.addEventListener('change', handleChangePrintRange);

    printModalBodyContentMainButton.addEventListener('click', handlePrint);

    printModalBodyContentMainPrintAllLabel.appendChild(printModalBodyContentMainPrintAllInput);
    printModalBodyContentMainPrintAll.appendChild(printModalBodyContentMainPrintAllLabel);

    printModalBodyContentMainPrintCurrentLabel.appendChild(printModalBodyContentMainPrintCurrentInput);
    printModalBodyContentMainPrintCurrent.appendChild(printModalBodyContentMainPrintCurrentLabel);

    printModalBodyContentMainPrintRangeInputsWrapper.appendChild(printModalBodyContentMainPrintRangeInputFrom);
    printModalBodyContentMainPrintRangeInputsWrapper.appendChild(printModalBodyContentMainPrintRangeInputTo);
    printModalBodyContentMainPrintRangeLabel.appendChild(printModalBodyContentMainPrintRangeInput);
    printModalBodyContentMainPrintRangeLabel.appendChild(printModalBodyContentMainPrintRangeLabelSpan);
    printModalBodyContentMainPrintRangeLabel.appendChild(printModalBodyContentMainPrintRangeInputsWrapper);

    printModalBodyContentMainPrintRange.appendChild(printModalBodyContentMainPrintRangeLabel);
    printModalBodyContentMainPrintRange.appendChild(printModalBodyContentMainPrintRangeError);

    printModalBodyContentMainContent.appendChild(printModalBodyContentMainPrintAll);
    printModalBodyContentMainContent.appendChild(printModalBodyContentMainPrintCurrent);
    printModalBodyContentMainContent.appendChild(printModalBodyContentMainPrintRange);
    printModalBodyContentMainButton.appendChild(printModalBodyContentMainButtonLoader);
    printModalBodyContentMain.appendChild(printModalBodyContentMainContent);
    printModalBodyContentMain.appendChild(printModalBodyContentMainButton);

    printModalBodyContentPreviewLeftPageImage.src = PATH_FROM;
    printModalBodyContentPreviewRightPageImage.src = PATH_TO;

    printModalBodyContentPreviewLeftPageNumber.innerHTML = APP._t('Print Form Preview Page', ['page_start', FROM]);
    printModalBodyContentPreviewRightPageNumber.innerHTML = APP._t('Print Form Preview Page', ['page_end', TO]);

    printModalBodyContentPreviewLeftPage.appendChild(printModalBodyContentPreviewLeftPageImage);
    printModalBodyContentPreviewLeftPage.appendChild(printModalBodyContentPreviewLeftPageNumber);
    printModalBodyContentPreviewRightPage.appendChild(printModalBodyContentPreviewRightPageImage);
    printModalBodyContentPreviewRightPage.appendChild(printModalBodyContentPreviewRightPageNumber);

    printModalBodyContentPreview.appendChild(printModalBodyContentPreviewLeftPage);
    printModalBodyContentPreview.appendChild(printModalBodyContentPreviewRightPage);
    
    printModalBodyContent.appendChild(printModalBodyContentMain);
    printModalBodyContent.appendChild(printModalBodyContentPreview);

    printModal.className = 'modal modal--print';
    printModalContainer.className = 'modal__container';
    printModalCloseBtn.className = 'modal__close';

    printModalHeaderIconWrapper.className = 'modal__header__icon';
    printModalHeader.className = 'modal__header';
    printModalHeaderTitle.className = 'modal__title';
    
    printModalBody.className = 'modal__body';
    printModalBodyContent.className = 'modal__body__content';

    printModalHeaderTitle.textContent = APP._t('Print Form Title');
    printModalHeaderIconWrapper.innerHTML = HELPER.buttonsIcons.printIcon;
    
    printModal.hidden = true;
    printModal.hide = closePrintModal;

    printModalCloseBtn.innerHTML = HELPER.buttonsIcons.popupCloseIcon;
    
    printModalCloseBtn.addEventListener('click', closePrintModal);
    printModal.addEventListener('click', closePrintModalOnClickOutside); 

    window.addEventListener('keydown', (event) => {
      if (!printModal || printModal.hasAttribute('hidden')) return;
      if (event.code !== 'Escape') return;

      closePrintModal();
    });

    printModalHeader.appendChild(printModalHeaderIconWrapper);
    printModalHeader.appendChild(printModalHeaderTitle);

    printModalBody.appendChild(printModalCloseBtn);
    printModalBody.appendChild(printModalBodyContent);

    printModalContainer.appendChild(printModalHeader);
    printModalContainer.appendChild(printModalBody);
    printModal.appendChild(printModalContainer);

    document.body.appendChild(printModal);

    if (!document.querySelector('script[data-id="print"]')) {
      const script = document.createElement('script');
      script.dataset.id = 'print';
      script.src = `${APP.PATH_CF_FLIP}assets/scripts/print.min.js`;
      document.body.appendChild(script);
    }

    elements.printModalElements = {
      modal: printModal,
      container: printModalContainer,
      bodyContent: printModalBodyContent,
      closeBtn: printModalCloseBtn,
      previewFrom: printModalBodyContentPreviewLeftPage,
      previewTo: printModalBodyContentPreviewRightPage,
      coverFrom: printModalBodyContentPreviewLeftPageImage,
      coverTo: printModalBodyContentPreviewRightPageImage,
      pageStart: printModalBodyContentPreviewLeftPageNumber.querySelector('span'),
      pageEnd: printModalBodyContentPreviewRightPageNumber.querySelector('span'),
      rangeFrom: printModalBodyContentMainPrintRangeInputFrom,
      rangeTo: printModalBodyContentMainPrintRangeInputTo,
      options: {
        all: printModalBodyContentMainPrintAllInput,
        current: printModalBodyContentMainPrintCurrentInput,
        range: printModalBodyContentMainPrintRangeInput
      },
      errorElement: printModalBodyContentMainPrintRangeError,
      printBtn: printModalBodyContentMainButton
    };
  };

  const printModalHandler = () => {
    definePrintModal();

    const FROM = 1;
    const TO = +bookModel.pages;

    const bookId = bookModel.ref_id || bookModel.id;
    const PATH_FROM = HELPER.getPageImageWithTxtUrl(FROM, 400).replaceAll(bookModel.id, bookId);
    const PATH_TO = HELPER.getPageImageWithTxtUrl(TO, 400).replaceAll(bookModel.id, bookId);

    elements.printModalElements.coverFrom.src = PATH_FROM;
    elements.printModalElements.pageStart.textContent = FROM;

    elements.printModalElements.coverTo.src = PATH_TO;
    elements.printModalElements.pageEnd.textContent = TO;

    elements.printModalElements.rangeFrom.value = FROM;
    elements.printModalElements.rangeTo.value = TO;

    elements.printModalElements.options.all.click();

    const printModal = elements.printModalElements.modal;
    const printModalContainer = elements.printModalElements.container;

    printModal.handleClose = () => {
      elements.printModalElements.errorElement.setAttribute('hidden', '');
    };

    HELPER.fadeInEffect(printModal, printModalContainer, MODAL_FADE_TIME, () => HELPER.sendPostMessageToParent('publuu.viewer.popup.print.open'));
  };

  this.showPageLikeIMG = () => {
    return settings.showPageLikeIMG;
  };

  this.getPageIndexOnContextMenuShare = function (e) {
    const composedPath = e.path || e.composedPath();
    if (composedPath.length) {
      const currentPage = composedPath.filter((item) => item.classList && item.classList.contains('Page'));
      return currentPage.length ? currentPage[0].page : this.getCurrentPage();
    } else {
      const currentPage = (e.target.parentNode && e.target.parentNode.classList.contains('Page'))
        ? e.target.parentNode
        : null;
      return currentPage ? currentPage.page : this.getCurrentPage();
    }
  };

  const renderContextMenu = function (event) {
    event.preventDefault();
    const eventPath = event.path || (event.composedPath && event.composedPath());

    if (eventPath.findIndex(item => item.classList && item.classList.contains('Book__inner')) === -1) {
      self.contextMenuToggle();
      return;
    }

    if (
      event.target.parentNode.className === 'context-menu__item' ||
      event.target.className === 'context-menu__item'
    ) {
      return;
    }

    if (scale > 0) {
      return;
    }

    const contextMenuElements = [
      {
        isAvailable: bookModel.settings.show_note_button,
        itemClass: '',
        class: 'note',
        text: APP._t('Context Menu Option Add Note'),
        func: () => {
          if (!APP.Note) {
            self.contextMenuToggle();
            return;
          }

          if (is_safari() || is_fireFox()) {
            APP.Note.createNoteByCoordinates(eventPath, event.layerX, event.layerY);
          }
          APP.Note.createNoteByCoordinates(eventPath, event.offsetX, event.offsetY);
          self.contextMenuToggle();
        },
        showOnlyOnPage: true,
      },
      {
        isAvailable: bookModel.settings.show_share_button && !parseInt(bookModel.settings.enable_protected_domain),
        itemClass: '',
        class: 'share',
        text: APP._t('Context Menu Option Share Page'),
        func: () => {
          onShareClick(event, self.getPageIndexOnContextMenuShare(event));
          self.contextMenuToggle();
        },
        showOnlyOnPage: true,
      },
      {
        isAvailable: bookModel.settings.show_fullscreen_button,
        itemClass: '',
        class: `${__is_full_screen ? 'fullscreen-on' : 'fullscreen'}`,
        text: APP._t('Context Menu Option Fullscreen'),
        func: () => {
          fullScreenToggle();
          self.contextMenuToggle();
        },
        showOnlyOnPage: true,
      },
    ];

    let contextMenu;
    const contextMenuList = document.createElement('ul');
    const bookInner = document.querySelector('.Book__wrapper');
    const CONTEXT_MENU_OFFSET = 10;
    contextMenuList.className = 'context-menu__list';

    if (document.querySelector('.context-menu')) {
      contextMenu = document.querySelector('.context-menu');
      contextMenu.querySelector('.context-menu__list').remove();
    } else {
      contextMenu = document.createElement('div');
      contextMenu.className = 'context-menu';
    }

    contextMenuElements.forEach((item) => {
      if (!item.isAvailable) return;

      if (eventPath && eventPath.some((item) => item.classList && item.classList.contains('Page'))) {
        const contextMenuItem = document.createElement('li');
        contextMenuItem.className = `context-menu__item ${item.itemClass}`;

        contextMenuItem.innerHTML = `<div class="context-menu__icon context-menu__icon--${item.class}"></div><span>${item.text}</span>`;
        contextMenuItem.addEventListener('click', item.func);

        contextMenuList.appendChild(contextMenuItem);
      } else {
        if (!item.showOnlyOnPage) {
          const contextMenuItem = document.createElement('li');
          contextMenuItem.className = `context-menu__item ${item.itemClass}`;

          contextMenuItem.innerHTML = `<div class="context-menu__icon context-menu__icon--${item.class}"></div><span>${item.text}</span>`;
          contextMenuItem.addEventListener('click', item.func);

          contextMenuList.appendChild(contextMenuItem);
        } else {
          if (!item.showOnlyOnPage) {
            const contextMenuItem = document.createElement('li');
            contextMenuItem.className = `context-menu__item ${item.itemClass}`;

            contextMenuItem.innerHTML = `<div class="context-menu__icon context-menu__icon--${item.class}"></div><span>${item.text}</span>`;
            contextMenuItem.addEventListener('click', item.func);

            contextMenuList.appendChild(contextMenuItem);
          }
        }
      }
    });

    contextMenu.appendChild(contextMenuList);
    bookInner.appendChild(contextMenu);

    const availableListItems = contextMenuList.querySelectorAll('.context-menu__item');

    if (availableListItems.length == 0) {
      contextMenu.setAttribute('hidden', true);
    } else {
      if (contextMenu.hasAttribute('hidden')) {
        contextMenu.removeAttribute('hidden');
      }
    }

    contextMenu.style = `top: ${
      event.clientY + contextMenu.clientHeight > bookInner.getBoundingClientRect().height
        ? bookInner.getBoundingClientRect().height - contextMenu.clientHeight - CONTEXT_MENU_OFFSET
        : Math.max(CONTEXT_MENU_OFFSET, event.clientY - bookInner.getBoundingClientRect().top)
    }px; left: ${
      event.clientX + contextMenu.clientWidth > bookInner.getBoundingClientRect().width
        ? bookInner.getBoundingClientRect().width - contextMenu.clientWidth - CONTEXT_MENU_OFFSET
        : Math.max(CONTEXT_MENU_OFFSET, event.clientX - bookInner.getBoundingClientRect().left)
    }px;`;
  };

  this.contextMenuToggle = function () {
    if (document.querySelector('.context-menu')) {
      document.querySelector('.context-menu').remove();
    }
  };

  const defineSourceModal = () => {
    if (elements.sourceModalElements) {
      return elements.sourceModalElements;
    }

    elements.sourceModalElements = {};

    const closeSourceModal = () => {
      const callbackFn = sourceModal.handleClose && typeof(sourceModal.handleClose) === 'function' ? sourceModal.handleClose : null;

      HELPER.fadeOutEffect(sourceModal, sourceModalContainer, MODAL_FADE_TIME, () => {
        HELPER.sendPostMessageToParent('publuu.viewer.popup.source.close');

        if (callbackFn) {
          callbackFn();
        }
      });
    };

    const closeSourceModalOnClickOutside = (event) => {
      const shouldCloseModal = !(event.path || event.composedPath()).find(el => el === sourceModalBodyContent);

      if (!shouldCloseModal) return;

      closeSourceModal();
    };

    const handleIframeLoad = function () {
      if (!this.src || this.src.trim().length === 0) return;
      
      elements.sourceModalElements.loader.setAttribute('hidden', '');
    };

    const sourceModal = document.createElement('div');
    const sourceModalContainer = document.createElement('div');
    const sourceModalCloseBtn = document.createElement('button');
    const sourceModalBody = document.createElement('div');
    const sourceModalBodyContent = document.createElement('div');
    const sourceModalLoader = document.createElement('div');
    const sourceModalIframe = document.createElement('iframe');

    sourceModal.className = 'modal modal--source';
    sourceModalContainer.className = 'modal__container';
    sourceModalCloseBtn.className = 'modal__close';
    sourceModalBody.className = 'modal__body';
    sourceModalBodyContent.className = 'modal__body__content';
    sourceModalIframe.className = 'modal__iframe';
    sourceModalLoader.className = 'modal__loader';

    sourceModal.hidden = true;
    sourceModalIframe.width = '100%';
    sourceModalIframe.height = '100%';
    sourceModalIframe.allow = 'autoplay; clipboard-write; clipboard-read; fullscreen';
    sourceModalIframe.setAttribute('allowfullscreen', '');

    sourceModalLoader.innerHTML = HELPER.buttonsIcons.loaderIcon;
    sourceModalCloseBtn.innerHTML = HELPER.buttonsIcons.popupCloseIcon;
    
    sourceModalIframe.addEventListener('load', handleIframeLoad);
    sourceModalCloseBtn.addEventListener('click', closeSourceModal);
    sourceModal.addEventListener('click', closeSourceModalOnClickOutside);

    window.addEventListener('keydown', (event) => {
      if (!sourceModal || sourceModal.hasAttribute('hidden')) return;
      if (event.code !== 'Escape') return;

      closeSourceModal();
    });
  
    sourceModalBodyContent.appendChild(sourceModalLoader);
    sourceModalBodyContent.appendChild(sourceModalIframe);
    sourceModalBody.appendChild(sourceModalCloseBtn);
    sourceModalBody.appendChild(sourceModalBodyContent);

    sourceModalContainer.appendChild(sourceModalBody);
    sourceModal.appendChild(sourceModalContainer);

    document.body.appendChild(sourceModal);

    elements.sourceModalElements = {
      modal: sourceModal,
      container: sourceModalContainer,
      bodyContent: sourceModalBodyContent,
      closeBtn: sourceModalCloseBtn,
      iframe: sourceModalIframe,
      loader: sourceModalLoader
    };
  };

  this.openLinkInFrame = function (url) {
    defineSourceModal();

    const sourceModal = elements.sourceModalElements.modal;
    const sourceModalContainer = elements.sourceModalElements.container;
    const sourceModalIframe = elements.sourceModalElements.iframe;
    const sourceModalLoader = elements.sourceModalElements.loader;

    sourceModalLoader.removeAttribute('hidden');
    sourceModalIframe.src = url;

    sourceModal.handleClose = () => sourceModalIframe.src = '';
    HELPER.fadeInEffect(sourceModal, sourceModalContainer, MODAL_FADE_TIME, () => HELPER.sendPostMessageToParent('publuu.viewer.popup.source.open'));
  };

  const openProductPopup = () => {
    const productId = window.productId;
    
    if (!productId || productId.length <= 0) {
      return;
    }

    if (!APP.Products || APP.Products.openProductByUid) {
      const waitForProductsInterval = setInterval(() => {
        if (APP.Products && APP.Products.openProductByUid) {
          clearInterval(waitForProductsInterval);
          APP.Products.openProductByUid(productId);
        }
      }, 10);
    }
  }

  this.handleUserInteraction = () => {
    if (isFirstUserInteractionHandled) return;

    isFirstUserInteractionHandled = true;

    APP.EMBED.unmuteAutoplayedVideo();
  };
};
